import("pathfinder.road", "RoadPathFinder", 3);

/* Manages the tile system.*/
class TileManager
{
	constructor() {
	}
	function GetAdjacentTiles(tile);
	function IsRoadBuildable(tile);
}	

/* Returns adjacent tiles.*/
function TileManager::GetAdjacentTiles(tile)
{
	local adjTiles = AITileList();
	
	adjTiles.AddTile(tile - AIMap.GetTileIndex(1,0));
	adjTiles.AddTile(tile - AIMap.GetTileIndex(0,1));
	adjTiles.AddTile(tile - AIMap.GetTileIndex(-1,0));
	adjTiles.AddTile(tile - AIMap.GetTileIndex(0,-1));
	
	return adjTiles;
}

/* Returns TRUE if the road is buildable.*/
function TileManager::IsRoadBuildable(tile)
{
	if (AITile.IsBuildable(tile) || AIRoad.IsRoadTile(tile)) return true;
	return false;
}

/* Manages Town-related mechanisms.*/
class TownManager
{
	constructor() {
	}
	function BuildBusPostStop(tile);
	function BuildDepot(tile);
	function FindBusPostLocation(town, pass_cargo_id, estimate);
	function FindTruckLocation(industry, pass_cargo_id, estimate);
}


/*Searches for suitable truck stop near the industry.*/
function TownManager::FindTruckLocation(industry, pass_cargo_id, estimate)
{
	local tl = AITileList();
	local aitile = null;
	local tile = AIIndustry.GetLocation(industry);
	local found = false;
	local ret_tile = null;
	local station_type = AIRoad.GetRoadVehicleTypeForCargo(pass_cargo_id);
	
	tl.AddRectangle(tile + AIMap.GetTileIndex(-10, -10), tile + AIMap.GetTileIndex(10, 10));
	
	/* remove all tiles that are already covered by a station */
	local tl2 = AITileList();

	tl2.AddList(tl);
	tl2.Valuate(AIRoad.IsRoadStationTile);
	tl2.KeepValue(1);

	for (local rstl = tl2.Begin(); tl2.HasNext() ; rstl = tl2.Next()){
			tl.RemoveRectangle(rstl + AIMap.GetTileIndex(-1, -1), rstl + AIMap.GetTileIndex(1, 1));
	}	 	
	   		/* find all tiles that are not sloped */
		   	tl.Valuate(AITile.GetSlope);	
   		   tl.KeepValue(0);
				if (tl.Count()) {
					tl.Valuate(AITile.GetCargoAcceptance, pass_cargo_id,
								 1, 1, 
								 AIStation.GetCoverageRadius (station_type));
					for (aitile = tl.Begin(); tl.HasNext(); aitile = tl.Next()){
  							found = AITile.IsBuildable(aitile);
							if (estimate) {
								found = true;
							}	
							if (!found)	{
								found = AITile.DemolishTile(aitile);
							}
							if (found) {
								ret_tile = aitile;
								break;
							}	
					}
				}
				else {
					AILog.Info("No usloped tiles available for bus loader. No unsloped tiles available for truck loader.");
				}
	
	if (found) {
		return ret_tile;	
	}	
	else {
		return null;	
	}
}

/*Searches for suitable bus stop in the city.*/
function TownManager::FindBusPostLocation(town, pass_cargo_id, estimate)
{
	local tl = AITileList();
	local aitile = null;
	local tile = AITown.GetLocation(town);
	local found = false;
	local ret_tile = null;
	local station_type = null;
	station_type = AIRoad.GetRoadVehicleTypeForCargo(pass_cargo_id);
	
	tl.AddRectangle(tile + AIMap.GetTileIndex(-10, -10), tile + AIMap.GetTileIndex(10, 10));
	
	/* remove all tiles that are already covered by a station */
	local tl2 = AITileList();

	tl2.AddList(tl);
	tl2.Valuate(AIRoad.IsRoadStationTile);
	tl2.KeepValue(1);

	for (local rstl = tl2.Begin(); tl2.HasNext() ; rstl = tl2.Next()){
		/* Keep our own stations a bit apart */
		if (AITile.GetOwner(rstl) == AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)) {
			tl.RemoveRectangle(rstl + AIMap.GetTileIndex(-4, -4), rstl + AIMap.GetTileIndex(4, 4));
		}
		else {    /* but don't be so modest with other players stations ... */
			tl.RemoveRectangle(rstl + AIMap.GetTileIndex(-1, -1), rstl + AIMap.GetTileIndex(1, 1));
		}
	}	 
	if (tl.Count()) {
		/* find all tiles that are next to a road tile */
		tl.Valuate(AIRoad.GetNeighbourRoadCount);	
		tl.KeepAboveValue(0);

		if (tl.Count()) {
			/* find all tiles that are not road */
	   	tl.Valuate(AIRoad.IsRoadTile);	
   	  	tl.KeepValue(0);
			if (tl.Count()) {
	   		/* find all tiles that are not sloped */
		   	tl.Valuate(AITile.GetSlope);	
   		   tl.KeepValue(0);
				if (tl.Count()) {
					tl.Valuate(AITile.GetCargoAcceptance, pass_cargo_id,
								 1, 1, 
								 AIStation.GetCoverageRadius (station_type));
					for (aitile = tl.Begin(); tl.HasNext(); aitile = tl.Next()){
  						found = AITile.IsBuildable(aitile);
						if (estimate) {
							found = true;
						}	
						if (!found)	{
							found = AITile.DemolishTile(aitile);
						}
						if (found) {
							ret_tile = aitile;
							break;
						}	

					}
				}
				else {
					AILog.Info("No usloped tiles available for bus loader. No unsloped tiles available for truck loader.");
				}
		}
		else {
			AILog.Info("No not road tiles available for bus loader.");
		}
	}
	else {
		AILog.Info("No next to road tiles available for bus loader.");
	}
	}
	

	
	if (found) {
		return ret_tile;	
	}	
	else {
		return null;	
	}	
}

/*Tries to build Road Depot on a given tile.*/
function TownManager::BuildDepot(tile)
{
	local found = false;
	local success = false;

	local adjacentTiles = TileManager.GetAdjacentTiles(tile);

	for(local tile2 = adjacentTiles.Begin(); 
			adjacentTiles.HasNext() && !found; 
			tile2 = adjacentTiles.Next()) {
		if(AIRoad.IsRoadTile(tile2) ) {
			success = true;
			local count = 0;
			while(!success && (count < 100)){
				local pathlen = AIRoad.BuildRoad(tile2, tile);
				if (pathlen == 2) {
					success = true; 
				} else {
					count = count + 1;
				}
			}
			local truck = AIRoad.ROADVEHTYPE_BUS; 
			local adjacent = AIStation.STATION_NEW;
			if (success) {
				AITile.DemolishTile(tile);
				AIRoad.BuildRoad(tile2, tile);
				success = AIRoad.BuildRoadDepot(tile, tile2);
			}
			found = true;
		}
	}

	if (!success) {
		AILog.Info("Build depot failed with error: "+ AIError.GetLastErrorString());
	}
	return tile;	
}

/*Builds Bus/Truck(for post trucks) loader on a given tyle in the city.*/
function TownManager::BuildBusPostStop(tile,vehicle_type)
{
	local found = false;
	local success = false;

	local adjacentTiles = TileManager.GetAdjacentTiles(tile);

	for(local tile2 = adjacentTiles.Begin(); 
			adjacentTiles.HasNext() && !found; 
			tile2 = adjacentTiles.Next()) {
		if(AIRoad.IsRoadTile(tile2) ) {
			success = true;
			local count = 0;
			while(!success && (count < 100)){
				local pathlen = AIRoad.BuildRoad(tile2, tile);
				if (pathlen == 2) {
					success = true; 
				} else {
					count = count + 1;
				}
			}
			local adjacent = AIStation.STATION_NEW;
			if (success) {
				AITile.DemolishTile(tile);
				AIRoad.BuildRoad(tile2, tile);
				success = AIRoad.BuildRoadStation(tile, tile2, vehicle_type, adjacent);
			}
			found = true;
		}
	}

	if (!success) {
		AILog.Info("Build bus loader failed with error: "+ AIError.GetLastErrorString());
	}
	return tile;	
}

class BankManager
{
	constructor() {
	}
	function HasMoney(money);
	function GetMoney(money);
}

/*Returns true if total money (account + loan) equals of exceeds given money.*/
function BankManager::HasMoney(money)
{
	if (AICompany.GetBankBalance(AICompany.COMPANY_SELF) + (AICompany.GetMaxLoanAmount() - AICompany.GetLoanAmount()) > money) return true;
	return false;
}

/*Checks if there is enough money and if not, loans it.*/
function BankManager::GetMoney(money)
{
	if (!BankManager.HasMoney(money)) return;
	if (AICompany.GetBankBalance(AICompany.COMPANY_SELF) > money) return;

	local loan = money - AICompany.GetBankBalance(AICompany.COMPANY_SELF) + AICompany.GetLoanInterval() + AICompany.GetLoanAmount();
	loan = loan - loan % AICompany.GetLoanInterval();
	AILog.Info("Need a loan to get " + money + ": " + loan);
	AICompany.SetLoanAmount(loan);
}

/*Main class of AI*/
class MpAI extends AIController
{
	name = null;
	towns_used = null;
	route_1 = null;
	route_2 = null;
	distance_of_route = {};
	vehicle_to_depot = {};
	delay_build_airport_route = 1000;
	passenger_cargo_id = -1;
	mail_cargo_id = -1;
	armoured_cargo_id = -1;
	
	function Start();

/*Initialization - loading cargo indexes.*/
	constructor() {
		this.towns_used = AIList();
		this.route_1 = AIList();
		this.route_2 = AIList();

		local list = AICargoList();
		for (local i = list.Begin(); list.HasNext(); i = list.Next()) {
			if (AICargo.HasCargoClass(i, AICargo.CC_PASSENGERS)) {
				this.passenger_cargo_id = i;
			}
			if (AICargo.HasCargoClass(i, AICargo.CC_MAIL)) {
				this.mail_cargo_id = i;
			}
			if (AICargo.HasCargoClass(i, AICargo.CC_ARMOURED)) {
				this.armoured_cargo_id = i;
			}
		}
	}
};

/*Base bus route for basic income.*/
function MpAI::BuildBusRoute()
{
	local bus_model = 0;
	local build = 0;
	local depot = 0;
	local station_a = 0;
	local station_b = 0;
		
	/* Get a list of all towns on the map. */
	local townlist = AITownList();
	/* Sort the list by population, highest population first. */
	townlist.Valuate(AITown.GetPopulation);
	townlist.Sort(AIAbstractList.SORT_BY_VALUE, false);
	townlist.RemoveList(this.towns_used);
	/* Pick the two towns with the highest population. */
	local townid_a = townlist.Begin();
	local teller = 1;
	
	/* Makes a list of different bus types so the AI knows which bus to build */
	local engine_list = AIEngineList(AIVehicle.VT_ROAD);
	
	engine_list.Valuate(AIEngine.GetRoadType);
	engine_list.KeepValue(AIRoad.ROADTYPE_ROAD);
	
	local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
	engine_list.Valuate(AIEngine.GetPrice);
	engine_list.KeepBelowValue(balance);

	engine_list.Valuate(AIEngine.GetCargoType)
	engine_list.KeepValue(passenger_cargo_id); 

	engine_list.Valuate(AIEngine.GetCapacity)
	engine_list.KeepTop(1);

	bus_model = engine_list.Begin();
	
	for( local i = 1; i<=5; i++ ) {
		BankManager.GetMoney(50000);
		local townid_b = townlist.Next();
		/* Print the names of the towns we'll try to connect. */
		AILog.Info("Connecting " + AITown.GetName(townid_a) + " and " + AITown.GetName(townid_b));
		/* Tell OpenTTD we want to build normal road. */
		AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);
		/* Create an instance of the pathfinder. */
		local pathfinder = RoadPathFinder();
		pathfinder.cost.turn = 1;
		/* Give the source and goal tiles to the pathfinder. */
		pathfinder.InitializePath([AITown.GetLocation(townid_a)], [AITown.GetLocation(townid_b)]);
		
		/* Try to find a path. */
		local path = false;
		while (path == false) {
			path = pathfinder.FindPath(100);
		}
		
		if (path == null) {
			/* No path was found. */
			AILog.Error("pathfinder.FindPath return null");
		}
		
		/* If a path was found, build a road over it. */
		while (path != null) {
			local par = path.GetParent();
			if (par != null) {
				local last_node = path.GetTile();
				if (AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) == 1 ) {
					if (!AIRoad.BuildRoad(path.GetTile(), par.GetTile())) {
						/* An error occured while building a piece of road. TODO: handle it.
						* Note that is can also be the case that the road was already build. */
					}
				}
				else {
				/* Build a bridge or tunnel. */
					if (!AIBridge.IsBridgeTile(path.GetTile()) && !AITunnel.IsTunnelTile(path.GetTile())) {
						/* If it was a road tile, demolish it first. Do this to work around expended roadbits. */
						if (AIRoad.IsRoadTile(path.GetTile())) AITile.DemolishTile(path.GetTile());
						if (AITunnel.GetOtherTunnelEnd(path.GetTile()) == par.GetTile()) {
							if (!AITunnel.BuildTunnel(AIVehicle.VT_ROAD, path.GetTile())) {
								/* An error occured while building a tunnel. TODO: handle it. */
							}
						} 
						else {
							local bridge_list = AIBridgeList_Length(AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) + 1);
							bridge_list.Valuate(AIBridge.GetMaxSpeed);
							bridge_list.Sort(AIAbstractList.SORT_BY_VALUE, false);
							if (!AIBridge.BuildBridge(AIVehicle.VT_ROAD, bridge_list.Begin(), path.GetTile(), par.GetTile())) {
								/* An error occured while building a bridge. TODO: handle it. */
							}
						}
					}
				}
			}
			path = par;
		}
		AILog.Info("Connected " + AITown.GetName(townid_a) + " and " + AITown.GetName(townid_b));
		
		if (teller == 1) {
		/* In the largest city we make both a bus depot and a bus stop */
		station_a = TownManager.BuildBusPostStop(TownManager.FindBusPostLocation(townid_a, this.passenger_cargo_id, true),AIRoad.ROADVEHTYPE_BUS);
		depot = TownManager.BuildDepot(TownManager.FindBusPostLocation(townid_a, passenger_cargo_id, true));
		}
		/* For every other city we create a bus stop */
		local station_b = TownManager.BuildBusPostStop(TownManager.FindBusPostLocation(townid_b, this.passenger_cargo_id, true),AIRoad.ROADVEHTYPE_BUS);
		
		/* We create 2 buses for every bus line */
		for (local j = 1; j<=2; j++ ) {
			local build = AIVehicle.BuildVehicle(depot, bus_model);
			AIOrder.AppendOrder(build, depot, AIOrder.AIOF_SERVICE_IF_NEEDED);
			AIOrder.AppendOrder(build, station_a, AIOrder.OF_FULL_LOAD); 
			AIOrder.AppendOrder(build, station_b, AIOrder.OF_FULL_LOAD);
			AIVehicle.StartStopVehicle(build);
		}
		
		teller++;
	}
}

/* Builds airport route for basic income.*/
function MpAI::BuildAirportRoute()
{
	local airport_type = (AIAirport.IsValidAirportType(AIAirport.AT_LARGE) ? AIAirport.AT_LARGE : AIAirport.AT_SMALL);

	/* Get enough money to work with */
	BankManager.GetMoney(150000);

	AILog.Info("Build airport route.");

	local tile_1 = this.FindSuitableAirportSpot(airport_type, 0);
	if (tile_1 < 0) return -1;
	local tile_2 = this.FindSuitableAirportSpot(airport_type, tile_1);
	if (tile_2 < 0) {
		this.towns_used.RemoveValue(tile_1);
		return -2;
	}

	/* Build the airports for real */
	if (!AIAirport.BuildAirport(tile_1, airport_type, AIStation.STATION_NEW)) {
		AILog.Error("Although the testing told us we could build 2 airports, it still failed on the first airport at tile " + tile_1 + ".");
		this.towns_used.RemoveValue(tile_1);
		this.towns_used.RemoveValue(tile_2);
		return -3;
	}
	if (!AIAirport.BuildAirport(tile_2, airport_type, AIStation.STATION_NEW)) {
		AILog.Error("Although the testing told us we could build 2 airports, it still failed on the second airport at tile " + tile_2 + ".");
		AIAirport.RemoveAirport(tile_1);
		this.towns_used.RemoveValue(tile_1);
		this.towns_used.RemoveValue(tile_2);
		return -4;
	}

	local ret = this.BuildAircraft(tile_1, tile_2);
	if (ret < 0) {
		AIAirport.RemoveAirport(tile_1);
		AIAirport.RemoveAirport(tile_2);
		this.towns_used.RemoveValue(tile_1);
		this.towns_used.RemoveValue(tile_2);
		return ret;
	}

	AILog.Info("Built a route");
	return ret;
}

/*Builds aircraft with orders from tile_1 to tile_2. Picks best available.*/
function MpAI::BuildAircraft(tile_1, tile_2)
{
	/* Build an aircraft */
	local hangar = AIAirport.GetHangarOfAirport(tile_1);
	local engine = null;

	local engine_list = AIEngineList(AIVehicle.VT_AIR);

	/* When bank balance < 300000, buy cheaper planes */
	local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
	engine_list.Valuate(AIEngine.GetPrice);
	engine_list.KeepBelowValue(balance < 300000 ? 50000 : (balance < 1000000 ? 300000 : 1000000));

	engine_list.Valuate(AIEngine.GetCargoType);
	engine_list.KeepValue(this.passenger_cargo_id);

	engine_list.Valuate(AIEngine.GetCapacity);
	engine_list.KeepTop(1);

	engine = engine_list.Begin();

	if (!AIEngine.IsValidEngine(engine)) {
		AILog.Error("Couldn't find a suitable engine");
		return -5;
	}
	local vehicle = AIVehicle.BuildVehicle(hangar, engine);
	if (!AIVehicle.IsValidVehicle(vehicle)) {
		AILog.Error("Couldn't build the aircraft");
		return -6;
	}
	/* Send him on his way */
	AIOrder.AppendOrder(vehicle, tile_1, AIOrder.OF_FULL_LOAD);
	AIOrder.AppendOrder(vehicle, tile_2, AIOrder.OF_FULL_LOAD);
	AIVehicle.StartStopVehicle(vehicle);
	this.distance_of_route.rawset(vehicle, AIMap.DistanceManhattan(tile_1, tile_2));
	this.route_1.AddItem(vehicle, tile_1);
	this.route_2.AddItem(vehicle, tile_2);

	AILog.Info("Built an aircraft");

	return 0;
}

/*Finds suitable Airport spot around the center tile.*/
function MpAI::FindSuitableAirportSpot(airport_type, center_tile)
{
	local airport_x, airport_y, airport_rad;

	airport_x = AIAirport.GetAirportWidth(airport_type);
	airport_y = AIAirport.GetAirportHeight(airport_type);
	airport_rad = AIAirport.GetAirportCoverageRadius(airport_type);

	local town_list = AITownList();
	/* Remove all the towns we already used */
	town_list.RemoveList(this.towns_used);

	town_list.Valuate(AITown.GetPopulation);
	town_list.KeepTop(10);

	/* Now find 2 suitable towns */
	for (local town = town_list.Begin(); town_list.HasNext(); town = town_list.Next()) {
		/* Don't make this a CPU hog */
		Sleep(1);

		local tile = AITown.GetLocation(town);

		/* Create a 30x30 grid around the core of the town and see if we can find a spot for a small airport */
		local list = AITileList();
		/* XXX -- We assume we are more than 15 tiles away from the border! */
		list.AddRectangle(tile - AIMap.GetTileIndex(15, 15), tile + AIMap.GetTileIndex(15, 15));
		list.Valuate(AITile.IsBuildableRectangle, airport_x, airport_y);
		list.KeepValue(1);
		if (center_tile != 0) {
			/* If we have a tile defined, we don't want to be within 25 tiles of this tile */
			list.Valuate(AITile.GetDistanceSquareToTile, center_tile);
			list.KeepAboveValue(625);
		}
		/* Sort on acceptance, remove places that don't have acceptance */
		list.Valuate(AITile.GetCargoAcceptance, this.passenger_cargo_id, airport_x, airport_y, airport_rad);
		list.RemoveBelowValue(10);

		/* Couldn't find a suitable place for this town, skip to the next */
		if (list.Count() == 0) continue;
		/* Walk all the tiles and see if we can build the airport at all */
		{
			local test = AITestMode();
			local good_tile = 0;

			for (tile = list.Begin(); list.HasNext(); tile = list.Next()) {
				Sleep(1);
				if (!AIAirport.BuildAirport(tile, airport_type, AIStation.STATION_NEW)) continue;
				good_tile = tile;
				break;
			}

			/* Did we found a place to build the airport on? */
			if (good_tile == 0) continue;
		}

		AILog.Info("Built airport in town: " + town);

		/* Make the town as used, so we don't use it again */
		this.towns_used.AddItem(town, tile);

		return tile;
	}

	AILog.Info("Couldn't find a suitable town to build an airport in");
	return -1;
}

/*Manages vehicle routes.*/
function MpAI::ManageRoutes()
{
	local list = AIVehicleList();
	list.Valuate(AIVehicle.GetAge);
	/* Give the plane at least 2 years to make a difference */
	list.KeepAboveValue(365 * 4);
	list.Valuate(AIVehicle.GetProfitLastYear);

	for (local i = list.Begin(); list.HasNext(); i = list.Next()) {
		local profit = list.GetValue(i);
		/* Profit last year and this year bad? Let's sell the vehicle */
		if (profit < 10000 && AIVehicle.GetProfitThisYear(i) < 10000) {
			/* Send the vehicle to depot if we didn't do so yet */
			if (!vehicle_to_depot.rawin(i) || vehicle_to_depot.rawget(i) != true) {
				AILog.Info("Vehicle " + i + " ordered to go to depot due to small profit.");
				AIVehicle.SendVehicleToDepot(i);
				vehicle_to_depot.rawset(i, true);
			}
		}
		/* Try to sell it over and over till it really is in the depot */
		if (vehicle_to_depot.rawin(i) && vehicle_to_depot.rawget(i) == true) {
			if (AIVehicle.SellVehicle(i)) {
				AILog.Info("Selling vehicle " + i + ".");
				/* Check if we are the last one serving those airports; else sell the airports */
				local list2 = AIVehicleList_Station(AIStation.GetStationID(this.route_1.GetValue(i)));
				if (list2.Count() == 0) this.SellAirports(i);
				vehicle_to_depot.rawdelete(i);
			}
		}
	}

	/* Don't try to add planes when we are short on cash */
	if (!BankManager.HasMoney(50000)) return;

	list = AIStationList(AIStation.STATION_AIRPORT);
	list.Valuate(AIStation.GetCargoWaiting, this.passenger_cargo_id);
	list.KeepAboveValue(250);

	for (local i = list.Begin(); list.HasNext(); i = list.Next()) {
		local list2 = AIVehicleList_Station(i);
		/* No vehicles going to this station, abort and sell */
		if (list2.Count() == 0) {
			this.SellAirports(i);
			continue;
		};

		/* Find the first vehicle that is going to this station */
		local v = list2.Begin();
		local dist = this.distance_of_route.rawget(v);

		list2.Valuate(AIVehicle.GetAge);
		list2.KeepBelowValue(dist/2);
		/* Do not build a new vehicle if we bought a new one in the last DISTANCE days */
		if (list2.Count() != 0) continue;

		AILog.Info("Adding new vehicle to station " + AIStation.GetLocation(i) + " as it has too much cargo.");

		/* Make sure we have enough money */
		BankManager.GetMoney(50000);

		return this.BuildAircraft(this.route_1.GetValue(v), this.route_2.GetValue(v));
	}
}

/*Sells airports from basic income if they no longer generate no money.*/
function MpAI::SellAirports(i) {
	/* Remove the airports */
	AILog.Info("Removed unoccupied airports.");
	AIAirport.RemoveAirport(this.route_1.GetValue(i));
	AIAirport.RemoveAirport(this.route_2.GetValue(i));
	/* Free the towns_used entries */
	this.towns_used.RemoveValue(this.route_1.GetValue(i));
	this.towns_used.RemoveValue(this.route_2.GetValue(i));
	/* Remove the route */
	this.route_1.RemoveItem(i);
	this.route_2.RemoveItem(i);
}

/*Builds road transport from town to town with given cargo type.*/
function MpAI::BuildRoadTransport(townid_a, townid_b,cargo_type)
{
	local location_a = AITown.GetLocation(townid_a);
	local location_b = AITown.GetLocation(townid_b);
	AILog.Info("Town locations are: " + location_a + " " + location_b);
	
	local pathfinder = RoadPathFinder();
	pathfinder.cost.turn = 1;
	/* Give the source and goal tiles to the pathfinder. */
	AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);
	pathfinder.InitializePath([AITown.GetLocation(townid_a)], [AITown.GetLocation(townid_b)]);
	
	/* Try to find a path. */
	local path = false;
	while (path == false) {
		path = pathfinder.FindPath(100);
	}
		
	if (path == null) {
		/* No path was found. */
		AILog.Error("pathfinder.FindPath return null");
	}
		
	/* If a path was found, build a road over it. */
	while (path != null) {
			local par = path.GetParent();
			if (par != null) {
				local last_node = path.GetTile();
				if (AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) == 1 ) {
					if (!AIRoad.BuildRoad(path.GetTile(), par.GetTile())) {
						/* An error occured while building a piece of road. TODO: handle it.
						* Note that is can also be the case that the road was already build. */
					}
				}
				else {
				/* Build a bridge or tunnel. */
					if (!AIBridge.IsBridgeTile(path.GetTile()) && !AITunnel.IsTunnelTile(path.GetTile())) {
						/* If it was a road tile, demolish it first. Do this to work around expended roadbits. */
						if (AIRoad.IsRoadTile(path.GetTile())) AITile.DemolishTile(path.GetTile());
						if (AITunnel.GetOtherTunnelEnd(path.GetTile()) == par.GetTile()) {
							if (!AITunnel.BuildTunnel(AIVehicle.VT_ROAD, path.GetTile())) {
								/* An error occured while building a tunnel. TODO: handle it. */
							}
						} 
						else {
							local bridge_list = AIBridgeList_Length(AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) + 1);
							bridge_list.Valuate(AIBridge.GetMaxSpeed);
							bridge_list.Sort(AIAbstractList.SORT_BY_VALUE, false);
							if (!AIBridge.BuildBridge(AIVehicle.VT_ROAD, bridge_list.Begin(), path.GetTile(), par.GetTile())) {
								/* An error occured while building a bridge. TODO: handle it. */
							}
						}
					}
				}
			}
			path = par;
		}
	local vehicle_type = null;
	if(cargo_type == this.passenger_cargo_id)
		vehicle_type = AIRoad.ROADVEHTYPE_BUS;
	else
		vehicle_type = AIRoad.ROADVEHTYPE_TRUCK;
	AILog.Info("Connected " + AITown.GetName(townid_a) + " and " + AITown.GetName(townid_b));
	local station_a = TownManager.BuildBusPostStop(TownManager.FindBusPostLocation(townid_a, cargo_type, true),vehicle_type);
	local depot = TownManager.BuildDepot(TownManager.FindBusPostLocation(townid_a, cargo_type, true));
	local station_b = TownManager.BuildBusPostStop(TownManager.FindBusPostLocation(townid_b, cargo_type, true),vehicle_type);
		
	/* Makes a list of different bus types so the AI knows which bus to build */
	local engine_list = AIEngineList(AIVehicle.VT_ROAD);
	
	engine_list.Valuate(AIEngine.GetRoadType);
	engine_list.KeepValue(AIRoad.ROADTYPE_ROAD);
	
	local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
	engine_list.Valuate(AIEngine.GetPrice);
	engine_list.KeepBelowValue(balance);

	engine_list.Valuate(AIEngine.GetCargoType)
	engine_list.KeepValue(cargo_type); 

	engine_list.Valuate(AIEngine.GetCapacity)
	engine_list.KeepTop(1);

	local bus_model = engine_list.Begin();
		
		/* We create 2 buses for every bus line */
		for (local i = 1; i<=2; i++ ) {
			local build = AIVehicle.BuildVehicle(depot, bus_model);
			AIOrder.AppendOrder(build, depot, AIOrder.AIOF_SERVICE_IF_NEEDED);
			AIOrder.AppendOrder(build, station_a, AIOrder.OF_FULL_LOAD); 
			AIOrder.AppendOrder(build, station_b, AIOrder.OF_FULL_LOAD);
			AIVehicle.StartStopVehicle(build);
		}
}

/*Handles game events.*/
function MpAI::HandleEvents()
{
	while (AIEventController.IsEventWaiting()) {
		local e = AIEventController.GetNextEvent();
		switch (e.GetEventType()) {
			case AIEvent.AI_ET_VEHICLE_CRASHED: {
				local ec = AIEventVehicleCrashed.Convert(e);
				local v = ec.GetVehicleID();
				AILog.Info("Crashed vehicle (" + v + "), buy replacement");
				this.BuildAircraft(this.route_1.GetValue(v), this.route_2.GetValue(v));
				this.route_1.RemoveItem(v);
				this.route_2.RemoveItem(v);
			} break;
			case AIEvent.ET_SUBSIDY_OFFER: {
				if(AICompany.GetLoanAmount() > 0){
				AILog.Info("We still have a loan, we will not accept any subsidies.");
				break;
				}
				local ec = AIEventSubsidyOffer.Convert(e);
				local subsidyID = ec.GetSubsidyID();
				local subsidySource = AISubsidy.GetSourceIndex(subsidyID);
				local subsidyDestination = AISubsidy.GetDestinationIndex(subsidyID);
				local cargo = AISubsidy.GetCargoType(subsidyID);
				AILog.Info("We have a subsidy with ID: " + subsidyID);
				AILog.Info("Source is: " + subsidySource);
				AILog.Info("Destination is: " + subsidyDestination);
				AILog.Info("Cargo is: " + cargo);
				if(cargo == this.passenger_cargo_id)
					this.BuildRoadTransport(subsidySource,subsidyDestination,passenger_cargo_id);
				if(cargo == this.mail_cargo_id)
					this.BuildRoadTransport(subsidySource,subsidyDestination,mail_cargo_id);
				if(cargo == this.armoured_cargo_id)
					this.BuildRoadTransport(subsidySource,subsidyDestination,armoured_cargo_id);
			} break;

			default:
				break;
		}
	}
}

/*Main function.*/
function MpAI::Start()
{
	/* Rename the company.*/
	if (!AICompany.SetName("mpAI")) {
		local i = 2;
		while (!AICompany.SetName("mpAI #" + i)) {
			i++;
		}
	}
	
	/* We start with almost no loan, and we take a loan when we want to build something */
	AICompany.SetLoanAmount(AICompany.GetLoanInterval());
	

	/* We need our local ticker, as GetTick() will skip ticks */
	local ticker = 0;
	/* Determine time we may sleep */
	local sleepingtime = 100;
		
	local ret = this.BuildAirportRoute();
	
	if (ret < 0 && ticker == 0) {
		/* The AI failed to build a first airport and is deemed */
		AICompany.SetName("Failed " + this.name);
		AILog.Error("Failed to build airport route for basic income, giving up building.");
		AICompany.SetLoanAmount(0);
		return;
	}

	this.BuildBusRoute();
	/*Keep running forever.*/
	while (true) {
		/* Manage the routes once in a while */
		if (ticker % 2000 == 0) {
			this.ManageRoutes();
		}
		/* Try to get rid of our loan once in a while */
		if (ticker % 2500 == 0) {
			AICompany.SetLoanAmount(0);
		}
		/* Check for events once in a while */
		if (ticker % 100 == 0) {
			this.HandleEvents();
		}
		/* Make sure we do not create infinite loops */
		Sleep(sleepingtime);
		ticker += sleepingtime;
	}
}

