/**
 *    This file is part of OtviAI.
 *
 *    OtviAI is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    OtviAI is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with OtviAI.  If not, see <http://www.gnu.org/licenses/>.
 * 
 *    This file is for the greater part based on the work of Truebrain 
 *    and only differs in some speed optimizations.
 *
 * Copyright 2008 Marnix Bakker, Willem de Neve, Michiel Konstapel and Otto Visser.
 * Suggestions/comments and bugs can go to: rondjeomdekerk@konstapel.nl
 */

/**
 * Binary Heap.
 *  Peek and Remove always return the current lowest value in the list.
 *  Sort is done on insertion and on deletion.
 */
class BinHeap
{
	_queue = null;
	//_count = 0;

	constructor()
	{
		_queue = [];
	}

	/**
	 * Insert a new entry in the list.
	 *  The complexity of this operation is O(ln n).
	 * @param item The item to add to the list.
	 * @param priority The priority this item has.
	 */
	function Insert(item, priority);

	/**
	 * Pop the first entry of the list.
	 *  This is always the item with the lowest priority.
	 *  The complexity of this operation is O(ln n).
	 * @return The item of the entry with the lowest priority.
	 */
	function Pop();

	/**
	 * Peek the first entry of the list.
	 *  This is always the item with the lowest priority.
	 *  The complexity of this operation is O(1).
	 * @return The item of the entry with the lowest priority.
	 */
	function Peek();

	/**
	 * Get the amount of current items in the list.
	 *  The complexity of this operation is O(1).
	 * @return The amount of items currently in the list.
	 */
	function Count();

	/**
	 * Check if an item exists in the list.
	 *  The complexity of this operation is O(n).
	 * @param item The item to check for.
	 * @return True if the item is already in the list.
	 */
	function Exists(item);
}

function BinHeap::Insert(item, priority)
{
	/* Append dummy entry */
	_queue.append(0);
	//this._count++;

	local hole;
	/* Find the point of insertion */
	for (hole = _queue.len() - 1; hole > 0 && priority <= this._queue[hole / 2][1]; hole /= 2)
		_queue[hole] = _queue[hole / 2];
	/* Insert new pair */
	_queue[hole] = [item, priority];

	return true;
}

function BinHeap::Pop()
{
	if (_queue.len() == 0) return null;

	local node = _queue[0];
	/* Remove the item from the list by putting the last value on top */
	_queue[0] = _queue[_queue.len() - 1];
	_queue.pop();
	//_count--;
	/* Bubble down the last value to correct the tree again */
	_BubbleDown();

	return node[0];
}

function BinHeap::Peek()
{
	if (_queue.len() == 0) return null;

	return _queue[0][0];
}

function BinHeap::Count()
{
	return _queue.len();
}

function BinHeap::Exists(item)
{
	/* Brute-force find the item (there is no faster way, as we don't have the priority number) */
	foreach (node in _queue) {
		if (node[0] == item) return true;
	}

	return false;
}



function BinHeap::_BubbleDown()
{
	if (_queue.len() == 0) return;

	local hole = 1;
	local tmp = _queue[0];

	/* Start switching parent and child until the tree is restored */
	while (hole * 2 < _queue.len() + 1) {
		local child = hole * 2;
		if (child != _queue.len() && _queue[child][1] <= _queue[child - 1][1]) child++;
		if (_queue[child - 1][1] > tmp[1]) break;

		_queue[hole - 1] = _queue[child - 1], hole = child;
	}
	/* The top value is now at his new place */
	_queue[hole - 1] = tmp;
}
