/**
 *    This file is part of OtviAI.
 *
 *    OtviAI is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    OtviAI is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with OtviAI.  If not, see <http://www.gnu.org/licenses/>.
 * 
 *    This file is based on the RoadFinder class (which in turn is based on 
 *    Truebrains work), but is a slimmed down version optimized for one goal:
 *    check whether two tiles are connected. This pathfinder can be run from
 *    the constructor as it does no do-commands at all.
 */

require("graph/AStar.nut");

/**
 * A Water Pathfinder that is optimized for connectedness checking
 */
class ConnectedChecker {
	_pathfinder = null;            ///< A reference to the used AyStar object.
	_running = null;
	_max_cost = null;

	constructor(max_cost) {
		this._pathfinder = WaterAStar(this._Cost, this._Estimate, this._Neighbours, this, this, this);
		this._running = false;
		this._max_cost = max_cost;
	}

	/**
	 * Initialize a path search between sources and goals.
	 * @param sources The source tiles.
	 * @param goals The target tiles.
	 * @see AyStar::InitializePath()
	 */
	function InitializePath(sources, goals) {
		local nsources = [];

		foreach (node in sources) {
			nsources.push([node, 0xFF]);
		}
		this._pathfinder.InitializePath(nsources, goals);
	}
	
	/**
	 * Try to find the path as indicated with InitializePath with the lowest cost.
	 * @param iterations After how many iterations it should abort for a moment.
	 *  This value should either be -1 for infinite, or > 0. Any other value
	 *  aborts immediatly and will never find a path.
	 * @return A route if one was found, or false if the amount of iterations was
	 *  reached, or null if no path was found.
	 *  You can call this function over and over as long as it returns false,
	 *  which is an indication it is not yet done looking for a route.
	 * @see AStar::FindPath()
	 */
	function FindPath(iterations) {
		local ret = this._pathfinder.FindPath(iterations);
		this._running = (ret == false) ? true : false;
		return ret;
	}

	function reset() {
		this._pathfinder.reset();
		this._running = false;
	}

	function _Cost(path, new_tile, new_direction, self) {
		/* path == null means this is the first node of a path, so the cost is 0. */
		if (path == null)
			return 0;
	
		local prev_tile = path.GetTile();
	
		if (AIMarine.AreWaterTilesConnected(prev_tile, new_tile)) {
			return (path.GetCost() + 1);
		}
	
		return self._max_cost;
	}

	function _Estimate(cur_tile, cur_direction, goal_tiles, self) {
		local min_cost = self._max_cost;
		/* As estimate we take the minimum number of tiles we need to traverse. */
		foreach (tile in goal_tiles) {
			min_cost = min(AIMap.DistanceManhattan(cur_tile, tile), min_cost);
		}
		return min_cost;
	}
	
	function _Neighbours(path, cur_node, self) {
		/* If we go over max_cost, the path isn't valid. */
		if (path.GetCost() >= self._max_cost)
			return [];
		
		local tiles = [];
	
		local offsets = [AIMap.GetTileIndex(0, 1), AIMap.GetTileIndex(0, -1), AIMap.GetTileIndex(1, 0), AIMap.GetTileIndex(-1, 0)];
		/* Check all tiles adjacent to the current tile. */
		foreach (offset in offsets) {
			local next_tile = cur_node + offset;
			/* We add them to the to the neighbours-list if one of the following applies:
			 * There already is a connections between the current tile and the next tile.
			 */
			if (AITile.IsWaterTile(next_tile)) {
				tiles.push([next_tile, self._GetDirection(cur_node, next_tile)]);
			}
		}
		return tiles;
	}
	
	function _GetDirection(from, to) {
		// TODO check correctness
		//return 0xFF;

		if (from - to == 1) return 1;
		if (from - to == -1) return 2;
		if (from - to == AIMap.GetMapSizeX()) return 4;
		if (from - to == -AIMap.GetMapSizeX()) return 8;
		throw ("Can't happen??");
	}
}

