/**
 *    This file is part of OtviAI.
 *
 *    OtviAI is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    OtviAI is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with OtviAI.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2008-2013 Otto Visser.
 * Suggestions/comments and bugs can go to: OtviAI@otvi.nl
 */

require("util.nut");

class OtviAIr {
	engine = -1;
	airportType = -1;

	maxPlanes = -1;

	stationRadius = -1;

	myAirportTownList = null;
	myAirportTypeList = null;
	myAirportTileList = null;

	function constructor() {
		Debug("Initializing air transportation...");

		updateEngines();

		stationRadius = AIStation.GetCoverageRadius(AIStation.STATION_AIRPORT);

		// init town list
		myAirportTownList = AITownList();
		myAirportTypeList = AITownList();
		myAirportTileList = AIList();

		maxPlanes = AIGameSettings.GetValue("vehicle.max_aircraft");
		Debug("Max planes: " + maxPlanes);
	}

	function getName() {
		return "air";
	}

	function isOperational() {
		if (engine == -1 || airportType == -1)
			return false;
		return true;
	}

	function getExpansionCost(cargo, distance = 100) {
		local eng = engine;
		if (distance != 100 && cargo != ::PASSENGER_ID)
			eng = getEngine(cargo, distance);

		local expandMoneyMin = 2 * AIEngine.GetPrice(eng);	// with one plane it's not worth it
		expandMoneyMin += 2 * AIAirport.GetPrice(airportType);
		return expandMoneyMin;
	}

	function canEnhance() {
		return false;
	}

	function getEnhanceCost() {
		local expandMoneyMin = AIEngine.GetPrice(engine);
		return expandMoneyMin;
	}

	function getTownTransportCost(startLoc, endLoc) {
		local dist = abs(sqrt(AIMap.DistanceSquare(AITown.GetLocation(startLoc), AITown.GetLocation(endLoc))));
		if (engine  == -1) {
			Warning("Air transport not yet available!");
			return 255;
		}

		if (getEngine(::PASSENGER_ID, dist)  == -1) {
			Warning("Air transport not available!");
			return 255;
		}

		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_AIR);
		if (vehList.Count() >= maxPlanes) {
			Warning("Reached maximum amount of planes!");
			return 254;
		}

		local ret = 2;
		if (AITown.GetPopulation(startLoc) < 2500 || AITown.GetPopulation(endLoc) < 2500)
			ret = 4;

		if (dist < 100) {
			// too damn close for air
			return (ret * 10);
		}
		if (dist < 250) {
			return (ret * 5);
		}
		if (dist < 500) {
			return ret * 2;
		}
		return ret;
	}

	// TODO
	function getIndustryTransportCost(start, endLoc, cargoID) {
		// TODO:something with production levels
		local dist = abs(sqrt(AIMap.DistanceSquare(AIIndustry.GetLocation(start), endLoc)));

		if (AIIndustry.IsBuiltOnWater(start) && AIIndustry.HasHeliport(start) && getEngine(cargoID, dist, AIAirport.PT_HELICOPTER) != -1) {
			Debug("Industry on water and has heliport!");
			return 2;
		}

		if (getEngine(cargoID, dist) == -1)
			return 255;

		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_AIR);
		if (vehList.Count() >= maxPlanes) {
			Warning("Reached maximum amount of planes!");
			return 254;
		}

		if (dist < 100) {
			return 20;
		}
		if (dist < 250) {
			return 10;
		}
		if (dist < 400) {
			return 6;
		}
		return 4;
	}

	function getEngine(cargoID, distance = 100, craftType = 0) {
		local engine_list = AIEngineList(AIVehicle.VT_AIR);
		engine_list.Valuate(AIEngine.CanRefitCargo, cargoID);
		engine_list.KeepValue(1);
		if (craftType != 0) {
			engine_list.Valuate(AIEngine.GetPlaneType);
			engine_list.KeepValue(craftType);
		}

		engine_list.Valuate(OtviTransport.profitValuator, cargoID, distance);
		engine_list.KeepAboveValue(0);

		engine_list.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
		if (engine_list.IsEmpty()) {
			Warning("Can't find a (profitable) aircraft for this cargo!!!");
			return -1;
		}

		local engine = engine_list.Begin();
		while (AIEngine.GetMaximumOrderDistance(engine) != 0 && AIEngine.GetMaximumOrderDistance(engine) < (distance + 20) && !(engine_list.IsEnd())) {
			Warning("Picked next engine due to order distance limitations on " + AIEngine.GetName(engine));
			engine = engine_list.Next();
		}

		if (AIEngine.IsValidEngine(engine)) {
			if (distance != 100)
				Debug("Picked aircraft: " + AIEngine.GetName(engine) + "; max distance: " + AIEngine.GetMaximumOrderDistance(engine));
			return engine;
		}
		return -1;
	}

	function updateEngines() {
		engine = getEngine(::PASSENGER_ID);

		if(AIAirport.IsValidAirportType(AIAirport.AT_SMALL)) {
			airportType = AIAirport.AT_SMALL;
			return;
		}
		if(AIAirport.IsValidAirportType(AIAirport.AT_COMMUTER)) {
			airportType = AIAirport.AT_COMMUTER;
			return;
		}
		if(AIAirport.IsValidAirportType(AIAirport.AT_METROPOLITAN)) {
			airportType = AIAirport.AT_METROPOLITAN;
			return;
		}
		if(AIAirport.IsValidAirportType(AIAirport.AT_LARGE)) {
			airportType = AIAirport.AT_LARGE;
			return;
		}
		if(AIAirport.IsValidAirportType(AIAirport.AT_INTERNATIONAL)) {
			airportType = AIAirport.AT_INTERNATIONAL;
			return;
		}
		if(AIAirport.IsValidAirportType(AIAirport.AT_INTERCON)) {
			airportType = AIAirport.AT_INTERCON;
			return;
		}
	}

	function stationLocationValuator(tile, airportType) {
		local landTiles = AITileList();
		local height = AIAirport.GetAirportHeight(airportType);
		local width = AIAirport.GetAirportWidth(airportType);

		if (AITile.IsBuildableRectangle(tile, width , height)) {
			landTiles.AddRectangle(tile, tile + AIMap.GetTileIndex(width - 1, height - 1));
			//landTiles.Valuate(AITile.GetSlope);
			//landTiles.KeepValue(AITile.SLOPE_FLAT);
			if (landTiles.Count() == height * width) {
				return 1;
			}
		}
		return 0;
	}
	
	function connectIndustry(source, dest, cargo, destIsTown) {
		if (getEngine(cargo) == -1) {
			Warning("Can't connect industries as there's no suitable aircraft available");
			return false;
		}
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_AIR);
		if (vehList.Count() >= maxPlanes) {
			Warning("Reached maximum amount of planes!");
			return false;
		}

		// trivial derivation:
		local source_type = AIIndustry.GetIndustryType(source);
		local location1 = AIIndustry.GetLocation(source);
		local location2;

		local airport2 = null;
		local build1 = false;
		local build2 = false;
		local distance;
		local airport1;

		if (destIsTown) {
			location2 = AITown.GetLocation(dest);
		} else {
			location2 = AIIndustry.GetLocation(dest);
		}
		distance = abs(sqrt(AITile.GetDistanceSquareToTile(location1, location2)));

		if (AIIndustry.IsBuiltOnWater(source))
			Debug("Source is on water");
		if (AIIndustry.HasHeliport(source))
			Debug("Source has heliport");
		Debug("Helicopter for this cargo: " + getEngine(cargo, distance, AIAirport.PT_HELICOPTER))

		if (AIIndustry.IsBuiltOnWater(source) && AIIndustry.HasHeliport(source) && getEngine(cargo, distance, AIAirport.PT_HELICOPTER) != -1) {
			Warning("Source industry is on water and has an heliport!");
			airport1 = AIIndustry.GetHeliportLocation(source);
		} else {
			airport1 = buildAirport(location1, cargo, true);
			build1 = true;	// always building, TODO: check for existing station?
		}

		if (!airport1)
			return false; // bugger

		airport2 = buildAirport(location2, cargo, false);
		if (!airport2) {
			if (build1)
				AITile.DemolishTile(airport1);
			return false; // bugger
		}
		build2 = true;

		local cargoAmount = (AIIndustry.GetLastMonthProduction(source, cargo) - AIIndustry.GetLastMonthTransported(source, cargo));
		if (!connectAirports(airport1, airport2, cargo, cargoAmount, AIAirport.PT_HELICOPTER)) {
			Warning("Failed to connect airports");
			if (build1)
				AITile.DemolishTile(airport1);
			if (build2)
				AITile.DemolishTile(airport2);
			return false;
		} 
	}

	function connectAirports(airport1, airport2, cargo, cargoAmount, planeType = 0) {
		local hangar;
		if (planeType == AIAirport.PT_HELICOPTER)
			hangar = AIAirport.GetHangarOfAirport(airport2);	// the oil rigs don't have a hangar
		else
			hangar = AIAirport.GetHangarOfAirport(airport1);

		local engine = getEngine(cargo, abs(sqrt(AIMap.DistanceSquare(airport1, airport2))), planeType);
		local count = cargoAmount/AIEngine.GetCapacity(engine);
		local ret = false;

		Debug("Want to deploy " + count + " aircraft of type: " + AIEngine.GetName(engine));
		if (count < 2) {
			Warning("changed to 2");
			count = 2;
		}
		if (count > 6) {
			Warning("changed to 6");
			count = 6;
		}

		while (count--) {
			arrangeBalance(AIEngine.GetPrice(engine));
			local plane1 = AIVehicle.BuildVehicle(hangar, engine);
			if (!AIVehicle.IsValidVehicle(plane1)) {
				Error("Error building plane: " + AIError.GetLastErrorString());
				return ret;
			}
			if(!AIVehicle.RefitVehicle(plane1, cargo)) {
				Debug("Error refitting plane: " + AIError.GetLastErrorString());
				return ret;
			}

			Debug("Built plane: " + plane1);
			//lastPassengerVehicle = train1;
			if (!AIOrder.AppendOrder(plane1, airport1, AIOrder.OF_FULL_LOAD_ANY))
				Error("Error giving orders: " + AIError.GetLastErrorString());
			if(!AIOrder.AppendOrder(plane1, airport1, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED))
				Error("Error giving orders: " + AIError.GetLastErrorString());

			if (OtviTransport.isHinUndWiederCargo(cargo)) {
				if (!AIOrder.AppendOrder(plane1, airport2, AIOrder.OF_FULL_LOAD_ANY))
					Error("Error giving orders: " + AIError.GetLastErrorString());
			} else {
				if(!AIOrder.AppendOrder(plane1, airport2, 0))
					Error("Error giving orders: " + AIError.GetLastErrorString());
			}

			if(!AIOrder.AppendOrder(plane1, airport2, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED))
				Error("Error giving orders: " + AIError.GetLastErrorString());
			AIVehicle.StartStopVehicle(plane1);
			ret = true;

			if (OtviTransport.isHinUndWiederCargo(cargo)) {
				local airportSwap = airport1;
				airport1 = airport2;
				airport2 = airportSwap;
			}
		}
		return ret;	// TODO change to count 
	}
	
	function connect2Towns(biggestTown, connectTown, cargo = 0) {
		local build1 = false;
		local build2 = false;

		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_AIR);
		if (vehList.Count() >= maxPlanes) {
			Warning("Reached maximum amount of aircraft!");
			return false;
		}

		if (cargo == 0) {
			cargo = ::PASSENGER_ID;
		}

		if (getEngine(cargo, AIMap.DistanceManhattan(AITown.GetLocation(biggestTown), AITown.GetLocation(connectTown))) == -1) {
			Warning("Can't connect towns as there's no suitable aircraft available!");
			return false;
		}

		local airport1 = getAirport(biggestTown);
		if (airport1 != 0) {
			Debug("Reusing old existing airport");
		} else {
			airport1 = buildAirport(AITown.GetLocation(biggestTown), cargo, true);
			if (!airport1) {
				Warning("!!! Building airport failed in: " + AITown.GetName(biggestTown));
				return false;
			}
			build1 = true;
		}
		
		local airport2 = getAirport(connectTown);
		if (airport2 != 0) {                                                                                                             
			Debug("Using old existing station instead...");
		} else {
			airport2 = buildAirport(AITown.GetLocation(connectTown), cargo, false);
			if (!airport2) {
				Warning("!!! Building airport failed in: " + AITown.GetName(connectTown));
				if (build1) {
					Warning("Removing other airport...");
					AITile.DemolishTile(airport1);
				}
				return false;
			}
			build2 = true;
		}

		local prod1 = AITown.GetLastMonthProduction(biggestTown, ::PASSENGER_ID) * (100 - AITown.GetLastMonthTransportedPercentage(biggestTown, ::PASSENGER_ID));
		local prod2 = AITown.GetLastMonthProduction(biggestTown, ::PASSENGER_ID) * (100 - AITown.GetLastMonthTransportedPercentage(biggestTown, ::PASSENGER_ID));
		local prod = prod1 < prod2 ? prod1 : prod2;
		if (connectAirports(airport1, airport2, cargo, prod)) {
			if (build1)
				addAirport(biggestTown, airport1, airportType);
			if (build2)
				addAirport(connectTown, airport2, airportType);
			return true;
		} else {
			Warning("Failure... cleaning up....");
			// remove stuff we have build
			if (build1) {
				Warning("Cleaning airport 1");
				local planes = AIVehicleList_Station(AIStation.GetStationID(airport1));
				if (planes.IsEmpty())
					AITile.DemolishTile(airport1);
			}
			if (build2) {
				Warning("Cleaning airport 2");
				local planes = AIVehicleList_Station(AIStation.GetStationID(airport2));
				if (planes.IsEmpty())
					AITile.DemolishTile(airport2);
			}
			return false;
		}
	}

	function getAirport(town) {
		return myAirportTownList.GetValue(town);
	}

	function getAirportType(town) {
		if (getAirport(town))
			return myAirportTypeList.GetValue(town);
		return 0;
	}

	function addAirport(town, station, airportType) {
		myAirportTownList.SetValue(town, station);
		myAirportTypeList.SetValue(town, airportType);
	}

	function buildAirport(target, cargo, isSource) {
		Debug("Building airport..." + isSource);
		// Find empty block of squares as close to target as possible
		local spotFound = false;
		local curRange = 1;
		local maxRange = 20; //TODO check value correctness (coverage + size of something?)
		local width = AIAirport.GetAirportWidth(airportType);
		local height = AIAirport.GetAirportHeight(airportType);

		Debug("Looking for a spot: " + width + " by " + height);
		local terraformed = 0;	//prevent AI from trying to level a town

		local area = AITileList();

		arrangeBalance(AIAirport.GetPrice(airportType) * 2);

		while (curRange <= maxRange) {
			//Debug("looking with range " + curRange);
			SafeAddRectangle(area, target, curRange, curRange - 1);
			if (isSource || OtviTransport.isHinUndWiederCargo(cargo)) {
				area.Valuate(AITile.GetCargoProduction, cargo, width, height, stationRadius);
				area.RemoveValue(0);
			} 
			if (!isSource || OtviTransport.isHinUndWiederCargo(cargo))  {
				area.Valuate(AITile.GetCargoAcceptance, cargo, width, height, stationRadius);
				area.RemoveBelowValue(8);
			}
			//Debug("Found " + area.Count() + " possible options before valuator");
			area.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
			area.Valuate(stationLocationValuator, airportType);
			area.RemoveValue(0);
			if (area.Count()) {
				Debug("Found " + area.Count() + " possible options");
				for (local t = area.Begin(); !area.IsEnd(); t = area.Next()) {
					if (AIAirport.BuildAirport(t, airportType, AIStation.STATION_JOIN_ADJACENT)) {
						return t;
					} else if (terraformed++ < 4){
						Warning("Building airport failed due to: " + AIError.GetLastErrorString());
						// try again with terraforming
						if (!AITile.LevelTiles(t, t + AIMap.GetTileIndex(width, height)))
							Error(AIError.GetLastErrorString());
						if (AIAirport.BuildAirport(t, airportType, AIStation.STATION_JOIN_ADJACENT)) {
							return t;
						} else {
							Warning("Building airport failed due to: " + AIError.GetLastErrorString());
						}
					}
				}
				curRange++;
			} else {
				curRange++;
			}
		}
		Debug("No possible airport location found");
		return null;
	}

	/**
	 * upgrades small station to bigger one
	 * also reroutes trains to new station if needed
	 */
	function UpgradeAirport(town) {}
}

