/**
 *    This file is part of OtviAI.
 *
 *    OtviAI is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    OtviAI is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with OtviAI.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2008-2013 Otto Visser.
 * Suggestions/comments and bugs can go to: OtviAI@otvi.nl
 */

require("util.nut");

class OtviMap {

	function constructor() {
		::servedTownList <- AITownList();
		::servedIndustryList <- AIIndustryList();
	}

	static function isTownServed(town) {
		return ::servedTownList.GetValue(town); 
	}

	static function isIndustryServed(industry) {
		return ::servedIndustryList.GetValue(industry); 
	}

	function setTownServed(town, service = 1) {
		servedTownList.SetValue(town, service);
	}

	function setIndustryServed(industry, service = 1) {
		servedIndustryList.SetValue(industry, service);
	}

	function cleanUp() {
		//check stations with no vehicles and remove those
		Debug("Cleaning up stations...");
		local stations = AIStationList(AIStation.STATION_ANY);
		local station = stations.Begin();
		while (!stations.IsEnd()) {
			local success = false;
			if (AIVehicleList_Station(station).IsEmpty()) {
				Warning("Cleaning up unused station: " + AIStation.GetName(station));
				arrangeBalance(5000);	// TODO random value that should be enough?

				foreach (tile, val in AITileList_StationType(station, AIStation.STATION_BUS_STOP)) {
					if (AIRoad.HasRoadType(tile, AIRoad.ROADTYPE_TRAM))
						AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_TRAM);
					else
						AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);

					if (AIRoad.RemoveRoadStation(tile)) {
						Warning("Removed bus stop");
						success = true;
					} else {
						Error("Couldn't properly remove bus stop: " + AIError.GetLastErrorString());
					}
				}
				foreach (tile, val in AITileList_StationType(station, AIStation.STATION_TRUCK_STOP)) {
					if (AIRoad.HasRoadType(tile, AIRoad.ROADTYPE_TRAM))
						AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_TRAM);
					else
						AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);

					if (AIRoad.RemoveRoadStation(tile)) {
						Warning("Removed truck stop");
						success = true;
					} else {
						Error("Couldn't properly remove truck stop: " + AIError.GetLastErrorString());
					}
				}
				foreach (tile, val in AITileList_StationType(station, AIStation.STATION_TRAIN)) {
					if (AIRail.RemoveRailStationTileRectangle(tile, tile, false)) {
						Warning("Removed piece of a train station");
						success = true;
					} else {
						Error("Couldn't remove part of a rail station: " + AIError.GetLastErrorString());
					}
				}

				if (AIStation.HasStationType(station, AIStation.STATION_AIRPORT)) {
					if (AIAirport.RemoveAirport(AIStation.GetLocation(station))) {
						Warning("Removed unused airport");
						success = true;
					} else {
						Error("Couldn't properly remove airport: " + AIError.GetLastErrorString());
					}
				}
				foreach (tile, val in AITileList_StationType(station, AIStation.STATION_DOCK)) {
					if (AIMarine.RemoveDock(tile)) {
						Warning("Removed a dock");
						success = true;
					} else {
						Error("Couldn't remove dock: " + AIError.GetLastErrorString());
					}
				}

				if (!success) {
					foreach (tile, val in AITileList_StationType(station, AIStation.STATION_ANY)) {
						if (AITile.DemolishTile(tile))
							Warning("Successfully demolished part of a station");
						else
							Error("Couldn't even demolish this part of the station!" + AIError.GetLastErrorString());
					}
				}
			}
			station = stations.Next();
		}
		// TODO clean up depots; requires adding depots to orders for busses/trucks/trams!
		// OR: get all vehicles, get their orders, get their destination tileindexes, get all depottiles
		// keep depottiles used in orders

		// TODO cleanup unused road and rail

		Debug("Cleaning up town list...");
		local townlist = AITownList();
		townlist.Valuate(isTownServed);
		townlist.KeepValue(-1);
		local town = townlist.Begin();
		while (!townlist.IsEnd()) {
			Warning("Re-enabling town " + AITown.GetName(town));
			setTownServed(town, 0);
			town = townlist.Next();
		}

		Debug("Cleaning up industry list...");
		local indlist = AIIndustryList();
		indlist.Valuate(isIndustryServed);
		indlist.KeepValue(-1);
		local ind = indlist.Begin();
		while (!indlist.IsEnd()) {
			Warning("Re-enabling industry " + AIIndustry.GetName(ind));
			setIndustryServed(ind, 0);
			ind = indlist.Next();
		}

		Debug("Cleaning up town service status...");
		townlist = AITownList();
		townlist.Valuate(isTownServed);
		townlist.KeepAboveValue(0);
		town = townlist.Begin();
		while (!townlist.IsEnd()) {
			stations = AIStationList(AIStation.STATION_ANY);
			stations.Valuate(AIStation.GetNearestTown);
			stations.KeepValue(town);
			if (stations.IsEmpty()) {
				Warning("Town " + AITown.GetName(town) + " is no longer getting service!");
				setTownServed(town, -1);
			}
			town = townlist.Next();
		}
		Debug("Cleaning up industry service status...");
		local industryList = AIIndustryList();
		industryList.Valuate(isIndustryServed);
		industryList.KeepAboveValue(0);
		local industry = industryList.Begin();
		while (!industryList.IsEnd()) {
			if (AIIndustry.GetAmountOfStationsAround(industry) == 0) {
				Warning("Industry " + AIIndustry.GetName(industry) + " is no longer getting service!");
				setIndustryServed(industry, -1);
			}
			industry = industryList.Next();
		}
	}

	// 6.3 tiles per day for passengers? // TODO: check or calculate
	// TODO: subtract maintenance!
	function incomeSort(targetTown, startTile, cargoID) {
		local distance = AIMap.DistanceManhattan(startTile, AITown.GetLocation(targetTown));
		local days = distance/7 + 1;	//underestimate
		//Debug(AITown.GetName(targetTown) + " at distance: " + distance + ", days: " + days + " income: " + AICargo.GetCargoIncome(cargoID, distance, days) + " maxProd: " + AITown.GetMaxProduction(targetTown, cargoID));
		return AICargo.GetCargoIncome(cargoID, distance, days) * (AITown.GetLastMonthProduction(targetTown, cargoID) - AITown.GetLastMonthSupplied(targetTown, cargoID)) * (365/days);
	}
	
	function getTownPotential(a) {
		return AITown.GetLastMonthProduction(a, ::MAIL_ID) - AITown.GetLastMonthSupplied(a, ::MAIL_ID);
	}
}

