/**
 *    This file is part of OtviAI.
 *
 *    OtviAI is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    OtviAI is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with OtviAI.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2008-2013 Otto Visser.
 * Suggestions/comments and bugs can go to: otviAI@otvi.nl
 */

require("util.nut");
require("RailFinder/RailFinder.nut");

class OtviTrAIn {
	cargoTrainEngine = -1;
	passengerTrainEngine = -1;
	passengerCart = -1;
	mailCart = -1;

	railFinder = null;
	COST_RAIL = 500;
	stationRadius = -1;

	myRailStationTownList = null;
	myRailStationLengthList = null;
	myStationsTileList = null;

	current_town_rail_type = 0;
	current_cargo_rail_type = 0;
	maxTrains = -1;

	function constructor() {
		Debug("Initializing train transportation...");
		railFinder = RailMaker();

		railFinder.cost.tile = 1;
		railFinder.cost.max_cost = 30000;	// TODO: still an estimate
		railFinder.cost.turn = COST_RAIL;		// The cost that is added to _cost_tile if the direction changes.
		railFinder.cost.diagonal_tile = 0;
		railFinder.cost.new_rail = COST_RAIL;		// was: 5
		railFinder.cost.slope =	COST_RAIL * 2;		// The extra cost if a rail tile is sloped.
		railFinder.cost.bridge_per_tile = COST_RAIL * 4;	// The cost per tile of a new bridge, this is added to _cost_tile.
		railFinder.cost.tunnel_per_tile = COST_RAIL * 4;	// The cost per tile of a new tunnel, this is added to _cost_tile.
		railFinder.cost.coast =	COST_RAIL;		// The extra cost for a coast tile.
		railFinder.cost.max_bridge_length = 10;	// The maximum length of a bridge that will be build.
		railFinder.cost.max_tunnel_length = 10;	// The maximum length of a tunnel that will be build.

		stationRadius = AIStation.GetCoverageRadius(AIStation.STATION_TRAIN);

		// init town list
		myRailStationTownList = AITownList();
		myRailStationLengthList = AIList();
		myStationsTileList = AIList();

		maxTrains = AIGameSettings.GetValue("vehicle.max_trains");
		Debug("Max trains: " + maxTrains);
	}

	function getName() {
		return "train";
	}

	function isOperational() {
		if (passengerTrainEngine == -1)
			return false;
		return true;
	}

	function getExpansionCost(cargo, distance = 100) {
		local expandMoneyMin;
		if (cargo == ::PASSENGER_ID) {
			if (distance != 100) {
				expandMoneyMin = 3 * AIEngine.GetPrice(getCart(::PASSENGER_ID, current_town_rail_type));
				expandMoneyMin += 1 * AIEngine.GetPrice(getCart(::MAIL_ID, current_town_rail_type));
				expandMoneyMin += AIEngine.GetPrice(getEngine(current_town_rail_type, ::PASSENGER_ID, distance));
			} else {
				expandMoneyMin = 3 * AIEngine.GetPrice(passengerCart);
				expandMoneyMin += 1 * AIEngine.GetPrice(mailCart);
				expandMoneyMin += AIEngine.GetPrice(passengerTrainEngine);
			}
			// 2 stations (21 station tiles), rails and a train with carts, plus 5000 for extra things like depots and signals
			return (expandMoneyMin + distance * AIRail.GetBuildCost(current_town_rail_type, AIRail.BT_TRACK) * 2 + 21 * AIRail.GetBuildCost(current_town_rail_type, AIRail.BT_STATION) + 2 * AIRail.GetBuildCost(current_town_rail_type, AIRail.BT_DEPOT) + 6 * AIRail.GetBuildCost(current_town_rail_type, AIRail.BT_SIGNAL));
		} else {
			expandMoneyMin = 4 * AIEngine.GetPrice(getCart(cargo, current_cargo_rail_type));
			expandMoneyMin += AIEngine.GetPrice(getEngine(current_cargo_rail_type, cargo, distance));
			// 2 stations (21 station tiles), rails and a train with carts, plus 5000 for extra things like depots and signals
			return (expandMoneyMin + distance * AIRail.GetBuildCost(current_cargo_rail_type, AIRail.BT_TRACK) * 2 + 21 * AIRail.GetBuildCost(current_cargo_rail_type, AIRail.BT_STATION) + 2 * AIRail.GetBuildCost(current_cargo_rail_type, AIRail.BT_DEPOT) + 6 * AIRail.GetBuildCost(current_cargo_rail_type, AIRail.BT_SIGNAL));
		}
		//Debug("Expansion over " + distance + " distance will cost " + (expandMoneyMin + distance * 256 + 2 * 9000));
	}

	function canEnhance() {
		// TODO; dependant on track configuration, cargo, amount of trains etc
		return false;
	}

	function getEnhanceCost() {
		local expandMoneyMin = AIEngine.GetPrice(passengerTrainEngine);
		expandMoneyMin += 5 * AIEngine.GetPrice(passengerCart);
		expandMoneyMin += 2 * AIEngine.GetPrice(mailCart);
		return expandMoneyMin;
	}

	function getTownTransportCost(startLoc, endLoc) {
		if (passengerCart == -1) {
			Warning("Rail transport not yet available!");
			return 255;
		}
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_RAIL);
		if (vehList.Count() >= maxTrains) {
			Warning("Reached maximum amount of trains!");
			return 254;
		}

		local ret = 1;
		if (AITown.GetPopulation(startLoc) < 1000 || AITown.GetPopulation(endLoc) < 1000)
			ret = 3;

		local dist = abs(sqrt(AIMap.DistanceSquare(AITown.GetLocation(startLoc), AITown.GetLocation(endLoc))));
		//Debug("Train from " + AITown.GetName(startLoc) + " to " + AITown.GetName(endLoc) + " has a distance of " + dist);
		if (dist < 25 || dist > 200) {
			// too damn close for train or too damn far for current pathfinder
			return (ret * 8);
		}
		if (dist < 42 || dist > 100) {
			return (ret * 4);
		}
		return ret;
	}

	function getIndustryTransportCost(start, endLoc, cargoID) {
		if (getCart(cargoID, current_cargo_rail_type) == -1)
			return 255;

		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_RAIL);
		if (vehList.Count() >= maxTrains) {
			Warning("Reached maximum amount of trains!");
			return 254;
		}

		// TODO:something with production levels
		local dist = abs(sqrt(AIMap.DistanceSquare(AIIndustry.GetLocation(start), endLoc)));
		if (dist < 25 || dist > 200) {
			// too damn close for train or too damn far for current pathfinder
			return 9;
		}
		if (dist < 50 || dist > 100) {
			return 5;
		}
		else
			return 1;
	}

	function getEngine(rail_type, cargoID, distance = 100) {
		local trainEngine_list = AIEngineList(AIVehicle.VT_RAIL);
		trainEngine_list.Valuate(AIEngine.CanRunOnRail, rail_type);
		trainEngine_list.KeepValue(1);
		trainEngine_list.Valuate(AIEngine.HasPowerOnRail, rail_type);
		trainEngine_list.KeepValue(1);
		trainEngine_list.Valuate(AIEngine.IsWagon);
		trainEngine_list.KeepValue(0);
		trainEngine_list.Valuate(AIEngine.CanPullCargo, cargoID);
		trainEngine_list.KeepValue(1);	// false == false, true == maybe true
		if (cargoID != ::PASSENGER_ID && cargoID != ::MAIL_ID) {
			trainEngine_list.Valuate(AIEngine.GetCapacity);
			trainEngine_list.KeepValue(-1);	
		}

		Debug("Found possible trains: " + trainEngine_list.Count());
		trainEngine_list.Valuate(OtviTransport.profitValuator, cargoID, distance);	// TODO sub, production
		trainEngine_list.KeepAboveValue(0);
		Debug("Out of which would be profitable: " + trainEngine_list.Count());

		trainEngine_list.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
		if (trainEngine_list.IsEmpty()) {
			Warning("Can't find an engine for current rail type!!!");
			return -1;
		}

		local trainEngine = trainEngine_list.Begin();
		while (AIEngine.GetMaximumOrderDistance(trainEngine) != 0 && AIEngine.GetMaximumOrderDistance(trainEngine) < (distance + 20) && !(trainEngine_list.IsEnd())) {
			Warning("Picked next engine due to order distance limitations on " + AIEngine.GetName(trainEngine));
			trainEngine = trainEngine_list.Next();
		}

		if (AIEngine.IsValidEngine(trainEngine)) {
			Debug("Picked train engine: " + AIEngine.GetName(trainEngine));
			return trainEngine;
		}
		return -1;
	}

	function getCart(cargoID, railType, distance = 100, staticCall = false, skip = 0) {
		local cart_list = AIEngineList(AIVehicle.VT_RAIL);
		cart_list.Valuate(AIEngine.CanRefitCargo, cargoID);
		cart_list.KeepValue(1);

		cart_list.Valuate(AIEngine.IsWagon);
		cart_list.KeepValue(1);
		
		if (cart_list.IsEmpty()) {
			Warning("No suitable cart for " + AICargo.GetCargoLabel(cargoID) + " available (yet?) on any rail type!!");
			return -1;
		}

		cart_list.Valuate(AIEngine.CanRunOnRail, railType);
		cart_list.KeepValue(1);
		
		if (cart_list.IsEmpty()) {
			Warning("There is a cart for this cargo, but not on the current rail type!!!");
			if (staticCall)
				return -1;

			cart_list = AIEngineList(AIVehicle.VT_RAIL);
			cart_list.Valuate(AIEngine.CanRefitCargo, cargoID);
			cart_list.KeepValue(1);

			cart_list.Valuate(AIEngine.IsWagon);
			cart_list.KeepValue(1);
			cart_list.Valuate(AIEngine.GetCapacity);
			cart_list.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
			Debug("carts for cargo count: " + cart_list.Count());
			local cart = cart_list.Begin();
			while (skip-- > 0)
				cart = cart_list.Next();

			local railList = AIRailTypeList();
			railList.Valuate(AIRail.GetMaxSpeed);
			railList.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
			local new_type = railList.Begin();
			while (!AIEngine.CanRunOnRail(cart, new_type) && !railList.IsEnd() && getEngine(new_type, cargoID, distance) != -1)
				new_type = railList.Next();

			if (AIEngine.CanRunOnRail(cart, new_type)) {
				Warning("Found working rail type and cart combination; switching rail type to " + AIRail.GetName(new_type));
				if (cargoID == ::PASSENGER_ID || cargoID == ::MAIL_ID) {
					current_town_rail_type = new_type;
					passengerTrainEngine = getEngine(current_town_rail_type, cargoID, distance);
				} else {
					current_cargo_rail_type = new_type;
					cargoTrainEngine = getEngine(current_cargo_rail_type, cargoID, distance);
				}
				AIRail.SetCurrentRailType(new_type);
				return cart;
			}
		} else {
			cart_list.Valuate(AIEngine.GetCapacity);
			cart_list.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);

			local cart = cart_list.Begin();
			while (skip-- > 0)
				cart = cart_list.Next();

			if (!staticCall) {
				Debug("Picked cart for cargo " + AICargo.GetCargoLabel(cargoID) + ": " + AIEngine.GetName(cart));
				if (cargoID == ::PASSENGER_ID) {
					passengerTrainEngine = OtviTrAIn.getEngine(railType, ::PASSENGER_ID, distance);
				} else {
					cargoTrainEngine = OtviTrAIn.getEngine(railType, cargoID, distance);
				}
			}

			return cart;
		}
		return -1;
	}

	function getRailProfit(railType) {
		local train = OtviTrAIn.getEngine(railType, ::PASSENGER_ID);
		return OtviTransport.profitValuator(train, ::PASSENGER_ID);
	}

	function updateEngines() {
		local railList = AIRailTypeList();
		Debug("Rail type count: " + railList.Count());
		railList.Valuate(getRailProfit);
		railList.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);

		local new_type = railList.Begin();
		if (new_type != current_town_rail_type) {
			Warning("Changing passengers to new rail type: " + AIRail.GetName(new_type));
			/*for (local i = 0; i < 10; i++) {
			// TODO start index dependant on freeform edges!!!
			if (AIRail.ConvertRailType(AIMap.GetTileIndex(1, 1), AIMap.GetTileIndex(AIMap.GetMapSizeX() - 2, AIMap.GetMapSizeY() - 2 ), new_type)) {
			Debug("Rail conversion success");
			} else {
			Error("At least partial failure converting rail!");
			}
			::INSTANCE.Sleep(25);
			}*/
			current_town_rail_type = new_type;
		}
		passengerCart = getCart(::PASSENGER_ID, current_town_rail_type);
		mailCart = getCart(::MAIL_ID, current_town_rail_type);
	}

	function stationLocationValuator(tile, length, width) { // TODO check in 4 directions!!!
		local landTiles = AITileList();
		if (AITile.IsBuildableRectangle(tile, length + 1, width)) {
			landTiles.AddRectangle(tile, tile + AIMap.GetTileIndex(length - 1, width - 1));
			//landTiles.Valuate(AITile.GetSlope);
			//landTiles.KeepValue(AITile.SLOPE_FLAT);
			if (landTiles.Count() == length * width) {
				return AIRail.RAILTRACK_NE_SW;
			}
			// Warning("location only has limited flat tiles in NE_SW dir: " + landTiles.Count());
		}
		if (AITile.IsBuildableRectangle(tile, width, length + 1)) {
			landTiles.AddRectangle(tile, tile + AIMap.GetTileIndex(width - 1, length- 1));
			//landTiles.Valuate(AITile.GetSlope);
			//landTiles.KeepValue(AITile.SLOPE_FLAT);
			if (landTiles.Count() == length * width) {
				return AIRail.RAILTRACK_NW_SE;
			}
			// Warning("location only has limited flat tiles in NW_SE dir: " + landTiles.Count());
		}
		return 0;
	}
	
	function connectIndustry(source, dest, cargo, destIsTown) {
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_RAIL);
		if (vehList.Count() >= maxTrains) {
			Warning("Reached maximum amount of train!");
			return false;
		}
		// trivial derivation:
		local source_type = AIIndustry.GetIndustryType(source);
		local location1 = AIIndustry.GetLocation(source);
		//AISign.BuildSign(location1, "S");

		local location2;
		if (destIsTown) {
			location2 = AITown.GetLocation(dest);
		} else {
			location2 = AIIndustry.GetLocation(dest);
		}
		local distance = AITile.GetDistanceManhattanToTile(location1, location2);
		if (getEngine(current_cargo_rail_type, cargo, distance) == -1) {
			Warning("No train available for this connection");
			return false;
		}

		AIRail.SetCurrentRailType(current_cargo_rail_type);

		// TODO: check whether this makes sense; esp. with ECS where production will increase once serviced.
		local lengthProd = abs(AIIndustry.GetLastMonthProduction(source, cargo) / AIEngine.GetCapacity(getCart(cargo, current_cargo_rail_type)));
		local length1 = lengthProd;
		if (length1 < 3)
			length1 = 3;	// shorter looks ugly and weird
		if (length1 > 7) {
			Debug("Limiting station size from " + length1 + " to 7");
			length1 = 7;
		}
		Debug("Station length: " + length1);
		local length2 = length1;
		local width2 = 2;
		local station2 = null;
		local build1 = false;
		local build2 = false;
		local station1;

		if (destIsTown) {
			local goal_type = 0xFF;	// TODO: AIIndustryType.INDUSTRYTYPE_TOWN;
			station1 = buildIndustryStation(source, length1, 1, cargo, source_type, goal_type, distance, true);
			if (!station1)
				return false; // bugger
			build1 = true;	// always building, TODO: check for existing station?
			if (!station2 || AIRail.GetRailType(station2) != current_cargo_rail_type) {
				length2 = abs(sqrt(AITown.GetPopulation(dest)/100)); // adapt length1? TODO
				if (length2 < lengthProd)
					length2 = lengthProd;

				station2 = BuildTrainStation(dest, length2);
				if (!station2) {
					AITile.DemolishTile(station1);
					return false;
				}
				build2 = true;
			}
		} else {
			//AISign.BuildSign(location2, "D");
			local goal_type = AIIndustry.GetIndustryType(dest);
			station1 = buildIndustryStation(source, length1, 1, cargo, source_type, goal_type, distance, true);
			if (!station1)
				return false; // bugger
			build1 = true;	// always building, TODO: check for existing station?

			station2 = buildIndustryStation(dest, length2, width2, cargo, source_type, goal_type, distance, false);
			if (!station2)
				return false; // bugger
			build2 = true;
		}

		if (!connectStations(station1, length1, 1, station2, length2, width2)) {
			// check if route complete; build other orientation of depot
			Error("Incomplete route or unable to find suitable depot location");
			AITile.DemolishTile(station1);
			if (build1) {
				Warning("Removing train station...");
				AITile.DemolishTile(station1);
			}
			if (build2) {
				Warning("Removing train station...");
				AITile.DemolishTile(station2);
			}
			return false;
		} else {
			if (build1)
				myRailStationLengthList.AddItem(station1, length1);
			if (build2) {
				if (destIsTown)
					addTrainStation(dest, station2, length2);
				else
					myRailStationLengthList.AddItem(station2, length2);
			}
			return connectIndustryStations(station1, station2, cargo, AIIndustry.GetLastMonthProduction(source, cargo) - AIIndustry.GetLastMonthTransported(source, cargo));
		}
	}

	/**
	 * 2 stations build, make rail
	 */
	function connect(starts, goals, ignored = [], skipDepot = false, placeSignal = false) {
		local distance = AIMap.DistanceManhattan(starts[0][0], goals[0][0]);
		railFinder.cost.max_cost = 2 * COST_RAIL * distance;
		railFinder.InitializePath(starts, goals, ignored);
		local path = false;
		local i = 0;
		local err;
		// TODO: base on mountains?
		local iterations = 100 * distance; // if we pass this; something is wrong
		if (!placeSignal)
			iterations += 30000;
		else
			iterations += 1500;

		if (iterations > (200 * 100)) {	// it's roughly 100 iterations per day
			Warning("Limiting calculations to roughly 200 days");
			iterations = 200 * 100;
		}
		//local iterations = -1;
		// TODO check whether higher values helps finding faster
		Debug("pathfinding for max " + railFinder.cost.max_cost + " cost and iterations: " + iterations);
		local startTick = GetTick();
		while (path == false && iterations > 0) {
			path = railFinder.FindPath(100);
			iterations -= 100;
		}
		Debug("Total ticks passed with " + iterations + " left: " + (GetTick() - startTick));
		
		if (path == false) {
			Warning("!! No path found, too little iterations...");
			railFinder.reset();
			return -1;
		}
		if (path == null) {
			Warning("!! No path found, too expensive...");
			railFinder.reset();
			return -1;
		}

		Debug("path found for cost: " + path.GetCost() + ", building");

		arrangeBalance(distance * AIRail.GetBuildCost(AIRail.GetCurrentRailType(), AIRail.BT_TRACK) * 2 + AIRail.GetBuildCost(AIRail.GetCurrentRailType(), AIRail.BT_DEPOT) + 3 * AIRail.GetBuildCost(AIRail.GetCurrentRailType(), AIRail.BT_SIGNAL));

		local prev = null;
		local prevprev = null;
		local prev3 = null;
		local depotBuilt = false;
		local built = 0;
		local depotLoc = null;
		local depotOpening;
		local railCount = 0;

		while (path != null) {
			if (prevprev != null) {
				if (AIMap.DistanceManhattan(prev, path.GetTile()) > 1) {
					if (AITunnel.GetOtherTunnelEnd(prev) == path.GetTile()) {
						AITunnel.BuildTunnel(AIVehicle.VT_RAIL, prev);
					} else {
						local bridge_list = AIBridgeList_Length(AIMap.DistanceManhattan(path.GetTile(), prev) + 1);
						bridge_list.Valuate(AIBridge.GetMaxSpeed);
						bridge_list.Sort(AIList.SORT_BY_VALUE, false);
						AIBridge.BuildBridge(AIVehicle.VT_RAIL, bridge_list.Begin(), prev, path.GetTile());
					}
					prevprev = prev;
					prev = path.GetTile();
					path = path.GetParent();
				} else {
					if (!AIRail.BuildRail(prevprev, prev, path.GetTile())) {
						switch(AIError.GetLastError()) {
							case AIError.ERR_VEHICLE_IN_THE_WAY:
								Warning("Building rail failed temporary: vehicle in the way...");
								// try again till the stupid vehicle moved?
								local tries = 100;
								while(!AIRail.BuildRail(prevprev, prev, path.GetTile()) && tries--) {
									::INSTANCE.Sleep(10);
								}
								if (tries == 0) {
									Warning("Vehicle didn't move, building around it...");
									return connect(starts, goals, ignored, skipDepot);	// try again
								} else {
									Debug("Vehicle moved; continueing...");
								}
								break;
							case AIRoad.ERR_ROAD_WORKS_IN_PROGRESS:
								Warning("Building rail failed: road works in progress");
								// good, someone else is building this, skip tile?
								break;
							case AIError.ERR_ALREADY_BUILT:
								// Debug("Building rail failed: already built");
								// Convert if needed
								if (AIRail.ConvertRailType(prevprev, prev, AIRail.GetCurrentRailType()))
									Warning("Converted some rail to new type");
								break;
							case AIError.ERR_LAND_SLOPED_WRONG:
								Warning("Building rail failed: wrong land slope");
								// not good; redo from start or terraform?
								return connect(starts, goals, ignored, skipDepot);
							case AIError.ERR_AREA_NOT_CLEAR:
								if (AITile.IsStationTile(prev)) {
									Warning("Building route through a station");
									break;	// just continue
								} else {
									Warning("Building rail failed: not clear, demolishing tile...");
									AISign.BuildSign(prev, "I'm on your land, demolishing your stuffs");
									// demolish and retry?
									if (AITile.DemolishTile(path.GetTile())) {
										AIRail.BuildRail(prevprev, prev, path.GetTile());
									} else {
										return connect(starts, goals, ignored, skipDepot);
									}
								}
								break;
							case AIRail.ERR_CROSSING_ON_ONEWAY_ROAD:
								Warning("Can't build crossing on one way road");
								return connect(starts, goals, ignored, skipDepot);
							case AIError.ERR_NOT_ENOUGH_CASH:
								Warning("Panic: Out of cash!!");
								local count = 10;
								local res = false;
								arrangeBalance(5 * AIRail.GetBuildCost(AIRail.GetCurrentRailType(), AIRail.BT_TRACK));
								while(!(res = AIRail.BuildRail(prevprev, prev, path.GetTile())) && count-- > 0)
									::INSTANCE.Sleep(100);

								if (!res)
									return -1;

								break;
							case AIError.ERR_OWNED_BY_ANOTHER_COMPANY:
								Warning("Owned by another company; retry...");
								return connect(starts, goals, ignored, skipDepot);
							default:
								Warning("Unknown error during track laying " + AIError.GetLastErrorString());
								break;	// no clue what this is; silent ignore for now
						}
					}
				}

				//if (!skipDepot && railCount++  == 0) {
				if (placeSignal) {
					if (AIRail.BuildSignal(prev, prevprev, AIRail.SIGNALTYPE_PBS))
						placeSignal = false;
				}
				// TODO FIXME
				/*if (railCount++ % 8 == 0) {
					AIRail.BuildSignal(prevprev, prev, AIRail.SIGNALTYPE_PBS_ONEWAY);
				}*/
			}
			if (!skipDepot && !depotBuilt && built++ > 5 && (prev3 != null) && (prevprev != null)) {
				if ((AIRail.GetRailTracks(prevprev) & AIRail.RAILTRACK_NE_SW) && (AIRail.GetRailTracks(prev) & AIRail.RAILTRACK_NE_SW) && (AIRail.GetRailTracks(prev3) & AIRail.RAILTRACK_NE_SW) ) {
					depotLoc = prevprev + AIMap.GetTileIndex(0,-1);
					depotOpening = prevprev;
					depotBuilt = AIRail.BuildRailDepot(depotLoc, prevprev);
					if (depotBuilt) {
						if (!(AIRail.BuildRailTrack(prevprev, AIRail.RAILTRACK_NW_NE) && AIRail.BuildRailTrack(prevprev, AIRail.RAILTRACK_NW_SW))) {
							depotBuilt = false;
							AITile.DemolishTile(depotLoc);
						}
					}
				}
				if ((AIRail.GetRailTracks(prevprev) & AIRail.RAILTRACK_NW_SE) && (AIRail.GetRailTracks(prev) & AIRail.RAILTRACK_NW_SE) && (AIRail.GetRailTracks(prev3) & AIRail.RAILTRACK_NW_SE)) {
					depotLoc = prevprev + AIMap.GetTileIndex(-1, 0);
					depotOpening = prevprev;
					depotBuilt = AIRail.BuildRailDepot(depotLoc, prevprev);
					if (depotBuilt) {
						if (!(AIRail.BuildRailTrack(prevprev, AIRail.RAILTRACK_NW_NE) && AIRail.BuildRailTrack(prevprev, AIRail.RAILTRACK_NE_SE))) {
							depotBuilt = false;
							AITile.DemolishTile(depotLoc);
						}
					}
				}
			}
			prev3 = prevprev;
			prevprev = prev;
			prev = path.GetTile();
			path = path.GetParent();
		}

		if (depotBuilt) {
			Debug("path done, depot built");
			foreach (track in goals) {
				if (AITile.IsBuildable(track[0]) || AIRoad.IsRoadTile(track[0]) || AIRail.IsRailTile(track[0]))	// don't bother if the exit is blocked
					connect([[depotOpening, depotLoc]], [track], [], true, true);
				else
					Warning("Station has a track side that can't be connected");
			}

			//connect([[depotOpening, depotLoc]], starts, [], true);
			return depotLoc;
		} 
		if (!skipDepot && !depotBuilt){
			Warning("Could not place depot, but path done");
			return -1;
		}

		return 0;
	}

	function connectIndustryStations(station1, station2, cargo, cargoAmount) {
		local depot = myStationsTileList.GetValue(station1);
		if (!AIRail.IsRailDepotTile(depot))
			depot = myStationsTileList.GetValue(station2);
		if (!AIRail.IsRailDepotTile(depot)) {
			Error("This route has no depots!!");
			return false;
		}
		
		local cartCount = 0;
		local len1 =  myRailStationLengthList.GetValue(station1);
		local len2 =  myRailStationLengthList.GetValue(station2);
		local maxLen16 = (len1 < len2 ? len1 : len2) * 16;
		Debug("Station lengths: " + len1 + ", " + len2 + "; max 16th units: " + maxLen16);

		arrangeBalance(AIEngine.GetPrice(cargoTrainEngine));
		local train1 = AIVehicle.BuildVehicle(depot, cargoTrainEngine);
		if (!AIVehicle.IsValidVehicle(train1)) {
			Error("Failure to build train: " + AIError.GetLastErrorString());
			return false;
		}

		if (AIEngine.CanRefitCargo(cargoTrainEngine, cargo)) {
			if (AIVehicle.RefitVehicle(train1, cargo))	// if it fails: no problem, we have the carts for the cargo
				cartCount++;
		}

		local cart;
		local cartType = getCart(cargo, current_cargo_rail_type);
		local exitLoop = false;
		Debug("Needed cargo capacity: " + cargoAmount);
		if (cargoAmount < 1) {
			Warning("Fixed cargo demand to 1");
			cargoAmount = 1;
		}
		while (AIVehicle.GetCapacity(train1, cargo) < cargoAmount && AIVehicle.GetLength(train1) <= maxLen16 && !exitLoop) {
			arrangeBalance(AIEngine.GetPrice(cartType));
			cart = AIVehicle.BuildVehicle(depot, cartType);
			if (!AIVehicle.IsValidVehicle(cart)) {
				Error(AIError.GetLastErrorString());
				exitLoop = true;
			} else {
				// we need at least some transport container but preferably the train should not be too long
				if (cartCount > 0 && (AIVehicle.GetLength(train1) + AIVehicle.GetLength(cart)) > maxLen16) {
					AIVehicle.SellVehicle(cart);
					exitLoop = true;
				} else {
					if(!AIVehicle.RefitVehicle(cart, cargo)) {
						Error("Error refitting cart: " + AIError.GetLastErrorString());
						return false;
					}
					AIVehicle.MoveWagon(cart, 0, train1, 0);
					Debug("Built cart; new cargo capacity: " + AIVehicle.GetCapacity(train1, cargo));
				}
			}
		}

		if (AIVehicle.GetCapacity(train1, cargo) < cargoAmount)
			Warning("Couldn't built full desired train length! " + AIVehicle.GetCapacity(train1, cargo) + ", " + cargoAmount);

		Debug("Built train: " + AIVehicle.GetName(train1) + ", 16-length: " + AIVehicle.GetLength(train1));
		local res = AIOrder.AppendOrder(train1, station1, AIOrder.OF_FULL_LOAD_ANY | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(train1, station1, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		res = res && AIOrder.AppendOrder(train1, station2, AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(train1, station2, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		if (res)
			AIVehicle.StartStopVehicle(train1);
		else
			Error("Couldn't give train proper orders: " + AIError.GetLastErrorString());

		return true;
	}

	function connectTownStations(station1, station2, passengers, mails) {
		local len1 =  myRailStationLengthList.GetValue(station1);
		local len2 =  myRailStationLengthList.GetValue(station2);
		local distance = AIMap.DistanceManhattan(station1, station2);
		local maxLen16 = (len1 < len2 ? len1 : len2) * 16;
		Debug("Station lengths: " + len1 + ", " + len2 + "; max 16th units: " + maxLen16);
		Debug("Mail, passenger capacity wanted: " + mails + ", " + passengers);

		local depot = myStationsTileList.GetValue(station1);
		arrangeBalance(AIEngine.GetPrice(passengerTrainEngine));
		local train1 = AIVehicle.BuildVehicle(depot, passengerTrainEngine);
		if (!AIVehicle.IsValidVehicle(train1)) {
			Error("Error building train engine: " + AIError.GetLastErrorString());
			return false;
		}
		if(AIEngine.CanRefitCargo(passengerTrainEngine, ::MAIL_ID)) {
			AIVehicle.RefitVehicle(train1, ::MAIL_ID);	// no problem if it fails, it's just extra
			Debug("Mail capacity engine: " + AIVehicle.GetCapacity(train1, ::MAIL_ID));
		} else if(AIEngine.CanRefitCargo(passengerTrainEngine, ::PASSENGER_ID)) {
			AIVehicle.RefitVehicle(train1, ::PASSENGER_ID);	// no problem if it fails, it's just extra
			Debug("Passenger capacity engine: " + AIVehicle.GetCapacity(train1, ::PASSENGER_ID));
		}

		local cart;
		local skip = 1;
		local exitLoop = false;
		while (AIVehicle.GetCapacity(train1, ::PASSENGER_ID) < passengers && !exitLoop && AIVehicle.GetLength(train1) <= maxLen16) {
			arrangeBalance(AIEngine.GetPrice(passengerCart));
			cart = AIVehicle.BuildVehicle(depot, passengerCart);
			if (!AIVehicle.IsValidVehicle(cart)) {
				Error(AIError.GetLastErrorString());
				exitLoop = true;
			} else {
				if (AIVehicle.GetLength(train1) + AIVehicle.GetLength(cart) > maxLen16) {
					AIVehicle.SellVehicle(cart);
					exitLoop = true;
				} else {
					if (!AIVehicle.RefitVehicle(cart, ::PASSENGER_ID)) {
						exitLoop = true;
						Error("Can't refit passenger wagon: " + AIError.GetLastErrorString());
					}
					if (!AIVehicle.MoveWagon(cart, 0, train1, 0)) {
						Error("Can't move wagon: " + AIError.GetLastErrorString());
						Warning("Trying different cart...");
						passengerCart = getCart(::PASSENGER_ID, AIRail.GetCurrentRailType(), distance, false, skip++);
						if (passengerCart == -1)
							exitLoop = true;
						AIVehicle.SellVehicle(cart);
						if (AIVehicle.GetCapacity(train1, ::PASSENGER_ID) > 0 && passengers > AIVehicle.GetCapacity(train1, ::PASSENGER_ID)) {
							Debug("Building extra train instead in opposite direction...");
							connectTownStations(station2, station1, (AIVehicle.GetCapacity(train1, ::PASSENGER_ID) - 1), 0);
						}
					}
					Debug("Passenger capacity: " + AIVehicle.GetCapacity(train1, ::PASSENGER_ID));
				}
			}
		}

		skip = 1;
		exitLoop = false;
		while (AIVehicle.GetCapacity(train1, ::MAIL_ID) < mails && !exitLoop && AIVehicle.GetLength(train1) <= maxLen16) {
			arrangeBalance(AIEngine.GetPrice(mailCart));
			cart = AIVehicle.BuildVehicle(depot, mailCart);
			if (!AIVehicle.IsValidVehicle(cart)) {
				Error(AIError.GetLastErrorString());
				exitLoop = true;
			} else {
				if ((AIVehicle.GetLength(train1) + AIVehicle.GetLength(cart)) > maxLen16) {
					AIVehicle.SellVehicle(cart);
					exitLoop = true;
				} else {
					if (!AIVehicle.RefitVehicle(cart, ::MAIL_ID)) {
						Error("Can't refit mail wagon: " + AIError.GetLastErrorString());
						exitLoop = true;
					}
					if (!AIVehicle.MoveWagon(cart, 0, train1, 0)) {
						Error("Can't move mail wagon: " + AIError.GetLastErrorString());
						Warning("Trying different cart...");
						mailCart = getCart(::MAIL_ID, AIRail.GetCurrentRailType(), distance, false, skip++);
						if (mailCart == -1)
							exitLoop = true;
						AIVehicle.SellVehicle(cart);
					}
					Debug("Mail capacity: " + AIVehicle.GetCapacity(train1, ::MAIL_ID));
				}
			}
		}

		if (mails - AIVehicle.GetCapacity(train1, ::MAIL_ID) > 0 || passengers - AIVehicle.GetCapacity(train1, ::PASSENGER_ID) > 0)
			Warning("Couldn't built full desired train length! " + mails + ", " + passengers);

		Debug("Built train: " + train1 + ", 16-length: " + AIVehicle.GetLength(train1));
		//lastPassengerVehicle = train1;
		local res = AIOrder.AppendOrder(train1, station1, AIOrder.OF_FULL_LOAD_ANY | AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(train1, station1, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		//res = res && AIOrder.AppendOrder(train1, station2, AIOrder.OF_FULL_LOAD_ANY | AIOrder.OF_NON_STOP_INTERMEDIATE);
		res = res && AIOrder.AppendOrder(train1, station2, AIOrder.OF_NON_STOP_INTERMEDIATE);
		if (::breakdownLevel != 0)
			AIOrder.AppendOrder(train1, station2, AIOrder.OF_GOTO_NEAREST_DEPOT | AIOrder.OF_SERVICE_IF_NEEDED);
		if (res)
			AIVehicle.StartStopVehicle(train1);
		else
			Error("Couldn't give train proper orders: " + AIError.GetLastErrorString());

		/* local train2 = AIVehicle.BuildVehicle(myStationsTileList.GetValue(station2), passengerTrainEngine);
		if (!AIVehicle.IsValidVehicle(train2))
			return false; 	// TODO change to return 1 for 1 success
		if(!AIVehicle.RefitVehicle(train2, ::PASSENGER_ID)) {
			Debug("Error refitting train: " + AIError.GetLastErrorString());
			return false; // TODO
		}
		Debug("Built train: " + train2);
		AIOrder.AppendOrder(train2, station1, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		AIOrder.AppendOrder(train2, station2, AIOrder.OF_FULL_LOAD | AIOrder.OF_NON_STOP_INTERMEDIATE);
		AIVehicle.StartStopVehicle(train2); */

		return true;	// TODO change to 2 
	}
	
	function buildIndustryStation(target, length, width, cargoID, source_type, goal_type, distance, isSourceStation) {
		// replace parameter with industryID to determine station size and acceptance
		Debug("Building " + AICargo.GetCargoLabel(cargoID) + (isSourceStation ? " source" : " destination") + " station...");
		// Find empty block of squares as close to target as possible
		local terraformed = 0;	//prevent AI from trying to level a town
		if (length > 7)
			length = 7;
		//local width = (length == 7 ? 2 : 1);
		// you need changing tracks for multiple platforms
		//local totalLength = length + width;
		Debug("building station (lxb) " + length + " x " + width);

		arrangeBalance(2 * length * width * AIRail.GetBuildCost(AIRail.GetCurrentRailType(), AIRail.BT_STATION));

		local area;
		if (isSourceStation) {
			// getlocation is top, not centre; trying to build closer to centre
			local targetLoc = AIIndustry.GetLocation(target) + AIMap.GetTileIndex(0, 2);
			area = AITileList_IndustryProducing(target, stationRadius);
			area.Valuate(AITile.IsBuildable);
			area.KeepValue(1);
			area.Valuate(stationLocationValuator, length, width);
			area.RemoveValue(0);

			area.Valuate(AITile.GetDistanceManhattanToTile, targetLoc);
			area.Sort(AIList.SORT_BY_VALUE, AIList.SORT_ASCENDING);
			//area.Valuate(AITile.GetCargoProduction, cargoID, 1, 1, stationRadius);
		} else {
			area = AITileList_IndustryAccepting(target, stationRadius);
			area.Valuate(AITile.IsBuildable);
			area.KeepValue(1);
			area.Valuate(stationLocationValuator, length, width);
			area.RemoveValue(0);
			area.Valuate(AITile.GetCargoAcceptance, cargoID, 1, 1, stationRadius);
			area.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
		}

		if (area.Count()) {
			for (local t = area.Begin(); !area.IsEnd(); t = area.Next()) {
				local orient = stationLocationValuator(t, length, width);
				if (orient != 0) {
					if (AIRail.BuildNewGRFRailStation(t, orient, width, length, AIStation.STATION_JOIN_ADJACENT, cargoID, source_type, goal_type, distance, isSourceStation)) {
						return t;
					} else if (terraformed++ < 4){
						// try again with terraforming
						if (orient == AIRail.RAILTRACK_NE_SW) {
							if (!AITile.LevelTiles(t, t + AIMap.GetTileIndex(length + 1, width)))
								Error(AIError.GetLastErrorString());
						} else if (orient == AIRail.RAILTRACK_NW_SE) {
							if (!AITile.LevelTiles(t, t + AIMap.GetTileIndex(width, length + 1)))
								Error(AIError.GetLastErrorString());
						}
						if (AIRail.BuildNewGRFRailStation(t, orient, width, length, AIStation.STATION_JOIN_ADJACENT, cargoID, source_type, goal_type, distance, isSourceStation)) {
							return t;
						}
					}
				}
			}
		}
		Debug("No possible station location found");
		return null;
	}
	
	function connect2Towns(biggestTown, connectTown, cargo = 0) {
		local vehList = AIVehicleList();
		vehList.Valuate(AIVehicle.GetVehicleType);
		vehList.KeepValue(AIVehicle.VT_RAIL);
		if (vehList.Count() >= maxTrains) {
			Warning("Reached maximum amount of train!");
			return false;
		}
		if (cargo == 0)
			cargo = ::PASSENGER_ID;

		local distance = AIMap.DistanceManhattan(AITown.GetLocation(biggestTown), AITown.GetLocation(connectTown));
		if (getEngine(current_town_rail_type, cargo, distance) == -1) {
			Warning("No train available for this connection");
			return false;
		}

		AIRail.SetCurrentRailType(current_town_rail_type);
		local build1 = false;
		local build2 = false;

		local length1 = abs(sqrt(AITown.GetPopulation(biggestTown)/100));
		if (length1 < 2)
			length1 = 2;
		local station1 = getTrainStation(biggestTown);
		if (station1) {
			if (getStationLength(biggestTown) < length1) {
				Debug("Building a bigger and better station in " + AITown.GetName(biggestTown));
				station1 = BuildTrainStation(biggestTown);
				if (!station1) {
					Warning("!!! Building bigger station failed in: " + AITown.GetName(biggestTown));
					Debug("Using old existing station instead...");
					station1 = getTrainStation(biggestTown);
					length1 = getStationLength(station1);
				} else {
					build1 = true;
				}
			} else {
				Debug("Reusing existing station in " + AITown.GetName(biggestTown));
			}
		} else {
			station1 = BuildTrainStation(biggestTown);
			if (!station1) {
				Warning("!!! Building station failed in: " + AITown.GetName(biggestTown));
				return false;
			}
			build1 = true;
		}
		
		local length2 = abs(sqrt(AITown.GetPopulation(connectTown)/100));
		if (length2 < 2)
			length2 = 2;
		local station2 = getTrainStation(connectTown);
		if (station2) {                                                                                                             
			if (getStationLength(connectTown) < length2) {
				Debug("Building a bigger and better station in " + AITown.GetName(connectTown));
				station2 = BuildTrainStation(connectTown);
				if (!station2) {
					Warning("!!! Building bigger station failed in: " + AITown.GetName(connectTown));
					Debug("Using old existing station instead...");
					station2 = getTrainStation(connectTown);
					length2 = getStationLength(station2);
				} else {
					build2 = true;
				}
			} else {
				Debug("Reusing existing station in " + AITown.GetName(connectTown));
			}
		} else {
			station2 = BuildTrainStation(connectTown);
			build2 = true;
		}

		if (!station2) {
			Warning("!!! Building station failed in: " + AITown.GetName(connectTown));
			Warning("Removing other station...");
			// TODO: hand to town and remove signal and rail as well
			if (build1) {
				AITile.DemolishTile(station1);
			}
			return false;
		}

		if (connectStations(station1, length1, 2, station2, length2, 2)) {
			//we picked BIG towns, lets try to build more carts in 1 go
			local pop1 = AITown.GetLastMonthProduction(biggestTown, ::PASSENGER_ID) - AITown.GetLastMonthSupplied(biggestTown, ::PASSENGER_ID);
			local pop2 = AITown.GetLastMonthProduction(connectTown, ::PASSENGER_ID) - AITown.GetLastMonthSupplied(connectTown, ::PASSENGER_ID);
			local mail1 = AITown.GetLastMonthProduction(biggestTown, ::MAIL_ID) - AITown.GetLastMonthSupplied(biggestTown, ::MAIL_ID);
			local mail2 = AITown.GetLastMonthProduction(connectTown, ::MAIL_ID) - AITown.GetLastMonthSupplied(connectTown, ::MAIL_ID);

			// TODO: see whether this value makes sense: assume we can get half of the untransported town max
			local maxPassengers = abs((pop1 < pop2 ? pop1 : pop2)/2);
			local maxMail = abs((mail1 < mail2 ? mail1 : mail2)/2);

			if (build1)
				addTrainStation(biggestTown, station1, length1);
			if (build2)
				addTrainStation(connectTown, station2, length2);

			if (!connectTownStations(station1, station2, maxPassengers, maxMail))
				return false; // TODO cleanup

			return true;
		} else {
			Warning("Failure... cleaning up....");
			// remove stuff we have build
			if (build1) {
				Warning("Cleaning station 1");
				local trains = AIVehicleList_Station(AIStation.GetStationID(station1));
				if (trains.IsEmpty())
					AITile.DemolishTile(station1);
			}
			if (build2) {
				Warning("Cleaning station 2");
				local trains = AIVehicleList_Station(AIStation.GetStationID(station2));
				if (trains.IsEmpty())
					AITile.DemolishTile(station2);
			}
			return false;
		}
	}

	// TODO support for bigger then 7x2
	function connectStations(station1, length1, width1, station2, length2, width2) {
		local openingStart;
		local openingGoal;
		local openingStart2;
		local openingGoal2;

		if (length1 > 7)
			length1 = 7;
		if (length2 > 7)
			length2 = 7;

		Debug("Route between 2 stations is " + AITile.GetDistanceManhattanToTile(station1, station2));

		if (AIRail.GetRailStationDirection(station1) == AIRail.RAILTRACK_NE_SW) {
			openingStart = station1 + AIMap.GetTileIndex(length1, 0); // TODO FIXME
			openingStart2 = station1 + AIMap.GetTileIndex(length1 - 1, 0); // TODO FIXME
		} else {
			openingStart = station1 + AIMap.GetTileIndex(0, length1); // TODO FIXME
			openingStart2 = station1 + AIMap.GetTileIndex(0, length1 - 1); // TODO FIXME
		}

		if (AIRail.GetRailStationDirection(station2) == AIRail.RAILTRACK_NE_SW) {
			openingGoal = station2 + AIMap.GetTileIndex(length2, 0); // TODO FIXME
			openingGoal2 = station2 + AIMap.GetTileIndex(length2 - 1, 0); // TODO FIXME generalyse
		} else {
			openingGoal = station2 + AIMap.GetTileIndex(0, length2); // TODO FIXME
			openingGoal2 = station2 + AIMap.GetTileIndex(0, length2 - 1); // TODO FIXME generalyse
		}
		/*AISign.BuildSign(openingStart, "f");
		AISign.BuildSign(openingStart2, "F");
		AISign.BuildSign(openingGoal, "t");
		AISign.BuildSign(openingGoal2, "T");*/
		local starts = [[openingStart, openingStart2]];
		local goals = [[openingGoal, openingGoal2]];
		// other end
		if (AIRail.GetRailStationDirection(station1) == AIRail.RAILTRACK_NE_SW) {
			openingStart2 = station1; // TODO FIXME
			openingStart = station1 + AIMap.GetTileIndex(-1, 0); // TODO FIXME
		} else {
			openingStart2 = station1; // TODO FIXME
			openingStart = station1 + AIMap.GetTileIndex(0, -1); // TODO FIXME
		}

		if (AIRail.GetRailStationDirection(station2) == AIRail.RAILTRACK_NE_SW) {
			openingGoal2 = station2; // TODO FIXME
			openingGoal = station2 + AIMap.GetTileIndex(-1, 0); // TODO FIXME generalyse
		} else {
			openingGoal2 = station2; // TODO FIXME
			openingGoal = station2 + AIMap.GetTileIndex(0, -1); // TODO FIXME generalyse
		}
		/*AISign.BuildSign(openingStart, "f");
		AISign.BuildSign(openingStart2, "F");
		AISign.BuildSign(openingGoal, "t");
		AISign.BuildSign(openingGoal2, "T");*/
		starts.append([openingStart, openingStart2]);
		goals.append([openingGoal, openingGoal2]);

		if (width1 == 2) {
			if (AIRail.GetRailStationDirection(station1) == AIRail.RAILTRACK_NE_SW) {
				openingStart = station1 + AIMap.GetTileIndex(length1, 1); // TODO FIXME
				openingStart2 = station1 + AIMap.GetTileIndex(length1 - 1, 1); // TODO FIXME
			} else {
				openingStart = station1 + AIMap.GetTileIndex(1, length1); // TODO FIXME
				openingStart2 = station1 + AIMap.GetTileIndex(1, length1 - 1); // TODO FIXME
			}
			/*AISign.BuildSign(openingStart, "f");
			AISign.BuildSign(openingStart2, "F");*/
			starts.append([openingStart, openingStart2]);
			if (AIRail.GetRailStationDirection(station1) == AIRail.RAILTRACK_NE_SW) {
				openingStart2 = station1 + AIMap.GetTileIndex(0, 1); // TODO FIXME
				openingStart = station1 + AIMap.GetTileIndex(-1, 1); // TODO FIXME
			} else {
				openingStart2 = station1 + AIMap.GetTileIndex(1, 0); // TODO FIXME
				openingStart = station1 + AIMap.GetTileIndex(1, -1); // TODO FIXME
			}
			/*AISign.BuildSign(openingStart, "f");
			AISign.BuildSign(openingStart2, "F");*/
			starts.append([openingStart, openingStart2]);
		}
		if (width2 == 2) {
			if (AIRail.GetRailStationDirection(station2) == AIRail.RAILTRACK_NE_SW) {
				openingGoal = station2 + AIMap.GetTileIndex(length2, 1); // TODO FIXME
				openingGoal2 = station2 + AIMap.GetTileIndex(length2 - 1, 1); // TODO FIXME generalyse
			} else {
				openingGoal = station2 + AIMap.GetTileIndex(1, length2); // TODO FIXME
				openingGoal2 = station2 + AIMap.GetTileIndex(1, length2 - 1); // TODO FIXME generalyse
			}
			/*AISign.BuildSign(openingGoal, "t");
			AISign.BuildSign(openingGoal2, "T");*/
			goals.append([openingGoal, openingGoal2]);
			if (AIRail.GetRailStationDirection(station2) == AIRail.RAILTRACK_NE_SW) {
				openingGoal2 = station2 + AIMap.GetTileIndex(0, 1); // TODO FIXME
				openingGoal = station2 + AIMap.GetTileIndex(-1, 1); // TODO FIXME generalyse
			} else {
				openingGoal2 = station2 + AIMap.GetTileIndex(1, 0); // TODO FIXME
				openingGoal = station2 + AIMap.GetTileIndex(1, -1); // TODO FIXME generalyse
			}
			/*AISign.BuildSign(openingGoal, "t");
			AISign.BuildSign(openingGoal2, "T");*/
			goals.append([openingGoal, openingGoal2]);
		}

		//local stations = [station1, station2];
		local skipDepot = false;
		if (AIRail.IsRailDepotTile(myStationsTileList.GetValue(station2))) {
			Debug("Reusing existing depot");
			skipDepot = true;
		}

		local depot = connect(starts, goals, [], skipDepot);
		//if(!AIRail.IsRailDepotTile(depot)) {
			//depot = connect(starts, goals, []);	// Try once more if it failed
		//}


		if (skipDepot) {
			if (depot == -1)
				return false;
			else
				depot = myStationsTileList.GetValue(station2);
		}

		if(!AIRail.IsRailDepotTile(depot)) {
			// check if route complete; build other orientation of depot
			Error("While connecting: incomplete route or unable to find suitable depot location");
			return false;
		} else {
			myStationsTileList.AddItem(station2, depot);

			// reverse track:
			if (AIRail.IsRailDepotTile(myStationsTileList.GetValue(station1)))
				skipDepot = true;
			else 
				skipDepot = false;
			depot = connect(goals, starts, [], skipDepot);
			if(!skipDepot && AIRail.IsRailDepotTile(depot))
				myStationsTileList.AddItem(station1, depot);

			// build signals
			/*foreach (track in goals) {
				AIRail.BuildSignal(track[0], track[1], AIRail.SIGNALTYPE_PBS);
			}
			foreach (track in starts) {
				AIRail.BuildSignal(track[0], track[1], AIRail.SIGNALTYPE_PBS);
			}*/

			return true;
		}
	}

	function getTrainStation(town) {
		local stationTile = myRailStationTownList.GetValue(town);
		local statID = AIStation.GetStationID(stationTile);
		local engineRailType = AIEngine.GetRailType(passengerTrainEngine);
		if (AIStation.IsValidStation(statID) && AIStation.HasStationType(statID, AIStation.STATION_TRAIN) && AIRail.TrainCanRunOnRail(engineRailType, current_town_rail_type) && AIRail.TrainHasPowerOnRail(engineRailType, current_town_rail_type)) {
			//Debug("found suitable station in " + AITown.GetName(town));
			return stationTile;
		}
		
		return null;
	}

	function getStationLength(town) {
		local station = getTrainStation(town);
		if (station)
			return myRailStationLengthList.GetValue(station);

		return 0;
	}

	function addTrainStation(town, station, length) {
		myRailStationTownList.SetValue(town, station);
		if (length > 7)
			length = 77;	// means 7x2 station, largest supported now
		myRailStationLengthList.AddItem(station, length);
	}

	function BuildTrainStation(town, length = -1) {
		Debug("Building train station...");
		// Find empty block of squares as close to target as possible
		local target = AITown.GetLocation(town);
		local curRange = 1;
		if (length == -1)	// -1 == auto
			length = abs(sqrt(AITown.GetPopulation(town)/100));
		local maxRange = 6 + length; //TODO check value correctness (coverage + size of something?)
		//local width = abs(length / 7) + 1;
		// length = abs(length % 7) + 1;
		// TODO support for bigger
		if (length > 7)
			length = 7;
		if (length < 2)
			length = 2;
		//local width = (length == 7 ? 2 : 1);
		// TODO
		//length = 7;
		local width = 2;

		local terraformed = 0;	//prevent AI from trying to level a town

		// you need changing tracks for multiple platforms
		//local totalLength = length + width;
		local area = AITileList();
		Debug("building station (lxb) " + length + " x " + width);

		arrangeBalance(2 * length * width * (AIRail.GetBuildCost(AIRail.GetCurrentRailType(), AIRail.BT_STATION) + AIRail.GetBuildCost(AIRail.GetCurrentRailType(), AIRail.BT_TRACK)));

		while (curRange <= maxRange) {
			//Debug("looking with range " + curRange);
			SafeAddRectangle(area, target, curRange, curRange - 1);
			area.Valuate(AITile.IsWithinTownInfluence, town);
			area.RemoveValue(0);
			area.Valuate(AITile.GetCargoAcceptance, ::PASSENGER_ID, 1, 1, stationRadius);
			area.RemoveBelowValue(8);
			area.Valuate(AITile.GetCargoProduction, ::PASSENGER_ID, 1, 1, stationRadius);
			area.RemoveValue(0);
			area.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
			area.Valuate(stationLocationValuator, length, width);
			area.RemoveValue(0);
			if (area.Count()) {
				Debug("Found " + area.Count() + " possible options");
				for (local t = area.Begin(); !area.IsEnd(); t = area.Next()) {
					if (AIRail.BuildRailStation(t, area.GetValue(t), width, length, AIStation.STATION_JOIN_ADJACENT)) {
						return t;
					} else if (terraformed++ < 4){
						// try again with terraforming
						if (area.GetValue(t) == AIRail.RAILTRACK_NE_SW) {
							if (!AITile.LevelTiles(t, t + AIMap.GetTileIndex(length + 1, width)))
								if (AIError.GetLastError() !=  AITile.ERR_AREA_ALREADY_FLAT)
									Error(AIError.GetLastErrorString());
						} else if (area.GetValue(t) == AIRail.RAILTRACK_NW_SE) {
							if (!AITile.LevelTiles(t, t + AIMap.GetTileIndex(width, length + 1)))
								if (AIError.GetLastError() !=  AITile.ERR_AREA_ALREADY_FLAT)
									Error(AIError.GetLastErrorString());
						}
						if (AIRail.BuildRailStation(t, area.GetValue(t), width, length, AIStation.STATION_JOIN_ADJACENT)) {
							return t;
						}
					}
				}
				curRange++;
			} else {
				curRange++;
			}
		}
		Debug("No possible station location found");
		return null;
	}

	/**
	 * upgrades small station to bigger one
	 * also reroutes trains to new station if needed
	 */
	function UpgradeTrainStation(town) {}
}

