/**
 *    This file is part of OtviAI.
 *
 *    OtviAI is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    OtviAI is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with OtviAI.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2008-2013 Otto Visser.
 * Suggestions/comments and bugs can go to: otviAI@otvi.nl
 */

class OtviTransport {
	YEAR_LENGTH = 365.25;	// in days
	KM_PER_TILE = 644.216;

	/**
	 * obvious function is obvious
	 */
	function getName();

	/**
	 * Returns whether anything can be done with this transport type
	 * Returns boolean
	 */
	function isOperational();

	/**
	 * Returns minimum amount of money needed to establish a new route
	 * default distance used for calculation is 100
	 * Returns max int if expansion not possible
	 */
	function getExpansionCost(cargo, distance = 100);

	/**
	 * Returns minimum amount of money needed to enhance the last build route
	 * i.e.: add more vehicles to the route
	 * returns max int if enhancing is not possible
	 */
	function getEnhanceCost();

	/**
	 * returns whether there is a route that could use enhancement at the moment
	 */
	function canEnhance();

	/**
	 * does the actual enhancing
	 */
	function enhance();

	/**
	 * replaces a vehicle with a newer one
	 */
	//TODO function renewVehicle(oldVehicle);

	/**
	 * Returns transport preference
	 * Lower is more preferable
	 */
	function getTownTransportCost(town1, town2);
	function getIndustryTransportCost(producer, endLocation, cargoID);

	/**
	 * Updates engines
	 */
	function updateEngines();

	/**
	 * Connects two given towns and starts transporting passengers (and mail if possible)
	 * Returns: true iff succesful, false otherwise
	 */
	function connect2Towns(town1, town2, cargo = 0);

	/**
	 * Connects industry to given town or industry and starts transporting given cargo
	 */
	function connectIndustry(supplier, destination, cargoID, destIsTown);

	/**
	 * Connects industry to given town or industry and starts transporting given cargo
	 */
	function connectTrade(tradeCenter, otherDest, cargoID, TCIsSender, destIsTown);

	/**
	 * sorts based on expansion cost
	 */
	function expansionSort(a, b) {
		local costa = a.getExpansionCost(::PASSENGER_ID);
		local costb = b.getExpansionCost(::PASSENGER_ID);

		if (costa > costb)
			return 1;
		else if (costa < costb)
			return -1;
		return 0;
	}

	/**
	 * sorts based on expansion cost
	 */
	function enhanceSort(a, b) {
		local costa = a.getEnhanceCost();
		local costb = b.getEnhanceCost();

		if (costa > costb)
			return 1;
		else if (costa < costb)
			return -1;
		return 0;
	}

	function townTransportSort(a, b) {
		local costa = a.getTownTransportCost(::startLoc, ::endLoc);
		local costb = b.getTownTransportCost(::startLoc, ::endLoc);

		if (costa > costb)
			return 1;
		else if (costa < costb)
			return -1;
		return 0;
	}

	function cargoTransportSort(a, b) {
		local costa = a.getIndustryTransportCost(::startLoc, ::endLoc, ::cargo_ID);
		local costb = b.getIndustryTransportCost(::startLoc, ::endLoc, ::cargo_ID);

		if (costa > costb)
			return 1;
		else if (costa < costb)
			return -1;
		return 0;
	}

	function getEngineCosts(engineID) {
		// if there are no breakdowns: return yearly running costs and assume we'll use the vehicle 4 times longer than designed for
		//Debug("  Costs: " + AIEngine.GetRunningCost(engineID) + "; " + AIEngine.GetPrice(engineID));
		if (::breakdownLevel == 0)
			return AIEngine.GetRunningCost(engineID) + AIEngine.GetPrice(engineID)/((AIEngine.GetMaxAge(engineID) * 4)/OtviTransport.YEAR_LENGTH);
		else
			return AIEngine.GetRunningCost(engineID) + AIEngine.GetPrice(engineID)/(AIEngine.GetMaxAge(engineID)/OtviTransport.YEAR_LENGTH);
	}

	function isHinUndWiederCargo(cargoID) {
		if (AICargo.HasCargoClass(cargoID, AICargo.CC_PASSENGERS) && AICargo.GetTownEffect(cargoID) == AICargo.TE_PASSENGERS)
			return true;	// humans on transport
		if (AICargo.HasCargoClass(cargoID, AICargo.CC_MAIL) && AICargo.GetTownEffect(cargoID) == AICargo.TE_MAIL)
			return true;	// you've got mail
		local accepting = AIIndustryList_CargoAccepting(cargoID);
		local producing = AIIndustryList_CargoProducing(cargoID);
		local count = producing.Count();
		producing.KeepList(accepting);
		if (producing.Count() == count) {
			// all producers are also accepters; it must therefore be a hin und wieder cargo!
			// Warning("Found unknown hin und wieder cargo: " + AICargo.GetCargoLabel(cargoID));
			return true;
		}

		return false;	//assuming all other cargo is one way
	}

	function getTurnover(engineID, distance, cargoID, production, isSubsidy) {
		local capacity;
		if (AIEngine.GetVehicleType(engineID) == AIVehicle.VT_RAIL) {
			local carrier;
			carrier = OtviTrAIn.getCart(cargoID, AIEngine.GetRailType(engineID), distance, true);
			if (carrier == -1)
				carrier = engineID;
			capacity = AIEngine.GetCapacity(carrier) * 3;	// Assuming an average of 3 carts for trains
		} else {
			capacity = AIEngine.GetCapacity(engineID);
		}

		local speedModifier = 0.9;	// assume we can run on average at 90% of our max TODO: verify
		if (::breakdownLevel != 0)
			speedModifier = (speedModifier * AIEngine.GetReliability(engineID)) / 100.0;

		local loadingTime = (20.0 + capacity)/74.0;
		if (production != 0)
			loadingTime = (loadingTime * capacity)/production;
		if (OtviTransport.isHinUndWiederCargo(cargoID))
			loadingTime *= 2;

		local daysPerTrip = ((distance * 2 * OtviTransport.KM_PER_TILE)/AIEngine.GetMaxSpeed(engineID)/24/speedModifier + loadingTime).tointeger() + 1;

		local turnoverPerTrip = capacity * AICargo.GetCargoIncome(cargoID, distance, daysPerTrip);
		if (OtviTransport.isHinUndWiederCargo(cargoID))
			turnoverPerTrip *= 2;

		local tripCount = OtviTransport.YEAR_LENGTH/daysPerTrip;
		local turnover = tripCount * turnoverPerTrip;
		if (isSubsidy) {
			local subsidyMultiplier = AIGameSettings.GetValue("difficulty.subsidy_multiplier");
			local total = turnover * subsidyMultiplier;
			total += (AIEngine.GetMaxAge(engineID)/OtviTransport.YEAR_LENGTH - 1) * turnover;
			turnover = total / (AIEngine.GetMaxAge(engineID) / OtviTransport.YEAR_LENGTH);
		}
		//Debug("  Turnover per trip  = " + turnoverPerTrip + "; days/trip = " + daysPerTrip);
		return turnover;
	}

	/**
	 * Returns the estimated profit for this engine for a default distance of 100 tiles per year
	 */
	function profitValuator(engineID, cargoID, distance = 100, production = 0, isSubsidy = false) {
		//Debug("----");
		local costs = OtviTransport.getEngineCosts(engineID);
		local turnover = OtviTransport.getTurnover(engineID, distance, cargoID, production, isSubsidy);

		local profit = (turnover - costs).tointeger();
		//Debug("PROFIT for " + AIEngine.GetName(engineID) + " transporting " + AICargo.GetCargoLabel(cargoID) + " = " + turnover + " - " + costs + " = " + profit);
		return profit;
	}
}

