/*
 *
 * Copyright (c) 2009, Dustin Andrews
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 
 * Redistributions in binary form must reproduce the above copyright notice, this
 * list of conditions and the following disclaimer in the documentation and/or 
 *  other materials provided with the distribution.
 * 
 * The names of its contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

require("AntPathFinder.nut");
require("CrowPathFinder.nut");
require("BridgePathFinder.nut");

class AntHill
{
  paths = null;
  ant = null;
  crow = null;
  start = null;
  goal = null;
  cost = null;
  last = null;
  ignoreTiles = null;
  
  prepend = null;
  postpend = null;
  
  display_thinking = null;
  
  routesBuilt = 0;
  
  RouteCache = [];
  
  constructor(start, goal, prepend, postpend, ignoreTiles)
  {
    assert(AIMap.IsValidTile(start));
    assert(AIMap.IsValidTile(goal));
    
    this.start = start;
    this.goal = goal;
    this.paths = [];
    this.ant = AntPathFinder(start, goal, ignoreTiles);
    this.crow = CrowPathFinder(start, goal, ignoreTiles);
    this.cost = null;
    routesBuilt = 0;
    this.prepend = prepend;
    this.postpend = postpend;
    this.display_thinking = AIController.GetSetting("display_thinking");
    
    local startList = AIList();
    startList.AddList(prepend);
    startList.Sort(AIAbstractList.SORT_BY_VALUE, false);
    this.last = startList.Begin();
    crow.last = last;
    ant.last = last;
    this.ignoreTiles = ignoreTiles;
  }

}


//callback is a function to call between each pathfinder iteration
//caller is the instance of the class the callback belongs to.
//This allows the AI to keep from being totally distracted by the pathfinder.
function AntHill::FindPathMixed(callback, caller)
{

    
  //print("Hill Last == " + last);
  if(display_thinking == 1)
  {
    AISign.BuildSign(start, "{+}");
    AISign.BuildSign(goal, "{=}");
  }
  local run = true;
  local currentStart = start;
  local currentGoal = goal;
  local pathSegments = [];
  local iteration = 0;
  local iterationLimit = 25;
  
  pathSegments.push(prepend);
  
  while(run && iteration < iterationLimit)
  {
    callback(caller);
    iteration++;
    crow = CrowPathFinder(currentStart, currentGoal, ignoreTiles);
    print ("starting crow.");
    if(crow.FindPath())
    {
      run = false;
      pathSegments.push(crow.path);
      //print ("crow found goal.");
    }
    else
    {
      pathSegments.push(crow.path);
      currentStart = crow.midStart;
      currentGoal = crow.midGoal;     
      last = crow.last;
      //print("AntHill Last == " + last);
      
      
      if(
            currentGoal == null
         || currentStart == null
         || !AIMap.IsValidTile(currentGoal) 
         || !AIMap.IsValidTile(currentStart)
         
        )
      {
        pathSegments = null;
        run = false;
      }
      else
      {
  
        AILog.Info("starting bridge.");
        local bridge = BridgePathFinder(currentStart, currentGoal, ignoreTiles);
        if(bridge.FindPath())
        {
          pathSegments.push(bridge.path);
          currentStart = bridge.midStart;
          currentGoal = this.goal;
          AILog.Info("Bridge found the way over.");
        }
        else
        { 
          local prepend = AIList();
          local postend = AIList();



          local hill = AntHill(currentStart, currentGoal, prepend, postpend, ignoreTiles);
          hill.last = last;
          AILog.Info("Starting Ant");
          local path = hill.FindPath(20);
          if(path != null)
          {
            path.AddItem(currentGoal, 10000);
            pathSegments.push(path);
            hill.last = currentGoal;
            currentStart = currentGoal;
            currentGoal = this.goal;
            AILog.Info("ant found way around.");
          }
          else
          {
            //print("Ant failed to find a way around.");
            run = false;
            pathSegments = null;
          }
        }
      }
    }
  }
  
  

  if(iteration == iterationLimit) 
  {
    AILog.Warning("Iteration Limit reached.");
    pathSegments = null;
  }
  
  local fullPath = AIList();
  local fullCount = 0;
  
  
  
  if(pathSegments != null)
  {
    pathSegments.push(postpend);
    foreach(i, val in pathSegments)
    { 
      if(val == null)
      {
       continue;
      }
      val.Sort(AIAbstractList.SORT_BY_VALUE, true);
      foreach(tile, x in val)
      {
        fullPath.AddItem(tile, fullCount++);
      }
    }
  
    
    this.cost = AntHill.CostPath(fullPath);
    fullPath.Sort(AIAbstractList.SORT_BY_VALUE, true);
    AILog.Info("Path Cost: " + cost);
    return fullPath;
  }
  else
  {
    AILog.Info("No path found.");
    return null;
  }
}


function AntHill::FindPath(iterations)
{

 foreach(i, cacheTable in AntHill.RouteCache)
 {
  if(cacheTable.start == this.start && cacheTable.goal == this.goal)
  {
    if(display_thinking == 1 && cacheTable.path != null )
    {
      foreach(tile, v in cacheTable.path)
      {
        AISign.BuildSign(tile, "{C}");
      }
    }
    return cacheTable.path;
  }
 }

 for(local i = 0; i < iterations; i++)
 {
  ant.last = last;
  local path = ant.FindPath();
  paths.push({ path=path, cost = ant.cost });
  ant = AntPathFinder(this.start, this.goal, ignoreTiles);
  ant.goalFocus += (i / iterations) * 5.0 ;
 }

 local lowCost = 999999999;
 local bestPath = null;
 foreach(i, val in paths)
 {
    if(val.cost < lowCost)
    {
      bestPath = val.path;
      lowCost = val.cost;
    }
 }

 this.cost = lowCost;
  
 if(bestPath != null && display_thinking == 1)
 //if(bestPath != null)
 {
   foreach(tile, v in bestPath)
   {
    AISign.BuildSign(tile, "{A}");
   }
 }
 AntHill.RouteCache.push({start = this.start, goal = this.goal, path = bestPath});
 return bestPath;
}

function AntHill::VerifyPath(path)
{
    if(path.Count() > 3) {return 1;}
    local test = AITestMode();
    local pp = null;
    local p = null;
    local count = 0;
    foreach(tile, val in path)
    {
      if(p != null && pp != null)
      {
       if (!AIRail.BuildRail(pp, p, tile))
       {
        return false;
       }
      }
      pp = p; p = tile;
    }
    return true;
}

function AntHill::CostPath(path)
{
    
    if(path.Count() < 3) {return path.Count() * 500;}
    local test = AITestMode();
    local cost = AIAccounting();
    local pp = null;
    local p = null;
    local count = 0;
    foreach(tile, val in path)
    {
      if(p != null && pp != null)
      {
       AIRail.BuildRail(pp, p, tile);
      }
      pp = p; p = tile;
    }
    //print ("AntHill: Route Cost:" + costs.GetCosts());
    return cost.GetCosts();
}


function AntHill::BuildRoute(path)
{   
    local pp = null;
    local p = null;
    local count = 0;
    foreach(tile, val in path)
    {
    
    
      if(p != null && pp != null)
      {
         //AISign.BuildSign(p, count++ + ":" + val);
         if (AIMap.DistanceManhattan(p, tile) > 1)
         {
            local bridge_list = AIBridgeList_Length(AIMap.DistanceManhattan(tile, p) + 1);
            bridge_list.Valuate(AIBridge.GetMaxSpeed);
            bridge_list.Sort(AIAbstractList.SORT_BY_VALUE, false);
            local result = AIBridge.BuildBridge(AIVehicle.VT_RAIL, bridge_list.Begin(), p, tile);
            if(result == false)
            {
              local temp = AIList();
              temp.AddList(path);
              temp.KeepAboveValue(val);
              temp.Sort(AIAbstractList.SORT_BY_VALUE, true);
              local nextTile = temp.Begin();
              if(AIMap.DistanceManhattan(tile, nextTile) == 1) { continue;} //TODO: Track down why there are sometimes these little disjoints.
              AILog.Warning("Bridge Failed! Marking path if thinking mode on." + AIError.GetLastErrorString());
              foreach(tile, val in path)
              {
                if(display_thinking == 1)
                {
                 AISign.BuildSign(tile, count++ + ":" + val);
                }
              }
              if(display_thinking == 1)
              {
                AISign.BuildSign(p, "BRIDGEHEAD");
                AISign.BuildSign(tile, "BRIDGEEND");
              }
              AILog.Error("Bridge Failed!");
             return false;
            }
         }
         else
         {  
           local result =  AIRail.BuildRail(pp, p, tile);
           if(result == false && AIMap.DistanceManhattan(pp, p) == 1)//pp, p > 1 after a bridge, which is an OK error.
           {
            local err = AIError.GetLastErrorString()
            if(err != "ERR_ALREADY_BUILT")//Already built doesn't stop the route from being completed.
            {
              AILog.Error(err);
              return false;
            }
           }
         }
      }
      pp = p; p = tile;
    }
    return true;
}

function AntHill::FindCompassPointsNunitsAway(point, n)
{
    //print("point =" + point);
    local tiles = AITileList();
    
    local x = AIMap.GetTileX(point);
    local y = AIMap.GetTileY(point);
    
    
    tiles.AddTile(AIMap.GetTileIndex(x , y - n));
    tiles.AddTile(AIMap.GetTileIndex(x - n, y));  
    tiles.AddTile(AIMap.GetTileIndex(x + n, y));
    tiles.AddTile(AIMap.GetTileIndex(x, y + n));
    
    return tiles;
}
