/*
 *
 * Copyright (c) 2009, Dustin Andrews
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 
 * Redistributions in binary form must reproduce the above copyright notice, this
 * list of conditions and the following disclaimer in the documentation and/or 
 *  other materials provided with the distribution.
 * 
 * The names of its contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

class BridgePathFinder
{
    start = null;
    goal = null;
    current = null;
    path = null;
    count = null;
    midStart = null;
  
    display_thinking = null;
    bridge_max_len = null;
    ignoreTiles = null;
  

  constructor(start, goal, ignoreTiles)
  {
        this.start = start;
        this.current = start;
        this.midStart = goal;
        this.ignoreTiles = ignoreTiles;
        this.goal = goal;
        this.count = 0;
        this.path = AIList();
        this.display_thinking = AIController.GetSetting("display_thinking");
        this.bridge_max_len = AIController.GetSetting("bridge_max_len");
  }

}

function BridgePathFinder::ReEnter()
{
    local result = FindPath();    
    return result;
}

function BridgePathFinder::FindPath()
{
  if(AITile.IsBuildable(goal) == false) {return false;} //can't build a bridge to nowhere
  if(CheckPathNeedsBridge(start, goal) == false) {return false;}
  if(display_thinking) { AISign.BuildSign(this.current, "{B}");}
  
  if(this.IsPathDiagonal())
  {
    if(HasWaterRoadOrRailTilesToCross()) 
    {
        local corner1 = AIMap.GetTileIndex(AIMap.GetTileX(start), AIMap.GetTileY(goal));
        local corner2 = AIMap.GetTileIndex(AIMap.GetTileX(goal), AIMap.GetTileY(start));
        if(display_thinking) { AISign.BuildSign(corner1, "{C1}");}
        if(display_thinking) { AISign.BuildSign(corner2, "{C2}");}
        if(display_thinking) { AISign.BuildSign(goal, "{BG}");}
        
        if(TryBridgeToCorner(corner1) || TryBridgeToCorner(corner2))
        {
            local count = 0;
            foreach(tile, v in path)
            {
                path.SetValue(tile, count++)
            }
            return true;
        }
    }
    return false;
  }

  //Making it to here means a strait, possibly bridgeable path.
  LevelTwoTiles(start, goal);
  local dist = AIMap.DistanceManhattan(this.current, goal);
  if(dist > this.bridge_max_len) {return false;}
  
  path = GetShortestBridge(start, goal);
  
  if(path == null)
  {
    midStart = goal;
    return false;
  }
  
  path.Sort(AIAbstractList.SORT_BY_VALUE, true);
  
  midStart = goal;
  return true;
}


function BridgePathFinder::LevelTwoTiles(tile1, tile2)
{
    //don't try to level unbuildable tile or station tiles.
    if(!AntPathFinder.isBuildable(tile1, ignoreTiles) || !AntPathFinder.isBuildable(tile2, ignoreTiles) ) { return;} 
    
    //If the hieght difference is > 0 and < 3 raise the low one.
    local th1 = AITile.GetMaxHeight(tile1);
    local th2 = AITile.GetMaxHeight(tile2);
    if(th1 == th2) {return;}
    if(abs(th1 - th2) > 2) {return;}
    local max = max(th1, th2);
    BringeTileToAltitidue(tile1, max);
    BringeTileToAltitidue(tile2, max);
}

function BridgePathFinder::BringeTileToAltitidue(tile, alt)
{    
    for(local i = 0; i < 4; i++)
    {
        while(AITile.GetCornerHeight(tile, i) < alt)
        {
            local r = AITile.RaiseTile(tile, 1 << i );
            if(r == false) {break;}
        }
        
        while(AITile.GetCornerHeight(tile, i) > alt)
        {
            local r = AITile.LowerTile(tile, 1 << i);
            if(r == false) {break;}
        }
    }
}

function BridgePathFinder::IsBridgeBuildable(bridgeHead, bridgeEnd)
{
   local dist = AIMap.DistanceManhattan(bridgeHead, bridgeEnd);
   if(dist == 1) {return false;}
   
   
   local bridge_list = AIBridgeList_Length(AIMap.DistanceManhattan(bridgeHead, bridgeEnd) + 1);
   if(bridge_list.Count() == 0) {AILog.Info("No bridges avialable.");return false;}
   bridge_list.Valuate(AIBridge.GetMaxSpeed);
   bridge_list.Sort(AIAbstractList.SORT_BY_VALUE, false);
   local bridgeBuildable = false;
   Util.GetMaxLoan();
   {
       local test = AITestMode();
       bridgeBuildable = AIBridge.BuildBridge(AIVehicle.VT_RAIL, bridge_list.Begin(), bridgeHead, bridgeEnd);
   }
   Util.PayDownLoan();
   return bridgeBuildable;
}

function BridgePathFinder::GetShortestBridge(start, goal)
{
    //if(!CheckPathNeedsBridge(start, goal)) {return false;}
    local dx = AIMap.GetTileX(start) - AIMap.GetTileX(goal);
    local dy = AIMap.GetTileY(start) - AIMap.GetTileY(goal);
    
    
    local startPath = GetTilesToFirstObstacle(start, goal);
    local endPath = GetTilesToFirstObstacle(goal, start);
    
    //return false if paths overlap (i.e. no real obstacles.)
    foreach(tile, v in startPath)
    {
        
        if(endPath.HasItem(tile)) {return null;}
    }
    
    local bridges = [];
    local count = 0;
    foreach(bridgeHead, v in startPath)
    {
        foreach(bridgeEnd, v2 in endPath)
        {
            if(CheckPathNeedsBridge(bridgeHead, bridgeEnd) && IsBridgeBuildable(bridgeHead, bridgeEnd))
            {
                bridges.push({bridgeHead = bridgeHead, bridgeEnd = bridgeEnd, dist = AIMap.DistanceManhattan(bridgeHead, bridgeEnd)});
            }
        }
    }
    
    if(bridges.len() == 0)
    {
        return null;
    }
    
    local minDist = 99999;
    local shortest = null;
    foreach(index, value in bridges)
    {
        if(value.dist < minDist)
        {
            shortest = value;
            minDist = value.dist;
        }
    }
    
    startPath = GetTilesToFirstObstacle(start, shortest.bridgeHead);
    endPath = GetTilesToFirstObstacle(shortest.bridgeEnd, goal);
    
    path.Clear();
    path.AddList(startPath);
    path.AddItem(shortest.bridgeHead, 0);
    path.AddItem(shortest.bridgeEnd, 0);
    path.AddList(endPath);

    local count = 0;
    
    if(display_thinking) { AISign.BuildSign(shortest.bridgeHead, "{BH}");}
    if(display_thinking) { AISign.BuildSign(shortest.bridgeEnd, "{BE}");}
    //DenverAndRioGrande.ClearPathSigns();
    local count = 0;
    path.Valuate(AIMap.DistanceManhattan, start);
    midStart = goal;
    //throw("a party.");
    return path;
}

function BridgePathFinder::GetTilesBetweenPoints(start, end)
{
    local dx = AIMap.GetTileX(start) - AIMap.GetTileX(end);
    local dy = AIMap.GetTileY(start) - AIMap.GetTileY(end);  
    
    
    local minDx = 0;
    local minDy = 0;
    
    if(dx < 0) {minDx = 1;}
    if(dx > 0) {minDx = -1;}
    if(dy > 0) {minDy = -1;}
    if(dy < 0) {minDy = 1;}

    local tiles = AIList();
    local count = 0;
    tiles.AddItem(start, count++);
    local nextTile = start;
    while(nextTile != end)
    {
        nextTile = AIMap.GetTileIndex(minDx, minDy) + nextTile;
        tiles.AddItem(nextTile, count++);
    }
    return tiles;
}

function BridgePathFinder::GetTilesToFirstObstacle(start, goal)
{
    local returnPath = AIList();
    count = 0; 
    local tile = GetNextClosestTileAny(start, goal);
    while(AITile.IsBuildable(tile) && tile != goal)
    {
        returnPath.AddItem(tile, count++);
        tile = GetNextClosestTileAny(tile, goal)
    }
    
    return returnPath;
}

function BridgePathFinder::TryBridgeToCorner(corner)
{
    local newBridge = BridgePathFinder(start, corner, ignoreTiles); 
    
    if(newBridge.ReEnter() == true)
    {
        this.path = newBridge.path;
        midStart = newBridge.midStart;
        return true;
    }
    return false;
}

//Check that the proposed route isn't completely flat and clear
function BridgePathFinder::CheckPathNeedsBridge(start, goal)
{
  local tile = start;
  local result = false;
  while(tile != goal)
  {
    if(AITile.IsBuildable(tile) == false) {result = true;}
    if(AITile.GetSlope(tile) != AITile.SLOPE_FLAT) {result = true;}
    tile = GetNextClosestTileAny(tile, goal);
  }
  
  return result;

}

function BridgePathFinder::GetNextClosestTileAny(tile, goal)
{
  local nodelist = AntHill.FindCompassPointsNunitsAway(tile, 1);
  nodelist.Valuate(DistanceValuator, goal);
  nodelist.Sort(AIAbstractList.SORT_BY_VALUE, true);
  local node = nodelist.Begin();
  return node;
}

function BridgePathFinder::DistanceValuator(tile, goal)
{
 return AITile.GetDistanceSquareToTile(tile, goal);
}

function BridgePathFinder::IsPathDiagonal()
{
    local dx = AIMap.GetTileX(start) - AIMap.GetTileX(goal);
    local dy = AIMap.GetTileY(start) - AIMap.GetTileY(goal);
    
    if(dx == 0 || dy == 0)
    {
        return false;
    }
    return true;
}

function BridgePathFinder::HasWaterRoadOrRailTilesToCross()
{

    local tiles = AITileList();
    tiles.AddRectangle(start, goal);
    tiles.Valuate(AIRail.IsRailTile);
    tiles.KeepValue(1);
    local railTileCount = tiles.Count();
    
    tiles.Clear();
    tiles.AddRectangle(start, goal);
    tiles.Valuate(AITile.IsWaterTile);
    tiles.KeepValue(1);
    local waterTileCount = tiles.Count();
    
    tiles.Clear();
    tiles.AddRectangle(start, goal);
    tiles.Valuate(AIRoad.IsRoadTile);
    tiles.KeepValue(1);
    local roadTileCount = tiles.Count();
        
    if(railTileCount > 1 || waterTileCount > 1 || roadTileCount > 1)
    {
        return true;
    }
     return false;
}
