/*
 *
 * Copyright (c) 2009, Dustin Andrews
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * Redistributions of source code must retain the above copyright notice, this list
 * of conditions and the following disclaimer.
 * 
 * Redistributions in binary form must reproduce the above copyright notice, this
 * list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * 
 * The names of its contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

//An "as the crow flies" pathfinder for fast, gross pathfinding.

require("AntPathFinder.nut");

class CrowPathFinder
{
  start = null;
  goal = null;
  current = null;
  midStart = null;
  midGoal = null;
  path = null;
  count = null;
  last = null;
  display_thinking = null;
  ignoreTiles = null;
  trimTiles = null;
  
  

  constructor(start, goal, ignoreTiles)
  {
    this.start = start;
    this.current = start;
    this.goal = goal;
    this.midGoal = null;
    this.midStart = null;
    this.last = null;
  
    this.count = 0;
    this.path = AIList();
    this.path.AddItem(start, count);
    this.display_thinking = AIController.GetSetting("display_thinking");
    
    this.ignoreTiles = ignoreTiles;
    this.trimTiles = 3;
  }

}

function CrowPathFinder::FindPath()
{
  if(start == goal) {return true;}


  local distance = AITile.GetDistanceSquareToTile(this.current, goal);
  if(distance == -1) {return false;}
  AILog.Info("Distance to goal is:" + AITile.GetDistanceManhattanToTile(this.current, goal));

  if(!AntPathFinder.isBuildableOrMyRail(goal))
  {
    AILog.Warning("Destination is not valid.");
    return false;
  }

  while(CrowPathFinder.MoveToNext())
  {

    local distance = AITile.GetDistanceSquareToTile(this.current, this.goal);
    //if(this.current == this.goal)
    if(distance < 75) //crow isn't great at the "last mile";
    {
      count++;
      this.path.AddItem(this.current, count);
      this.path.Sort(AIAbstractList.SORT_BY_VALUE, false);
      this.midStart = path.Begin();
      this.last = path.Next();
     //print("CrowLast == " + this.last);
      this.midGoal = goal;
      return false;
    }
  }
  AILog.Info("crow: ran into something.");

  path.Sort(AIAbstractList.SORT_BY_VALUE, false);
  path.RemoveTop(trimTiles);
  
  if(path.Count() == 0)
  {
    midStart = start;
  }
  else
  {
    this.midStart = path.Begin();
  }
  
  if(display_thinking == 1) 
  { AISign.BuildSign(midStart, "{<}");}
  
  this.last = path.Next();

  this.midGoal = CrowPathFinder.FindNextSafeTile();
  return false;
}

function CrowPathFinder::MoveToNext()
{
  local tile = CrowPathFinder.GetNextClosestTile(this.current);
  
  if(tile != null && AntPathFinder.isBuildable(tile, ignoreTiles) )
  {
    if(display_thinking == 1) { AISign.BuildSign(tile, "{.}")};
    last = current;
    current = tile;
    count++;
    this.path.AddItem(current, count);
  }
  else
  { 
    return false;
  }
  return true;
}

function CrowPathFinder::DetermineDirection(tile, last)
{
    local cardinal = true;
    if(last == null) {return;}
    local x1 = AIMap.GetTileX(tile);
    local y1 = AIMap.GetTileY(tile);
  
    local x2 = AIMap.GetTileX(last);  
    local y2 = AIMap.GetTileY(last); 
    
    local dx = x1 - x2;
    local dy = y1 - y2;
    
    if(dx > 0 && dy > 0){cardinal = false;}
    if(dx > 0 && dy < 0){cardinal = false;}
    if(dx < 0 && dy > 0){cardinal = false;}
    if(dx < 0 && dy < 0){cardinal = false;}
    
    return cardinal;

}


function CrowPathFinder::FindNextSafeTile()
{
  local distanceToGoal = AITile.GetDistanceSquareToTile(this.current, goal);
  //print("distance left:" + distanceToGoal);
  if(distanceToGoal < 15) {return goal;}
  
  local safeTiles = 0;
  local limit = 0;
  // local tile = CrowPathFinder.GetNextClosestTile(this.current);
  local offset = current - last;
  local tile = current + offset;
  
  while(safeTiles < trimTiles - 1 && limit < distanceToGoal)
  {
    if(tile == goal) {return goal;}
    local slope = AITile.GetSlope(tile);  
    
    if(AntPathFinder.isBuildable(tile, ignoreTiles) && slope == AITile.SLOPE_FLAT)
    {
      safeTiles++;
    }
    else
    {
      safeTiles = 0;
    }
    
    if(display_thinking == 1) { AISign.BuildSign(tile, "{-}");}
    tile = CrowPathFinder.GetNextClosestTileAny(tile);
    limit ++;
  }
  if(display_thinking == 1) 
  { AISign.BuildSign(tile, "{*}");}
  return tile;
}


function CrowPathFinder::GetNextClosestTile(tile)
{
  local nodelist = AntHill.FindCompassPointsNunitsAway(tile, 1);
  
  nodelist.Valuate(this.IsInPathValuator, path);
  nodelist.KeepValue(0);
   
  nodelist.Valuate(DistanceValuator, goal);
  nodelist.Sort(AIAbstractList.SORT_BY_VALUE, true);
  
  local maxCount = nodelist.Count();
  local curCount = 0;
  local node = nodelist.Begin();
  while(!this.SlopeDirectionValuator(node, last) && curCount <= maxCount)
  {
    curCount++;
    node = nodelist.Next();
  }
  
  if(curCount >= maxCount)
  {
    return null;
  }
  
  return node;
}

function CrowPathFinder::GetNextClosestTileAny(tile)
{
  local nodelist = AntHill.FindCompassPointsNunitsAway(tile, 1);
  
   
  nodelist.Valuate(DistanceValuator, goal);
  nodelist.Sort(AIAbstractList.SORT_BY_VALUE, true);
  
  local node = nodelist.Begin();
  return node;
}

function CrowPathFinder::IsInPathValuator(idx, path)
{
  foreach(i, v in path)
  {
    if(idx == i) {return true;}
  }
  return false;
}

function CrowPathFinder::SlopeDirectionValuator(tile, last)
{
 if(last == null) {return true;}
 local slopeCurrent = AITile.GetSlope(current);
 local slopeTile = AITile.GetSlope(tile);
 local slopeLast = AITile.GetSlope(last);
 if(slopeCurrent == AITile.SLOPE_FLAT && slopeTile == AITile.SLOPE_FLAT && slopeLast == AITile.SLOPE_FLAT) {return true;}
 
 local cardinal = CrowPathFinder.DetermineDirection(tile, last);

 return cardinal;
}


function CrowPathFinder::DistanceValuator(tile, goal)
{
 return AITile.GetDistanceSquareToTile(tile, goal);
}
