/*
 *
 * Copyright (c) 2009, Dustin Andrews
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 
 * Redistributions in binary form must reproduce the above copyright notice, this
 * list of conditions and the following disclaimer in the documentation and/or 
 *  other materials provided with the distribution.
 * 
 * The names of its contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

class StationManager
{
  offset = 5;
  MaxTrainCount = null;
  
  industryIndex = null;
  Location = null;
  Length = null;
  DepotFront = null;
  Direction = null;
  StationId = null;
  industryLocation = null;
  Tiles = null;
  JoinTiles = null;
  Name = null;
  Depot = null;
  NumPlatforms = 1;
  TrackSignals = null;
  IgnoreTiles = null;
  testMode = false;
  treesFull = false;
  
  OutOfCash = false;
  Deleted = false;
  Deleting = false;
  
  IndustryType = null;
  Produces = null; 
  Accepts = null;
  
  TrainSize = null;
  Route = null;
  Consumer = null;
  
  Trains = null;
  PassingLen = null;
  TransPortMinPercent = null;
  
  LastCheckTime = null;
  RouteProfit = 0;
  CheckIntervalDays = null;
  
  constructor(industryIndex = 0, Name = "unnamed", NumPlatforms = 2, trainSize = 8)
  {
    this.industryIndex = industryIndex;
    this.Name = Name;
    this.industryLocation = AIIndustry.GetLocation(industryIndex);
    this.Location = null;
    this.DepotFront = null;
    this.Direction = null;
    this.Length = null;
    Tiles = AIList();
    JoinTiles = AIList();
    Depot = null;
    this.NumPlatforms = NumPlatforms;
    StationId = null;
    TrackSignals = [];
    IgnoreTiles = AITileList();
    testMode = false;
    treesFull = false;
    Deleted = false;
    Deleting = false;
    
    this.IndustryType = AIIndustry.GetIndustryType(industryIndex);
    this.Produces = AIIndustryType.GetProducedCargo(IndustryType);
    this.Accepts =  AIIndustryType.GetAcceptedCargo(IndustryType);
    this.TrainSize = trainSize;
    this.Route = {}; //{Path = null, Passing = null};
    this.Consumer = null;
    
    this.Trains = AIList();
    PassingLen = 0;    
    MaxTrainCount = 2;
    TransPortMinPercent = AIController.GetSetting("transport_min_percent");
    LastCheckTime = 0;
    RouteProfit = 0;
    CheckIntervalDays = 300; //how often station maintainence runs.
  }
}



function StationManager::Save()
{
  local saveTable = 
  {
    industryIndex = industryIndex,
    Name = Name,
    industryLocation = industryLocation,
    Location = Location,
    DepotFront = DepotFront,
    Length = Length,
    Depot = Depot,
    NumPlatforms = NumPlatforms,
    StationId = StationId,
    Deleted = Deleted,
    Deleting = Deleting,
    TrainSize = TrainSize,
    PassingLen = PassingLen,
    MaxTrainCount = MaxTrainCount,
    LastCheckTime = LastCheckTime,
    Tiles = AIListToTable(Tiles),
    JoinTiles = AIListToTable(JoinTiles),
    Trains = AIListToTable(Trains)  
  };
    
  if(Consumer != null)
  {
    saveTable.Consumer <- Consumer.Save();
  }
  
  if(Route != null)
  {
    if(Route.rawin("Path"))
      saveTable.Path <- AIListToTable(Route.Path);
    
    if(Route.rawin("Passing"))
      saveTable.Passing <- AIListToTable(Route.Passing);
  }
  return saveTable;
}

function StationManager::Load(data)
{
  industryIndex = LoadHelper("industryIndex", data);
  Name = LoadHelper("Name", data);
  industryLocation = LoadHelper("industryLocation", data);
  Location = LoadHelper("Location", data);
  DepotFront = LoadHelper("DepotFront", data);
  Length = LoadHelper("Length", data);
  Depot = LoadHelper("Depot", data);
  NumPlatforms = LoadHelper("NumPlatforms", data);
  StationId = LoadHelper("StationId", data);
  Deleted = LoadHelper("Deleted", data);
  Deleting = LoadHelper("Deleting", data);
  TrainSize = LoadHelper("TrainSize", data);
  //Consumer = LoadHelper("Consumer", data);
  PassingLen = LoadHelper("PassingLen", data);
  MaxTrainCount = LoadHelper("MaxTrainCount", data);
  LastCheckTime = LoadHelper("LastCheckTime", data);
  
  local arrayRoutePath = [];
  arrayRoutePath = LoadHelper("Path", data);
  Route.Path <- ArrayToAIList(arrayRoutePath);
  
  local arrayRoutePassing = [];
  arrayRoutePassing = LoadHelper("Passing", data);
  Route.Passing <- ArrayToAIList(arrayRoutePassing);
  
  local arrayTiles = [];
  arrayTiles = LoadHelper("Tiles", data);
  Tiles = ArrayToAIList(arrayTiles);
  
  local arrayJoinTiles = [];
  arrayJoinTiles = LoadHelper("JoinTiles", data);
  JoinTiles = ArrayToAIList(arrayJoinTiles);
  
  local arrayTrains = [];
  arrayTrains = LoadHelper("Trains", data);
  Trains = ArrayToAIList(arrayTrains);  
  
  
  if(data.rawin("Consumer"))
  {
    Consumer = StationManager();
    Consumer.Load(data.Consumer);
  }
  
}


function StationManager::LoadHelper(dataName, data)
{
  //print("Looking for " + dataName );
  local input;
  if(data.rawin(dataName))
  {
    input = data.rawget(dataName);
    //print("found the value: " + input);
  }
  return input;
}

function StationManager::AIListToTable(aiList)
{  
  local table = {};
  foreach(i, v in aiList)
  {
    table[i] <- v;
  }
  return table;
}


function StationManager::ArrayToAIList(array)
{
  if(array == null) {return null;}
  local aiList = AIList();
  foreach(i, v in array)
  {
    aiList.AddItem(i, v);
  }
  return aiList;
}

function StationManager::PlaceProducingStationTest(length)
{
  testMode = true;
  return this.PlaceProducingStation(length);
  testMode = false;
}

function StationManager::PlaceProducingStation(length)
{
  local producingTiles = AITileList_IndustryProducing(industryIndex, 4);
  local result = PlaceStation(length, producingTiles);
  //Tiles.Valuate(Inverse);
  return result;
}


function StationManager::Inverse(integer)
{
  return integer * -1;
}


function StationManager::PlaceConsumingStationTest(length, producer)
{
  testMode = true;
  return this.PlaceConsumingStation(length, producer);
  testMode = false;
}

function StationManager::PlaceConsumingStation(length, producer)
{
  local acceptingTiles = AITileList_IndustryAccepting(industryIndex, 4);

  acceptingTiles.Valuate(AITile.GetDistanceManhattanToTile, producer);
  acceptingTiles.Sort(AIAbstractList.SORT_BY_VALUE, true);  
  local result = PlaceStation(length, acceptingTiles);
  return result;
}

function StationManager::PlaceStation(length, tileList)
{  
  
  local result = TryLocations(length, tileList);

  foreach(tile, v in Tiles)
  {
    //add border tiles to list of tiles for joining track to.
    if((v + 1) % (length + offset) == 0 || v == 1 || (v - 1) % (length + offset) == 0)
    {
      if((length + offset) * NumPlatforms == v)
      {
        //special case the depot.
        IgnoreTiles.AddTile(tile);
      }
      else
      {
        JoinTiles.AddItem(tile, 0);
      }
    }
    else
    {
      //keep pathfinder from running under station to be.
      IgnoreTiles.AddTile(tile);
    }
  }
  
  
  if(testMode == true)
  {
    return result;
  }
  
  if(result == true)
  {
    result = _PlaceStation(length); 
  }
  

  
  return result;
}

function StationManager::CreateFeatures(length)
{
  foreach(tile, v in Tiles)
  {
    BringTileToLevel(tile);
  }
  
  Tiles.Sort(AIAbstractList.SORT_BY_VALUE, true);
  foreach(tile, v in Tiles)
  {
    //build parrallel rails.
    if(v % (length + offset) != 0 && (v + 1) % (length + offset) != 0 && v < ((length + offset) * 2))
    {
      AIRail.BuildRailTrack(tile, Direction);
    }
    
    //make ends into "diamonds"
    if((v + 1) % (length + offset) == 0 || v == 1 || (v - 1) % (length + offset) == 0)
    {
      this.BuildAllTrack(tile);
    }
  }
  
  foreach(i, v in TrackSignals)
  {
    //AIRail.BuildSignal(v.tile, v.front, AIRail.SIGNALTYPE_NORMAL);
    AIRail.BuildSignal(v.tile, v.front, AIRail.SIGNALTYPE_PBS); 
  }
  
}


function StationManager::BuildAllTrack(tile)
{  
      AIRail.BuildRailTrack(tile, AIRail.RAILTRACK_NW_NE);
      AIRail.BuildRailTrack(tile, AIRail.RAILTRACK_SW_SE);
      AIRail.BuildRailTrack(tile, AIRail.RAILTRACK_NW_SW); 
      AIRail.BuildRailTrack(tile, AIRail.RAILTRACK_NE_SE);
      AIRail.BuildRailTrack(tile, AIRail.RAILTRACK_NE_SW);
      AIRail.BuildRailTrack(tile, AIRail.RAILTRACK_NW_SE);
}

function StationManager::GetStationRectangle(tile, length, direction, platforms)
{
  assert(tile != null);
  platforms++;
  
  local tiles = AIList();
  local dx = 0;
  local dy = 0;
  if(direction == AIRail.RAILTRACK_NW_SE)
  {
    dx = 0;
    dy = 1;
  }
  
  if(direction == AIRail.RAILTRACK_NE_SW)
  {
    dx = 1;
    dy = 0;
  }

  for(local i = 0; i < platforms; i++)
  {
    //print("dy * i == " + (dy * i) + ", dx * i ==" + (dx * i));
    local newtile = tile + AIMap.GetTileIndex(dy * i, dx * i);
    tiles.AddList(GetTiles(newtile, dx, dy, length, i * length ));
  }

  return tiles;

}

function StationManager::GetTiles(tile, dx, dy, length, startIndex)
{
  local tiles = AIList();
  local count = startIndex;
  for(local i = 0; i < length; i++)
  {
    tiles.AddItem(tile, ++count);
    tile = tile + AIMap.GetTileIndex(dx, dy); 
  }
  return tiles;
}



function StationManager::IndustryDistanceValuator(tile, location)
{
  local score = AITile.GetDistanceManhattanToTile(tile, location);
  return score;
}

function StationManager::TryLocations(length, tileList)
{

  local directions = [AIRail.RAILTRACK_NE_SW, AIRail.RAILTRACK_NW_SE]; //  NE_SW = Y axis, NW_SE = X axis
  local result = false;
  
  tileList.Valuate(IndustryDistanceValuator, industryLocation);
  tileList.Sort(AIAbstractList.SORT_BY_VALUE, false);
  
  local count = 0;
  foreach(tile, v in tileList)
  {
    local q = GetQuadrant(tile, industryLocation);
    local startTile =  GetOffsetWide(tile, q);
    Util.DebugReplaceSign(startTile, count + "s" + q);
    Util.DebugReplaceSign(tile, count++ + "x" + q);
    throw("end");
  }

                              
  

//  foreach(tile, v in tileList)
//  {
//    if(AIStation.GetStationID(tile) != AIStation.STATION_INVALID) 
//    {
//      //AILog.Warning("Already a station here."); 
//      continue;
//   }
//    
//    foreach(i, direction in directions)
//    {
//      local localTestMode = AITestMode();
//      result = AIRail.BuildRailStation( tile, direction, NumPlatforms + 1, length + offset, AIStation.STATION_NEW );      
//      localTestMode = null; 
//      if(AIError.GetLastErrorString() == "ERR_LOCAL_AUTHORITY_REFUSES") 
//      {
//        //AILog.Warning("Local Authority Refuses");
//        local exec = AIExecMode();
//        this.PlantTrees();
//      }
//
//      if(result == true)
//      {
//        this.Direction = direction;
//        Tiles = GetStationRectangle(tile, length + offset, Direction, NumPlatforms);
//        this.GetStationFeatures(Tiles, length);
//       return true;
//      }  
//    }  
//  }
 return false;
}



function StationManager::GetOffsetWide(tile, quadrant)
{
  local newtile;
  switch(quadrant)
  {
    case 0:
      newtile = AIMap.GetTileIndex(2, 0) + tile;
      break;
    case 1:
      newtile = AIMap.GetTileIndex(2, 1) + tile;
      break;
    case 2:
      newtile = AIMap.GetTileIndex(6, 0) + tile;
      break;
    case 3:
      newtile = AIMap.GetTileIndex(6, 1) + tile;
      break;   
  }  

  return newtile;
}

function StationManager::GetQuadrant(tile, origin)
{

  local dx = AIMap.GetTileX(tile) - AIMap.GetTileX(origin);
  local dy = AIMap.GetTileY(tile) - AIMap.GetTileY(origin);
  
  if(dx > 1.1 && dy > 1.1)
  {
    return 2;
  }
  
  if(dx < 1.1 && dy > 1.1)
  {
    return 3;
  }
  
  if(dx > 1.1 && dy < 1.1)
  {
    return 0;
  }
  
  if(dx < 1.1 && dy < 1.1)
  {
    return 1;
  }
}

function StationManager::_PlaceStation(length)
{
  assert(this.Location != null);
  local tile = this.Location;
  local result = AIRail.BuildRailStation( Location, Direction, NumPlatforms, length, AIStation.STATION_NEW );
  
  if(result == true)
  {
    StationId = AIStation.GetStationID(Location);
    AIStation.SetName(StationId, this.Name);
  }

  if(result == true)
  {
    this.PlaceDepot();
  }
  
  this.CreateFeatures(length);
  
  
  
  return result;
}

function StationManager::LevelTiles(radius)
{
    //print("Leveling Around Industry with radius == " + radius);
    //AISign.BuildSign(industryLocation, "LEVEL" + radius * 4);
    if(radius == 0) {return;}
    
    local height = AITile.GetMinHeight(industryLocation);
    if(height == 0) {return;} //Don't level to sea level. 
    
    local levelTiles = []
    levelTiles.push( industryLocation + AIMap.GetTileIndex(-radius, - radius) );
    levelTiles.push( industryLocation + AIMap.GetTileIndex(radius + 3, radius + 3) );
    levelTiles.push( industryLocation + AIMap.GetTileIndex(-radius, radius + 3) );
    levelTiles.push( industryLocation + AIMap.GetTileIndex(radius + 3, -radius) );
    
    local randTile = levelTiles[AIBase.RandRange(levelTiles.len())];
    if(AIMap.DistanceManhattan(randTile, industryLocation) <= radius * 4)
    {
          //AISign.BuildSign(randTile, "LEVEL " + radius);
          AITile.LevelTiles(industryLocation, randTile);
    }
  
}



function StationManager::GetTownRating()
{
  local town = AITile.GetClosestTown(industryLocation);
  local rating = AITown.GetRating(town, AICompany.COMPANY_SELF );
  return rating;
}

function StationManager::PlantTrees()
{
  if(treesFull == true) {return;}
  local town = AITile.GetClosestTown(industryLocation);
  local townManager = TownManager(town);
  local rating = townManager.GetRating();
  
  local townLocation = AITown.GetLocation(town);
 //print(townManager.Name + " town rating is  " + rating +"."); 
  local influencedZone = townManager.GetInfluencedZone();

  
  local count = 0;
  while(rating  == AITown.TOWN_RATING_APPALLING ||  rating  == AITown.TOWN_RATING_VERY_POOR || rating == AITown.TOWN_RATING_POOR )
  {
    count++;
    rating = townManager.GetRating();
   //print (townManager.Name + ": Planting trees and waiting to improve rating "+ rating +". Tiles:" + influencedZone.Count() + " Attemp:" + count);
    local exec = AIExecMode();
    AISign.BuildSign(influencedZone.Begin(), townManager.Name + " Forest planted by Denver & Rio Grande Railroad.");
    foreach(tile, v in influencedZone)
    {
      AITile.PlantTree(tile);
    }
    
    if(AICompany.GetBankBalance(AICompany.COMPANY_SELF) > 1000000)
    {
      //burn money to increase rating.
      if(count > 4) { AIController.Sleep(1000); break;} //give up when all the tiles are treed eventually.
      AIController.Sleep(100);
    }
    else
    {
      //count on time to increase rating more than tree planting.
      //Give up after a few rounds when money is tight.
      if(count > 1) { AIController.Sleep(1000); break;}
      AIController.Sleep(2000);
    }
  }
  
  treesFull = true;
}


function StationManager::GetRadiusTiles(location, radius)
{
 //print("GetRadiusTiles(location, radius)");
  local tiles = [];  
  tiles.push(location + AIMap.GetTileIndex(0, radius));
  tiles.push(location + AIMap.GetTileIndex(radius, 0));
  tiles.push(location + AIMap.GetTileIndex(0, -radius));
  tiles.push(location + AIMap.GetTileIndex(-radius, 0));
  return tiles;
}

function StationManager::PlaceDepot()
{
  //print("PlaceDepot()"); 
  //print("PlaceDepot() Depot:"+ this.Depot + " DepotFront:" + this.DepotFront);
  //AISign.BuildSign(Depot, "D");
  //AISign.BuildSign(DepotFront, "F");
  local depotResult = AIRail.BuildRailDepot(this.Depot, this.DepotFront);
  return depotResult;
}




function StationManager::GetStationFeatures(tileList, length)
{
 //print ("GetStationFeatures()");
  
  Length = length;
  local temp = AIList();
  Tiles.AddList(tileList);
  
  //Mark Depot
 // temp.Clear();
 // temp.AddList(tileList);
 // temp.KeepValue( ((length + offset) * NumPlatforms) - 1 );
 // DepotFront = temp.Begin();
  
 // temp.Clear();
 // temp.AddList(tileList);
 // temp.KeepValue((length + offset) * NumPlatforms); 
 // Depot =  temp.Begin();
    
  //Mark Station location
  temp.Clear();
  temp.AddList(tileList);
  temp.KeepValue(3); 
  this.Location = temp.Begin();
  
 //Mark Depot
  temp.Clear();
  temp.AddList(tileList);
  temp.KeepValue( ((length + offset) * (NumPlatforms + 1)) - 1 );
  //local DepotSideSignalFront = temp.Begin();
  DepotFront = temp.Begin();
  
  
  temp.Clear();
  temp.AddList(tileList);
  temp.KeepValue( ((length + offset) * (NumPlatforms + 1)) - 2 );
  //local DepotSideSignal = temp.Begin();
  Depot = temp.Begin();
  
  //TrackSignals.push({tile = DepotSideSignal, front = DepotSideSignalFront});
  
  temp.Clear();
  temp.AddList(tileList);
  temp.KeepValue( ((length + offset) * NumPlatforms) + 2 );
  local frontSignal = temp.Begin();
  
  temp.Clear();
  temp.AddList(tileList);
  temp.KeepValue( ((length + offset) * NumPlatforms) + 3 );
  local frontSignalFront = temp.Begin(); 
  //TrackSignals.push({tile = frontSignal, front = frontSignalFront});
  
  
  //Mark side signals;
  local front = null;
  local tile = null;
  local value = 0;
  for(local i = 0; i < NumPlatforms; i++)
  {
    for(local j = 0; j < (length + offset); j++)
    {
      value++;
      if(j == 2)
      {
        temp.Clear();
        temp.AddList(tileList);
        temp.KeepValue( value );
        front = temp.Begin();      
      }
      
      
      if(j == 1)
      {
        temp.Clear();
        temp.AddList(tileList);
        temp.KeepValue( value );
        tile = temp.Begin();      
      }
      
      
      if(j == (length + offset) -4)
      {
        temp.Clear();
        temp.AddList(tileList);
        temp.KeepValue( value );
        front = temp.Begin();      
      }
      
      
      if(j == (length + offset) -3)
      {
        temp.Clear();
        temp.AddList(tileList);
        temp.KeepValue( value );
        tile = temp.Begin();      
      }
   
      
      
      if(tile != null && front != null)
      {
        TrackSignals.push({tile = tile, front = front});
        tile = null;
        front = null;
      }
    }
  }
  
  
  //remove unused tiles.
  for(local i = 1; i <= (length + offset) * (NumPlatforms + 1); i++)
  {
    if(i % (length+offset) == 0)
    {
      Tiles.RemoveValue(i);
    }
  }
  
}


function StationManager::BringTileToLevel(tile)
{
  local slope = AITile.GetSlope(tile);
  if(slope == AITile.SLOPE_FLAT) {return;} 
  local complement = AITile.GetComplementSlope(slope);
  AITile.RaiseTile(tile, complement); 
}

/* ********************* *
 * Route Management     *
 * ********************* */

function StationManager::RemoveRoute()
{
  if(Route == null) {return;}
 //print("RemoveRoute()");
  if(Route.Path != null)
  {
    RemoveTrack(Route.Path);
  }
  
  if(Route.Passing != null)
  {
    RemoveTrack(Route.Passing);
  }
}


function StationManager::RemoveTrack(path)
{
  foreach(tile, v in path)
  {
      //print("tile:" + tile + " value:" +v);
      //AIRail.RemoveRailTrack(tile, AIRail.RAILTRACK_NW_NE);
      //AIRail.RemoveRailTrack(tile, AIRail.RAILTRACK_SW_SE);
      //AIRail.RemoveRailTrack(tile, AIRail.RAILTRACK_NW_SW); 
      //AIRail.RemoveRailTrack(tile, AIRail.RAILTRACK_NE_SE);
      //AIRail.RemoveRailTrack(tile, AIRail.RAILTRACK_NE_SW);
      //AIRail.RemoveRailTrack(tile, AIRail.RAILTRACK_NW_SE);
      AITile.DemolishTile(tile);
      if(AIBridge.IsBridgeTile(tile) == true)
      {
        AIBridge.RemoveBridge(tile);
      }
  }

}

/* ********************* *
 * Station Management    *
 * ********************* */

function StationManager::PeriodicCheckup()
{ 
  if(this.Deleting == true)
  {
    DeleteTrainsTrackAndStations();
    return false;
  }

  if(this.Deleted == true || this.StationId == null || AIRail.IsRailDepotTile(this.Depot) == false)
  {
    DeleteTrainsTrackAndStations();
    return false;
  }
  
  
  if(CheckIndustryExists() == false)
  {
    DeleteTrainsTrackAndStations();
    return false;
  }
  
  local currentDate = AIDate.GetCurrentDate();
  if((currentDate - LastCheckTime) < CheckIntervalDays)
  {
    return false;
  }
   
  LastCheckTime = currentDate;
  this.MaxTrainCount = max(3, (PassingLen / (TrainSize * 2)));
  
  RouteProfit = 0;
  foreach(train, v in Trains)
  {
   if(AIVehicle.IsValidVehicle(train) == false)
   {
    Trains.RemoveItem(train);
   }
   else
   {
    RouteProfit += AIVehicle.GetProfitLastYear(train);
   }
  }
  
  //print("RouteProfit == " + RouteProfit);
  if(RouteProfit < -2000 * Trains.Count()) 
  {
    AILog.Warning("This route is a net loss. Removing. " + Name);
    DeleteTrainsTrackAndStations();
    return true;
  }
  
  foreach(cargo, v in Produces)
  {
    local produced = 1.0 * AIIndustry.GetLastMonthProduction(industryIndex, cargo);
    local transported = 1.0 * AIIndustry.GetLastMonthTransported(industryIndex, cargo);
    local transportPerCent = (transported / produced) * 100;
    //print(Name +": "+ transportPerCent);
    if(transportPerCent < TransPortMinPercent)
    {
     //print("This route needs more trains.");
      if(Trains.Count() < MaxTrainCount)
      {
       //print("Building more trains.");
        BuildTrain(cargo); 
      }
    }
  }
  
  AIStation.SetName(StationId, this.Name + " (" + Trains.Count() + "/" + MaxTrainCount + ")"); 
  return true;
}


function StationManager::DeleteTrainsTrackAndStations()
{
    if(this.Deleted){return;}

    if(this.Deleting != true) 
    {
        AILog.Warning("Deleting Trains Track And Stations for " + Name + ".");
        foreach(train, v in Trains)
        {
          if(this.Deleting != true) {AILog.Info("Sending Train to depot:" + AIVehicle.GetName(train));}
          AIVehicle.SendVehicleToDepot(train);
          //main.nut will delete any stopped trains during the main cycle.
        }
        this.Deleting = true; //it may take several management cycles for all the trains to get home. Set a flag.
    }
    
    local trainsStopped = true;
    foreach(train, v in Trains)
    {
     if(AIVehicle.IsValidVehicle(train) == false)
     {
      Trains.RemoveItem(train)
     }
     else
     {
      trainsStopped = false;
     }
    }
    
    if(trainsStopped == true)
    {
      RemoveRoute();
      _DeleteStation();
    }
}

function StationManager::CheckIndustryExists()
{
 //print("CheckIndustryExists()");
  local industryList = AIIndustryList();
  industryList.Valuate(AIIndustry.GetLocation);
  industryList.KeepValue(industryLocation);
  if(industryList.Count() == 0)
  {
    return false;
  }
  
  //if there is an industry here, check that it's the same one this station started with.
  
  return industryList.Begin() == industryIndex;
}


function StationManager::_DeleteStation()
{
 //print("_DeleteStation()");
  if(Consumer != null)
  {
    Consumer._DeleteStation();
  }
  
  foreach(tile, i in Tiles)
  {
    AITile.DemolishTile(tile);
  }
  AITile.DemolishTile(Depot);
  this.Deleted = true;
}

/* ********************* *
 * Train Management      *
 * ********************* */
function StationManager::BuildTrains()
{
  local rank = AIList();
  rank.AddList(Produces);
  foreach(cargo, v in rank)
  {
    rank.SetValue(cargo, AIIndustry.GetLastMonthProduction(industryIndex, cargo));
  }
  rank.Sort(AIAbstractList.SORT_BY_VALUE, false);
  
  local cargoIndex = rank.Begin();
  if(Trains.Count() < MaxTrainCount)
  {
    BuildTrain(cargoIndex);
  }
      
}

function StationManager::BuildTrain(cargoIndex)
{
    if(AIRail.IsRailDepotTile(this.Depot) == false) 
    {
        DeleteTrainsTrackAndStations();
        return;
    }
   
   local bestWagon = DenverAndRioGrande.FindBestWagon(cargoIndex);
   local bestEngine = DenverAndRioGrande.FindBestEngine(bestWagon, TrainSize, cargoIndex);
   
   local cost  = AIEngine.GetPrice(bestWagon) * TrainSize;
   cost += AIEngine.GetPrice(bestEngine);
   
   Util.GetMaxLoan();
   
   if(AICompany.GetBankBalance(AICompany.COMPANY_SELF) < cost)
   {
    return;
   }
   
   local engineId = AIVehicle.BuildVehicle(Depot, bestEngine);
   local err = AIError.GetLastErrorString();
   local name = AIEngine.GetName(bestEngine);


   if(AIVehicle.IsValidVehicle(engineId) == false) 
   {

    AILog.Warning("Failed to build engine '" + name +"':" + err);
    return;
   }
   
    Trains.AddItem(engineId, 0);
    local maxAge = AIVehicle.GetMaxAge(engineId) / 365 ;
    AILog.Info("Built Engine "+ name + "(" + maxAge +")");
      
   
   AIVehicle.RefitVehicle(engineId, cargoIndex);
//   if(cabooseId != null)
//   {
//    local caboose = AIVehicle.BuildVehicle(Depot, cabooseId);
//    AIVehicle.MoveWagon(caboose, 0, engineId, 0);
//   }
   
   for(local i = 0; i < TrainSize; i++)
   {
    local newWagon = AIVehicle.BuildVehicle(Depot, bestWagon);
        
    AIVehicle.RefitVehicle(newWagon, cargoIndex);
    local result = AIVehicle.MoveWagon(newWagon, newWagon, engineId, engineId);
              
    AIController.Sleep(10);
    if(result == false)
    {
      //AILog.Error("Couldn't join wagon to train: " + AIError.GetLastErrorString());
      result = AIVehicle.MoveWagon(newWagon, 0, engineId, 0);
      if(result == false)
      {
        //AILog.Error("Couldn't join wagon to train: " + AIError.GetLastErrorString());
        result = AIVehicle.MoveWagon(0, newWagon, 0, engineId);
        if(result == false)
        {                  
          if(i==0)
          {
            AILog.Error("Couldn't join wagon to train: " + AIError.GetLastErrorString());         
            return;
          }
        }
      }
    }
   }
   Util.PayDownLoan();
   AIOrder.AppendOrder(engineId, Route.Start, AIOrder.AIOF_FULL_LOAD_ANY | AIOrder.AIOF_NON_STOP_INTERMEDIATE);
   AIOrder.AppendOrder(engineId, Route.End, AIOrder.AIOF_NO_LOAD | AIOrder.AIOF_NON_STOP_INTERMEDIATE);
   AIOrder.AppendOrder(engineId, Consumer.Depot, AIOrder.AIOF_SERVICE_IF_NEEDED | AIOrder.AIOF_NON_STOP_INTERMEDIATE);
   AIOrder.AppendConditionalOrder(engineId, 0);
   AIOrder.SetOrderCondition(engineId, 3, AIOrder.OC_AGE);
   AIOrder.SetOrderCompareFunction(engineId, 3, AIOrder.CF_LESS_THAN); 
   AIOrder.SetOrderCompareValue(engineId, 3, maxAge - 3);
   AIOrder.AppendOrder(engineId, Consumer.Depot, AIOrder.AIOF_STOP_IN_DEPOT | AIOrder.AIOF_NON_STOP_INTERMEDIATE);
   local name = AIVehicle.GetName(engineId);
   local cargoId = AICargo.GetCargoLabel(cargoIndex);
   AIVehicle.SetName(engineId, name + " (" + Name + ") " + cargoId);
   
   AIVehicle.StartStopVehicle(engineId);
   
}

