/**
 * @author Daniela Plachtovs
 * @file direction.nut
 */
 
 /**
  * @brief static class Direction
  */
class Direction 
{
	static NORTH = 1;				/* north direction */
	static NORTHEAST = 2;			/* northeast direction */
	static EAST = 3;				/* east direction */
	static SOUTHEAST = 4;			/* southeast direction */
	static SOUTH = 5;				/* south direction */
	static SOUTHWEST = 6;			/* southwest direction */
	static WEST = 7;				/* west direction */
	static NORTHWEST = 8;			/* northwest direction */
	static INVALID_DIRECTION = 0;	/* invalid direction */

	static direction_names = ["INVALID_DIRECTION", "NORTH" , "NORTHEAST", "EAST", 
							  "SOUTHEAST", "SOUTH" , "SOUTHWEST", "WEST" , "NORTHWEST"];

	/**
	 * @brief ToString
	 * @return string direction
	 */
	function ToString(direction)
	{
		if(Direction.INVALID_DIRECTION <= direction && direction <= Direction.NORTHWEST){
			return Direction.direction_names[direction];
		} else throw("Invalid direction index.");
	}
	
	/**
	 * @brief Get Direction Turned
	 * @param direction start direction
	 * @param turn how much you want to turn (like clock)
	 * @return turned direction
	 * @note only go clockwise (do not use negative turn)
	 */
	function GetDirectionTurned(direction, turn) 
	{
		local direction_turn;
		if(direction + turn > 8) {
			direction_turn = (direction + turn) % 8;
		} else direction_turn = direction + turn;
		//AILog.Info("Dir: " + direction + " Dir Turn: " + direction_turn);
		return direction_turn;
	}
	/**
	 * Get Direction offset
	 * @param direction direction
	 * @param x x offset
	 * @param y y offset
	 * @return direction offset
	 */
	function GetDirectionOffset(direction, x, y)
	{
		switch(direction) {
			case Direction.NORTHEAST: return AIMap.GetTileIndex(-x, 0);
			case Direction.SOUTHWEST: return AIMap.GetTileIndex(x, 0);
			case Direction.NORTHWEST: return AIMap.GetTileIndex(0, -y);
			case Direction.SOUTHEAST: return AIMap.GetTileIndex(0, y);
			case Direction.NORTH: return AIMap.GetTileIndex(-x, -y);
			case Direction.SOUTH: return AIMap.GetTileIndex(x, y);
			case Direction.EAST: AIMap.GetTileIndex(-x, y);
			case Direction.WEST: AIMap.GetTileIndex(x, -y);
			default: throw("INVALID DIRECTION");
		}
	}

	/**
	 * Get Direction To tile
	 * @param from_tile from tile
	 * @param to_tile to tile
	 * @return to tile direction
	 */
	function GetDirectionToTile(from_tile, to_tile)
	{
		local f_x = AIMap.GetTileX(from_tile);
		local f_y = AIMap.GetTileY(from_tile);
		local t_x = AIMap.GetTileX(to_tile);
		local t_y = AIMap.GetTileY(to_tile);
		
		if(abs(f_x - t_x) == abs(f_y - t_y)) {
			if(f_x > t_x) {
				if(f_y > t_y) {
					return Direction.NORTH;
				} else {
					return Direction.EAST;
				}
			} else {
				if(f_y > t_y) {
					return Direction.WEST;
				} else {
					return Direction.EAST;
				}
			} 
		} else if(abs(f_x - t_x) > abs(f_y - t_y)){
			if(f_x < t_x) {
				return Direction.SOUTHWEST;
			} else {
				return Direction.NORTHEAST;
			}
		} else {
			if(f_y < t_y) {
				return Direction.SOUTHEAST;
			} else {
				return Direction.NORTHWEST;
			}
		}

	}
}
