/**
 * @author Daniela Plachtova
 * @file rail_tile_info.nut
 */

 
 /**
  * @brief class RailDemolisher manages all rail demolish functions 
  *        it also stores all railtrack tiles into table }because of save/load compatibility and good access time (better than array)
  */
 class RailDemolisher
 {
	static RAILTILE_TYPE_RAILTRACK = 0;			/* Railtile type railtrack */
	static RAILTILE_TYPE_BRIDGE = 1;			/* Railtile type bridge */
	static RAILTILE_TYPE_TUNNEL = 2;			/* Railtile type tunnel */
	
	
 	_rail_tiles = null;							/* table of built rail tiles */
	_using_routes_ids = null;					/* array of route ids that demolisher is using (even unfinished routes) */
	
	/**
	 * @brief constructor for RailDemolisher
	 */
	constructor() 
	{
		this._rail_tiles = {};
		this._using_routes_ids = array(0);
	}
	

	/**
	 * @brief Demolish station
	 * @param station station id to demolish
	 */
	function DemolishStation(station);
	
	/**
	 * @brief Demolish All railtracks of specific route (it will only demolish railtrack if it is the only one using route)
	 * @param route_id route id we want to demolish (if route_id == 0 we want to demolish all routes that are not in route list)
	 */
	function DemolishRouteTracks(route_id);
	
	/**
	 * @brief Remove bridge only if no other route uses it
	 * @param tile tile of the bridge
	 * @param rail_tile_info rail tile info
	 * @param route_id route id we want bridge demolished
	 * @return updated rail tile info (if null we can remove this tile from rail tiles list)
	 */
	function RemoveBridge(tile, rail_tile_info, route_id);
	
	/**
	 * @brief Remove tunnel only if no other route uses it
	 * @param tile tile of the tunnel
	 * @param rail_tile_info rail tile info
	 * @param route_id route id we want tunnel demolished
	 * @return updated rail tile info (if null we can remove this tile from rail tiles list)
	 */
	function RemoveTunnel(tile, rail_tile_info, route_id);
	
	/**
	 * @brief Remove Railtrack on this tile only if no other route uses it
	 * @param tile tile of the railtrack
	 * @param rail_tile_info rail tile info
	 * @param route_id route id we want railtracks demolished
	 * @return updated rail tile info (if null we can remove this tile from rail tiles list)
	 */
	function RemoveRailTracks(tile, rail_tile_info, route_id);
	
	/**
	 * @brief Removes Railtrack of the route id from the route owners list in tile
	 * @param rail_tile_info rail tile info we want railtrack of the route id removed
	 * @param route_id route id
	 * @param railtrack railtrack to remove
	 * @return updated rail tile info
	 */
	function RemoveRailTrackFromRouteOwnersIds(rail_tile_info, route_id, railtrack);
	
	/**
	 * @brief Get all railtracks on tile of this route 
	 * @param rail_tile_info rail tile info we want get route id railtracks
	 * @param route_id route id, if 0 we get all railtracks of all routes on tile
	 * @return railtrack array of all railtracks on this tile of this route
	 */
	function GetAllRailtracksOnTileOfThisRoute(rail_tile_info, route_id);
	
	/**
	 * @brief Is railtrack on this tile only used by this route
	 * @param rail_tile_info rail tile info 
	 * @param route_id route id
	 * @param railtrack railtrack
	 * @return true if the specific railtrack isonly used by this route and no other route (it can be demolished) 
			   else false if it is also uses another route
	 */
	function IsTileRailTrackUsedOnlyByThisRoute(rail_tile_info, route_id, railtrack);
	
	/**
	 * @brief Adds tile into rail tiles list 
			 if tile does not exist, creates rail tile info table with railtile type 
			 (tunnel, bridge or railtrack) and array of (route id, railtrack pair)
	 * @param tile tile to add
	 * @param type type of the tile (tunnel, bridge or railtrack)
	 * @param route id route id
	 * @param railtrack railtrack type (for tunnels and bridges it is AIRail.RAILTRACK_INVALID)
	 */
	function AddRouteTileIntoRailTilesList(tile, type, route_id, railtrack);
	
	/**
	 * @brief Adds tilearray of the same railtrack type into rail tiles list 
	 * @param tiles tile array to add
	 * @param type type of the tile (tunnel, bridge or railtrack)
	 * @param route id route id
	 * @param railtrack railtrack type (for tunnels and bridges it is AIRail.RAILTRACK_INVALID)
	 */
	function AddRouteTilesIntoRailTilesList(tiles, type, route_id, railtrack); 
	
	/**
	 * @brief Build signs with info of the route (for debug purposes only)
	 * @param route_id route id (if zero build sign for every route)
	 */
	function BuildSignRoute(route_id);
	
	/**
	 * @brief Get rail tile info of the tile
	 * @param tile tile
	 * @return rail tile info of the tile
	 */
	function GetRailTileInfo(tile);
	
	/**
	 * @brief auxiliary save function for the game
	 * @return RailDemolisher data in table
	 */
	function Save();
	
	/**
	 * @brief auxiliary load function for the game
	 * @param data data to load
	 */
	function Load(data);
 }
 
function RailDemolisher::BuildSignRoute(route_id) 
{
	foreach(tile, rail_tile_info in this._rail_tiles) {
		foreach(route_owner_id in rail_tile_info._route_owners_ids) {
			if(route_id != 0) {
				if(route_owner_id._route_id == route_id) {
					AISign.BuildSign(tile, route_id + "T" + route_owner_id._railtrack + "U" + IsTileRailTrackUsedOnlyByThisRoute(rail_tile_info, route_id, route_owner_id._railtrack) + rail_tile_info._route_owners_ids.len());
				}
			} else {
				AISign.BuildSign(tile, "R" + route_owner_id._route_id + "T" + route_owner_id._railtrack);
			}
		}
	}
} 
function RailDemolisher::DemolishStation(station) 
{
	//get all tiles of the station
	local station_tilelist = AITileList_StationType(station, AIStation.STATION_TRAIN);	
	//demolish all tiles of the station
	for(local tile = station_tilelist.Begin(); !station_tilelist.IsEnd(); tile = station_tilelist.Next()) {
		AITile.DemolishTile(tile);
	}
}
	
function RailDemolisher::DemolishRouteTracks(route_id) 
{
	local routes_ids = array(0);
	//if route_id == 0 we need to find all routes that are not in our routes list
	if(route_id == 0) {
		local existing_routes_ids = ::ai_instance._rail_manager.GetAllRoutesIds();
		local using_routes_ids_copy = array(0);
		using_routes_ids_copy.extend(this._using_routes_ids);
		local using_routes_ids_copy_length = using_routes_ids_copy.len();
		for(local i = 0; i < using_routes_ids_copy_length;) {
			for(local j = 0; j < existing_routes_ids.len(); j++) {
				if(using_routes_ids_copy[i] == existing_routes_ids[j]) {
					using_routes_ids_copy.remove(i);
					using_routes_ids_copy_length = using_routes_ids_copy.len();
					i--;
					break;
				}
			}
			i++;
		}
		routes_ids.extend(using_routes_ids_copy);
	} else {
		routes_ids.push(route_id);
	}
	//there are no noexisting routes, abort
	if(routes_ids.len() == 0) {
		AILog.Info("No routes to demolish");
		return;
	}
	//for all routes we want to demolish
	foreach(route_id in routes_ids) {
		foreach(tile, rail_tile_info in this._rail_tiles) {
			//AILog.Info(i + "DemolishRouteTracks rail_tiles count:  " + this._rail_tiles.len());
			//AILog.Info(this._rail_tiles[i]);
			
			//check for null (tile was demolished)
			//if(rail_tile_info != null) {
				//remove tile by type (bridge, tunnel, railtrack)
				switch(rail_tile_info._type) {
					case RailDemolisher.RAILTILE_TYPE_BRIDGE: this._rail_tiles[tile] <- RemoveBridge(tile, rail_tile_info, route_id);
															break;
					case RailDemolisher.RAILTILE_TYPE_TUNNEL: this._rail_tiles[tile] <- RemoveTunnel(tile, rail_tile_info, route_id);
															break;
					case RailDemolisher.RAILTILE_TYPE_RAILTRACK: this._rail_tiles[tile] <- RemoveRailTracks(tile, rail_tile_info, route_id);
															   break;
					default: break;
				}
			//}

		}
	}	
	
	//now we remove all tiles we properly demolished (this is tricky because we use table and it does not have support for looping by index)
	local last_used_tile = -1;
	local deleting;
	do {
		deleting = false;
		//tile is key and value is rail_tile_info (we do not use it here)
		foreach(tile, rail_tile_info in this._rail_tiles) {
			//we skip to where was the last tile ok
			if(this._rail_tiles[tile] == null) {
				delete this._rail_tiles[tile];
			} 
		}
		foreach(tile, rail_tile_info in this._rail_tiles) {
			if(this._rail_tiles[tile] == null) {
				//AILog.Warning("There are still null routes");
				deleting = true;
			}
		}
	} while(deleting);
	
	
	//now we update using_routes_ids list and remove routes ids we demolished
	local using_routes_ids_length = this._using_routes_ids.len();
	for(local i = 0; i < using_routes_ids_length;) {
		for(local j = 0; j < routes_ids.len(); j++) {
			if(this._using_routes_ids[i] == routes_ids[j]) {
				this._using_routes_ids.remove(i);
				using_routes_ids_length = this._using_routes_ids.len();
				i--;
				break;
			}
		}
		i++;
	}
	
}
	
function RailDemolisher::RemoveBridge(tile, rail_tile_info, route_id) 
{
	if(IsTileRailTrackUsedOnlyByThisRoute(rail_tile_info, route_id, AIRail.RAILTRACK_INVALID)) {
		AIBridge.RemoveBridge(tile);
		rail_tile_info = null;
	} else {
		rail_tile_info = RemoveRailTrackFromRouteOwnersIds(rail_tile_info, route_id, AIRail.RAILTRACK_INVALID);
	}
	return rail_tile_info;
}
	
function RailDemolisher::RemoveTunnel(tile, rail_tile_info, route_id) 
{
	if(IsTileRailTrackUsedOnlyByThisRoute(rail_tile_info, route_id, AIRail.RAILTRACK_INVALID)) {
		AITunnel.RemoveTunnel(tile);
		rail_tile_info = null;
	} else {
		rail_tile_info = RemoveRailTrackFromRouteOwnersIds(rail_tile_info, route_id, AIRail.RAILTRACK_INVALID);
	}
	return rail_tile_info;
}
	
function RailDemolisher::RemoveRailTracks(tile, rail_tile_info, route_id) 
{
	//we need to get all railtracks of this route id on this tile
	local railtracks = GetAllRailtracksOnTileOfThisRoute(rail_tile_info, route_id);
	foreach(railtrack in railtracks) {
		if(IsTileRailTrackUsedOnlyByThisRoute(rail_tile_info, route_id, railtrack)) {
			if(!AIRail.RemoveRailTrack(tile, railtrack)) {
				rail_tile_info = RemoveRailTrackFromRouteOwnersIds(rail_tile_info, route_id, railtrack);
				if(rail_tile_info._route_owners_ids.len() == 0) {
					rail_tile_info = null;
					AITile.DemolishTile(tile);
				}
			}
		} else {
			rail_tile_info = RemoveRailTrackFromRouteOwnersIds(rail_tile_info, route_id, railtrack);
		}
	}
	
	return rail_tile_info;
}
	
function RailDemolisher::RemoveRailTrackFromRouteOwnersIds(rail_tile_info, route_id, railtrack) 
{
	foreach(idx, route_owner_id in rail_tile_info._route_owners_ids) {
		if(route_owner_id._route_id == route_id && route_owner_id._railtrack == railtrack) {
			rail_tile_info._route_owners_ids.remove(idx);
			break;
		}
	}
	return rail_tile_info;
}

function RailDemolisher::GetRailTileInfo(tile) 
{
	if(this._rail_tiles.rawin(tile)) {
		return this._rail_tiles.rawget(tile);
	} else {
		return null;
	}
}

function RailDemolisher::GetAllRailtracksOnTileOfThisRoute(rail_tile_info, route_id) 
{
	local railtracks = array(0);
	foreach(route_owner_id in rail_tile_info._route_owners_ids) {
		if(route_id != 0) {
			if(route_owner_id._route_id == route_id) {
				railtracks.push(route_owner_id._railtrack);
			}
		} else {
			railtracks.push(route_owner_id._railtrack);
		}
	}
	return railtracks;
}
	
function RailDemolisher::IsTileRailTrackUsedOnlyByThisRoute(rail_tile_info, route_id, railtrack) 
{ 
	foreach(route_owner_id in rail_tile_info._route_owners_ids) {
		if(route_owner_id._route_id != route_id && route_owner_id._railtrack == railtrack) return false;
	}
	return true;
}
	
function RailDemolisher::AddRouteTileIntoRailTilesList(tile, type, route_id, railtrack) 
{
	if(railtrack == 0) return;
	
	if(this._rail_tiles.rawin(tile)) {
		foreach(route_owner_id in this._rail_tiles[tile]._route_owners_ids) {
			//check for duplicate
			if(route_owner_id._route_id == route_id && route_owner_id._railtrack == railtrack) return;
		}
		//add railtrack into route owners ids array
		this._rail_tiles[tile]._route_owners_ids.push({_route_id = route_id, _railtrack = railtrack});
	} else {
	//new tile, create table of railtile type and array of route owners ids
		local rail_tile_info = 
		{
			_type = type,
			_route_owners_ids = [{_route_id = route_id, _railtrack = railtrack}]
		}
		this._rail_tiles.rawset(tile, rail_tile_info);
	}
	/*
	foreach(item in this._rail_tiles[tile]._route_owners_ids) {
		AISign.BuildSign(tile, item._route_id + "R" + item._railtrack + "C" + this._rail_tiles[tile]._route_owners_ids.len());
	}*/
}
	
function RailDemolisher::AddRouteTilesIntoRailTilesList(tiles, type, route_id, railtrack) 
{
	foreach(tile in tiles) {
		RailDemolisher.AddRouteTileIntoRailTilesList(tile, type, route_id, railtrack);
	}
}
	
function RailDemolisher::Save()
{	
	local data = {
		railtiles = this._rail_tiles,
		usingroutesid = this._using_routes_ids
	}	
	return data;
}
	
function RailDemolisher::Load(data) 
{
	if("railtiles" in data) this._rail_tiles = data.railtiles;	
	if("usingroutesid" in data) this._using_routes_ids = data.usingroutesid;
}