/**
 * @author Daniela Plachtova
 * @file rail_manager.nut
 */

 /**
  * @brief class RailManager manages all train related functions
  */
class RailManager
{

	static NUMBER_OF_TRACKS = 2;						/* default number of tracks */
	static PLATFORM_LENGTH = 5;							/* default platform length */
	
	static RAILROAD_ROUTE_OPTIMAL_LENGTH = 220;			/* optimal railroute length with good conditions*/
	static RAILROAD_ROUTE_OPTIMAL_LENGTH_ROUGH = 110;	/* optimal railroute length with rough conditions (we we often fail to find path with RAILROAD_ROUTE_OPTIMAL_LENGTH)*/
	static RAILROAD_MAX_FAILED_TRIES_TO_FIND_ROUTE = 3; /* max failed tries to find route in a row */
	static RAILROAD_ROUTE_MINIMUM_LENGTH = 50;			/* minimal railroute length */
	static RAILROAD_ROUTE_LENGTH_TOLERANCE = 50;		/* railroute length tolerance */
	static RAILROAD_ROUTE_LENGTH_TOLERANCE_JUMP = 20;	/* railroute length tolerance jump (this is mainly for small maps so we can also find route there) */
	
	static MINIMUM_BALANCE_BUILD_RAIL_ROUTE = 200000;	/* minimal bank balanc to allow us build new rail route */

	_routes = null;							/* array of all built routes */
	_blacklist = null;						/* blacklist of all town/industry pairs that failed to build */
	_rail_demolisher = null;				/* RailDemolisher instance */
	
	route_id = null; 						/* every new created route gets its own route id */
	found_paths_failed_counter = null;		/* counter for how many routes failed to be found in a row */
	optimal_railroad_route_length = null; 	/* optimal railroad route length */
	
	/**
	 * @brief constructor for RailManager
	 */
	constructor()
	{
		this._routes = array(0);
		this._blacklist = array(0);
		this.route_id = 0;
		this._rail_demolisher = RailDemolisher();
		
		this.found_paths_failed_counter = 0;
		this.optimal_railroad_route_length = RailManager.RAILROAD_ROUTE_OPTIMAL_LENGTH;
	}
	
	/**
	 * @brief Creates New Industry Route
	 * @return true if creating new industry route suceeded else false
	 */
	function CreateNewIndustryRoute();
	/**
	 * @brief Creates New Town Route
	 * @return true if creating new town route suceeded else false
	 */
	function CreateNewTownRoute();
	
	/**
	 * @brief Print Routes
	 */
	function PrintRoutes();
	
	/**
	 * @brief Destroys and remove route from routes list
	 * @param route id route id of the route we want to remove
	 */
	function RemoveRoute(route_id);
	
	/**
	 * @brief Manages Rail Routes
	 * 		  1. check if there are vehicles in route, if not route is not ok
	 * 		  2. remove routes that are not ok
	 * 		  3. adjust number of trains in routes
	 */
	function ManageRailRoutes();
	
	/**
	 * @brief Handle crashed Train (buy new one if possible)
	 * @param vehicle crashed vehicle
	 */ 
	function HandleCrashedTrain(vehicle);
	
	/**
	 * @brief Sell Train
	 * @param vehicle train to sell
	 * @param sell_reason 0 - old age or terrible reliability (replace)
	 * 					  1 - low profit (no replace)
	 *					  2 - industry closed (no replace)
	 */
	function SellTrain(vehicle, sell_reason);
	
	/**
	 * @brief auxiliary save function for game
	 * @return RailManager in table
	 */
	function Save();
	
	/**
	 * @brief auxiliary load function for game
	 * @param data data to load
	 */
	function Load(data);
	
	/**
	 * @brief Find Suitable Industry Pair
	 * @param blacklist temporary blacklist for industries where we could not find suitable spot
	 * @return best industry pair
	 */
	function FindSuitableIndustryPair(blacklist);
	
	/**
	 * @brief Find Suitable Town Pair List
	 * @return array of suitable TownPair sorted by population
	 */
	function FindSuitableTownPairList();
	
	/**
	 * @brief Find Route
	 * @param station_from station we start from
	 * @param station_to station where we end
 	 * @return found path or null if path was not found
	 */
	function FindRoute(station_from, station_to);
	
	/**
	 * @brief Find and build paths in both ways
	 * @param route_id route id
	 * @param station_pro station producing cargo (if town station it is town1)
	 * @param station_acc station accepting cargo (if town station it is town2)
	 * @param found_existing_station_acc true if we found existing industry acc station (for town always false)
	 * @param suitable_pair suitable town or industry pair we build route for
	 * @param cargo cargo id
	 * @param railtype railtype
	 * @param is_town_route true if it is town route false if industry route
	 * @return Created RailRoute
	 */
	function BuildRoute(route_id, station_pro, station_acc, found_existing_station_acc, cargo, railtype, is_town_route);
	
	/**
	 * @brief gets all ids of existing routes
	 * @return array of existing routes
	 */	 
	function GetAllRoutesIds();
}

function RailManager::PrintRoutes() 
{
	AILog.Info("Routes count: " + this._routes.len());
	foreach(idx, route in this._routes) {
		AILog.Info(route.ToString());
		route.PrintVehicleList();
	}
}

function RailManager::GetAllRoutesIds() 
{
	local routes_ids = array(0);
	foreach(route in this._routes) {
		routes_ids.push(route._route_id);
	}
	return routes_ids;
}

function RailManager::CreateNewTownRoute() 
{
    //first thing to do is to increment route id
	this.route_id++;
	this._rail_demolisher._using_routes_ids.push(this.route_id);
	local best_town_pair;
	local town1_tilelist = AIList();
	local town2_tilelist = AIList();
	local station_pro, station_acc;
	local money_reservationID, cost;
	local railtype = AIRailTypeList().Begin();
	AIRail.SetCurrentRailType(railtype);
	
	local town_pair_array = RailManager.FindSuitableTownPairList();
	if(town_pair_array == null) {
		AILog.Info("No suitable town pair found");
		::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NO_SUITABLE_TOWN_PAIR_FOR_RAIL_ROUTE);
		return false;
	}
	//look for suitable spots for stations
	for(local i = 0; i < town_pair_array.len(); i++) {

		town1_tilelist = RailBuilder.GetTownStationSpotList(town_pair_array[i]._town1,town_pair_array[i]._town2, PLATFORM_LENGTH, 
															NUMBER_OF_TRACKS, AIStation.GetCoverageRadius(AIStation.STATION_TRAIN));
		//AILog.Info("town1_tilelist count: " + town1_tilelist.Count());
		//no suitable tiles found
		if(town1_tilelist.Count() == 0) continue; 
		
		town2_tilelist = RailBuilder.GetTownStationSpotList(town_pair_array[i]._town2, town_pair_array[i]._town1, PLATFORM_LENGTH, 
															NUMBER_OF_TRACKS, AIStation.GetCoverageRadius(AIStation.STATION_TRAIN));
		//AILog.Info("town2_tilelist count: " + town2_tilelist.Count());
		//no suitable tiles found
		if(town2_tilelist.Count() == 0) continue; 
		else {
			//we found some suitable spots for both stations breaking the loop
			AILog.Info(town_pair_array[i].ToString());
			best_town_pair = town_pair_array[i]
			break;
		}	
	}
	
	if(town1_tilelist.Count() == 0 || town2_tilelist.Count() == 0) {
		AILog.Info("No suitable town pair found");
		::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NO_SUITABLE_TOWN_PAIR_FOR_RAIL_ROUTE);
		return false;
	}
	
	//get approximate price for stations
	cost = RailBuilder.GetCostOfTownStationWithStationTracks(best_town_pair._town1, town1_tilelist, AITown.GetLocation(best_town_pair._town2), 
						RailManager.PLATFORM_LENGTH, RailManager.NUMBER_OF_TRACKS, ::ai_instance._general_manager.passenger_cargo_id, 
					    AIStation.GetCoverageRadius(AIStation.STATION_TRAIN)) + 
			RailBuilder.GetCostOfTownStationWithStationTracks(best_town_pair._town2, town2_tilelist, AITown.GetLocation(best_town_pair._town1), 
						RailManager.PLATFORM_LENGTH, RailManager.NUMBER_OF_TRACKS, ::ai_instance._general_manager.passenger_cargo_id, 
					    AIStation.GetCoverageRadius(AIStation.STATION_TRAIN));
	
	money_reservationID = ::ai_instance._money_manager.ReserveMoney(cost, (1.25*cost).tointeger());
	if(money_reservationID == null) {
		::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
		AILog.Warning("Reserve: " + ::ai_instance._error_manager.GetLastErrorString());
		return false;
	}
	
	//build station for town1
	station_pro = RailBuilder.BuildTownStationWithStationTracks(best_town_pair._town1, town1_tilelist, AITown.GetLocation(best_town_pair._town2), 
						RailManager.PLATFORM_LENGTH, RailManager.NUMBER_OF_TRACKS, ::ai_instance._general_manager.passenger_cargo_id, 
					    AIStation.GetCoverageRadius(AIStation.STATION_TRAIN));
	if(station_pro == null) {
		AILog.Warning("pro station not build");
		if(AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH) {
				::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
				AILog.Warning(::ai_instance._error_manager.GetLastErrorString());
		}
		::ai_instance._money_manager.ReleaseReservation(money_reservationID);
		return false;
	} else {
		AILog.Info("Build " + station_pro.ToString());
	}
	
	//build station for town2
	station_acc = RailBuilder.BuildTownStationWithStationTracks(best_town_pair._town2, town2_tilelist, AITown.GetLocation(best_town_pair._town1), 
					RailManager.PLATFORM_LENGTH, RailManager.NUMBER_OF_TRACKS, ::ai_instance._general_manager.passenger_cargo_id, 
				    AIStation.GetCoverageRadius(AIStation.STATION_TRAIN));
	if(station_acc == null) {
		AILog.Warning("acc station not build");
		RailDemolisher.DemolishStation(station_pro._stationID);
		if(AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH) {
				::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
				AILog.Warning(::ai_instance._error_manager.GetLastErrorString());
		}
		::ai_instance._money_manager.ReleaseReservation(money_reservationID);
		return false;
	} else {
		AILog.Info("Build " + station_acc.ToString());
	}
	//release station money
	::ai_instance._money_manager.ReleaseReservation(money_reservationID);
		
	//create route
	local rail_route = RailManager.BuildRoute(this.route_id, station_pro, station_acc, false, best_town_pair, ::ai_instance._general_manager.passenger_cargo_id, railtype, true); 	
	
	if(rail_route != null) {
		this._routes.push(rail_route);
	} else return false;
	this.PrintRoutes();
	//add trains to start
	if(rail_route.AddNumberOfTrains(2) == 0) {
		::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NO_VEHICLES_IN_ROUTE);
		rail_route._routeOK = false;
		AILog.Info("CreatingTownRoute suceeded no vehicles");
		return false;
	}

	::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NONE);
	AILog.Info("CreatingTownRoute suceeded");
	return true;						  
	
}

function RailManager::CreateNewIndustryRoute() 
{
	this.route_id++;
	this._rail_demolisher._using_routes_ids.push(this.route_id);
	
	local best_industry_pair;
	local iacc_tilelist, ipro_tilelist;
	local station_acc = null;
	local found_existing_station_acc = false;
	local station_pro = null;
	local money_reservationID, cost;
	local blacklist = AIList();	
	local railtype = AIRailTypeList().Begin();
	AIRail.SetCurrentRailType(railtype);
	//find suitable industry pair that has spots for stations
	while(1) {
		best_industry_pair = RailManager.FindSuitableIndustryPair(blacklist);
		if(best_industry_pair == null) {
			return false;
		}
		//find suitables spots for industry pro station
		ipro_tilelist = RailBuilder.GetIndustryStationSpotList(best_industry_pair._industry_pro, 
							        best_industry_pair._industry_acc, RailManager.PLATFORM_LENGTH, RailManager.NUMBER_OF_TRACKS, best_industry_pair._cargo, false, 
							        AIStation.GetCoverageRadius(AIStation.STATION_TRAIN));
		//if it is empty put this industry into blacklist and find another suitable industry pair
		if(ipro_tilelist.IsEmpty()) {
			AILog.Info("ipro tilelist not found");
			blacklist.AddItem(best_industry_pair._industry_pro, 0);
			continue;
		}
		//look if there is route that has industry acc station near industry acc
		for(local i = 0; i < this._routes.len(); i++) {
			if(!this._routes[i]._is_town_route) {
				if(this._routes[i]._station_accepting._industryID == best_industry_pair._industry_acc) {
					if(this._routes[i]._railtype == railtype) {
						AILog.Warning("Route Railtype: " + this._routes[i]._railtype + "New Railtype: " + railtype);
						station_acc = this._routes[i]._station_accepting;
						found_existing_station_acc = true;
						break;
					}
				}
			}
		}
		//look for industry acc station spots only if there is not station nearby
		if(station_acc == null) {
			iacc_tilelist = RailBuilder.GetIndustryStationSpotList(best_industry_pair._industry_acc, 
								  best_industry_pair._industry_pro, RailManager.PLATFORM_LENGTH, 
								  RailManager.NUMBER_OF_TRACKS, best_industry_pair._cargo, true, 
								  AIStation.GetCoverageRadius(AIStation.STATION_TRAIN));
			//if it is empty put this industry into blacklist and find another suitable industry pair
			if(iacc_tilelist.IsEmpty()) {
				AILog.Info("iacc tilelist not found");
				blacklist.AddItem(best_industry_pair._industry_acc, 0);
				continue;
			}
		}
		break; //everything is ok 
	}

	//get approximate cost of both stations
	cost = RailBuilder.GetCostOfIndustryStationWithStationTracks(best_industry_pair._industry_pro, 
						ipro_tilelist, AIIndustry.GetLocation(best_industry_pair._industry_acc), 
						RailManager.PLATFORM_LENGTH, RailManager.NUMBER_OF_TRACKS, best_industry_pair._cargo, 
						false, AIStation.GetCoverageRadius(AIStation.STATION_TRAIN));
	//we only include industry acc station if it does not exist
	if(!found_existing_station_acc && station_acc == null) {
	cost += RailBuilder.GetCostOfIndustryStationWithStationTracks(best_industry_pair._industry_acc, 
						iacc_tilelist, AIIndustry.GetLocation(best_industry_pair._industry_pro), 
						RailManager.PLATFORM_LENGTH,RailManager.NUMBER_OF_TRACKS, best_industry_pair._cargo,
						true, AIStation.GetCoverageRadius(AIStation.STATION_TRAIN));
	}
	
	
	money_reservationID = ::ai_instance._money_manager.ReserveMoney(cost, (1.25*cost).tointeger());
	if(money_reservationID == null) {
		::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
		AILog.Warning("Reserve: " + ::ai_instance._error_manager.GetLastErrorString());
		return false;
	}
	
	//build industry pro station
	station_pro = RailBuilder.BuildIndustryStationWithStationTracks(best_industry_pair._industry_pro, 
						ipro_tilelist, AIIndustry.GetLocation(best_industry_pair._industry_acc), 
						RailManager.PLATFORM_LENGTH, RailManager.NUMBER_OF_TRACKS, best_industry_pair._cargo, 
						false, AIStation.GetCoverageRadius(AIStation.STATION_TRAIN));
	if(station_pro == null) {
		AILog.Warning("pro station not build");
		if(AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH) {
				::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
				AILog.Warning(::ai_instance._error_manager.GetLastErrorString());
		}
		::ai_instance._money_manager.ReleaseReservation(money_reservationID);
		return false;
	} else {
		AILog.Info("Build " + station_pro.ToString());
	}
	
	//build industry acc station only if it does not exist
	if(!found_existing_station_acc && station_acc == null) {
		station_acc = RailBuilder.BuildIndustryStationWithStationTracks(best_industry_pair._industry_acc, 
							iacc_tilelist, AIIndustry.GetLocation(best_industry_pair._industry_pro), 
							RailManager.PLATFORM_LENGTH,RailManager.NUMBER_OF_TRACKS, best_industry_pair._cargo,
							true, AIStation.GetCoverageRadius(AIStation.STATION_TRAIN));
		if(station_acc == null) {
			AILog.Warning("acc station not build");
			RailDemolisher.DemolishStation(station_pro._stationID);
			if(AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH) {
					::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
					AILog.Warning(::ai_instance._error_manager.GetLastErrorString());
			}
			::ai_instance._money_manager.ReleaseReservation(money_reservationID);
			return false;
		} else {
			AILog.Info("Build " + station_acc.ToString());
		}
	} else {
		AILog.Info("We have found existing station");
	}
	//release station money
	::ai_instance._money_manager.ReleaseReservation(money_reservationID);
	
	//create route
	local rail_route = RailManager.BuildRoute(this.route_id, station_pro, station_acc, found_existing_station_acc, best_industry_pair, best_industry_pair._cargo, railtype, false); 
	if(rail_route != null) {
		this._routes.push(rail_route);
	} else return false;
	//add trains to start
	if(rail_route.AddNumberOfTrains(2) == 0) {
		::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NO_VEHICLES_IN_ROUTE);
		rail_route._routeOK = false;
		return false;
	}
	this.PrintRoutes();
	::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NONE);
	AILog.Info("CreatingIndustryRoute suceeded");
	return true;						  
}


function RailManager::BuildRoute(route_id, station_pro, station_acc, found_existing_station_acc, suitable_pair, cargo, railtype, is_town_route) 
{
	local cost;
	local money_reservationID;
	if(found_existing_station_acc) {
		RailBuilder.IndexStationTracks(route_id, station_acc._exit_tile_pair, station_acc._exit_direction);
	}
	//pathfinding: find pro-acc path
	local path_pro_acc = RailManager.FindRoute(station_pro, station_acc);
	if(path_pro_acc == null || path_pro_acc == false) {
		AILog.Warning("path_pro_acc not found");
		RailDemolisher.DemolishStation(station_pro._stationID);
		if(!found_existing_station_acc) RailDemolisher.DemolishStation(station_acc._stationID);
		this._rail_demolisher.DemolishRouteTracks(this.route_id);
		this._blacklist.push(suitable_pair);
		//check for failed tires to find route in a row (if true switch to RAILROAD_ROUTE_OPTIMAL_LENGTH_ROUGH)
		//if -1 we already switched optimal railroad route length to rough
		if(this.found_paths_failed_counter != -1) this.found_paths_failed_counter++;
		if(this.found_paths_failed_counter != -1 && this.found_paths_failed_counter == RAILROAD_MAX_FAILED_TRIES_TO_FIND_ROUTE) {
			this.optimal_railroad_route_length = RailManager.RAILROAD_ROUTE_OPTIMAL_LENGTH_ROUGH;
			this.found_paths_failed_counter = -1;
			AILog.Info("Switching to RAILROAD_ROUTE_OPTIMAL_LENGTH_ROUGH");
		}
		return null;
	}
	
	//get approximate cost of both paths
	cost = RailBuilder.GetCostOfPath(route_id, path_pro_acc)*2;  
	money_reservationID = ::ai_instance._money_manager.ReserveMoney(cost, (1.50*cost).tointeger());
	if(money_reservationID == null) {
		if(!found_existing_station_acc) RailDemolisher.DemolishStation(station_acc._stationID);
		RailDemolisher.DemolishStation(station_pro._stationID);
		::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
		AILog.Warning("Reserve: " + ::ai_instance._error_manager.GetLastErrorString());
		return null;
	}
	
	//try to build pro-acc path
	if(RailBuilder.CanBuildPath(route_id, path_pro_acc)) {
		//AILog.Warning("Test build passed");
		if(!RailBuilder.BuildPath(route_id, path_pro_acc)) {
			if(!found_existing_station_acc) RailDemolisher.DemolishStation(station_acc._stationID);
			RailDemolisher.DemolishStation(station_pro._stationID);
			this._rail_demolisher.DemolishRouteTracks(route_id);
			AILog.Warning("path_pro_acc not build");
			if(AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH) {
				::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
				AILog.Warning(ErrorManager.GetLastErrorString())
			}
			::ai_instance._money_manager.ReleaseReservation(money_reservationID);
			return null;
		} 
	} else {
		if(!found_existing_station_acc) RailDemolisher.DemolishStation(station_acc._stationID);
		RailDemolisher.DemolishStation(station_pro._stationID);
		AILog.Warning("path_pro_acc not build");
		if(AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH) {
				::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
				AILog.Warning(ErrorManager.GetLastErrorString());
		}
		::ai_instance._money_manager.ReleaseReservation(money_reservationID);
		return null;
	}
	//signal pro-acc path
	RailBuilder.SignalPath(path_pro_acc);
	
	//pathfinding: find acc-pro path
	local path_acc_pro = this.FindRoute(station_acc, station_pro);
	if(path_acc_pro == null || path_acc_pro == false) {
		AILog.Warning("path_acc_pro not found");
		RailDemolisher.DemolishStation(station_pro._stationID);
		if(!found_existing_station_acc) RailDemolisher.DemolishStation(station_acc._stationID);
		this._rail_demolisher.DemolishRouteTracks(route_id);
		this._blacklist.push(suitable_pair);
		::ai_instance._money_manager.ReleaseReservation(money_reservationID);
		//check for failed tires to find route in a row (if true switch to RAILROAD_ROUTE_OPTIMAL_LENGTH_ROUGH)
		//if -1 we already switched optimal railroad route length to rough
		if(this.found_paths_failed_counter != -1) this.found_paths_failed_counter++;
		if(this.found_paths_failed_counter != -1 && this.found_paths_failed_counter == RAILROAD_MAX_FAILED_TRIES_TO_FIND_ROUTE) {
			this.optimal_railroad_route_length = RailManager.RAILROAD_ROUTE_OPTIMAL_LENGTH_ROUGH;
			this.found_paths_failed_counter = -1;
			AILog.Info("Switching to RAILROAD_ROUTE_OPTIMAL_LENGTH_ROUGH");
		}
		return null;
	}
	
	//try to build acc-pro path
	if(RailBuilder.CanBuildPath(route_id, path_acc_pro)) {
		//AILog.Warning("Test build passed");
		if(!RailBuilder.BuildPath(route_id, path_acc_pro)) {
			if(!found_existing_station_acc) RailDemolisher.DemolishStation(station_acc._stationID);
			RailDemolisher.DemolishStation(station_pro._stationID);
			this._rail_demolisher.DemolishRouteTracks(route_id);
		
			if(AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH) {
				::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
				AILog.Warning(ErrorManager.GetLastErrorString())
			}
			::ai_instance._money_manager.ReleaseReservation(money_reservationID);
			return null;
		} 
	} else {
		if(!found_existing_station_acc) RailDemolisher.DemolishStation(station_acc._stationID);
		RailDemolisher.DemolishStation(station_pro._stationID);
		this._rail_demolisher.DemolishRouteTracks(route_id);
		if(AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH) {
				::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
				AILog.Warning(ErrorManager.GetLastErrorString())
		}
		::ai_instance._money_manager.ReleaseReservation(money_reservationID);
		return null;
	}

	//signal acc-pro path
	RailBuilder.SignalPath(path_acc_pro);
	
	//check signals
	RailBuilder.CheckSignalPath(path_acc_pro); 
	RailBuilder.CheckSignalPath(path_pro_acc); 

	//build depots
	local depot_from = RailBuilder.BuildDepot(route_id, path_pro_acc);
	local depot_to = RailBuilder.BuildDepot(route_id, path_acc_pro);

	if(depot_from == null && depot_to == null) {
		if(!found_existing_station_acc) RailDemolisher.DemolishStation(station_acc._stationID);
		RailDemolisher.DemolishStation(station_pro._stationID);
		this._rail_demolisher.DemolishRouteTracks(route_id); 
		::ai_instance._money_manager.ReleaseReservation(money_reservationID);
		return null;
	}
	::ai_instance._money_manager.ReleaseReservation(money_reservationID);
	//if route is build reset failed tries to find route counter (if -1 we already switched optimal railroad route length to rough)
	if(this.optimal_railroad_route_length != -1) this.found_paths_failed_counter = 0;
	return RailRoute(route_id, station_pro, station_acc, cargo, depot_from, depot_to, railtype, is_town_route)
}
function RailManager::FindSuitableIndustryPair(blacklist)
{
	AILog.Info("Finding industry pair...");
	local cargolist = GeneralManager.GetOrdedCargoList();
	local industry_pro_all_cargo_list = array(0);
	
	//create industry pro list of all available cargo sorted by producing value 
	for(local cargo = cargolist.Begin(); !cargolist.IsEnd(); cargo = cargolist.Next()) {
		//industry list of specific cargo
		local industry_pro_list = AIIndustryList_CargoProducing(cargo);

		industry_pro_list.Valuate(AIIndustry.IsValidIndustry);
		industry_pro_list.KeepValue(1);	
		industry_pro_list.Valuate(AIIndustry.IsBuiltOnWater);
		industry_pro_list.KeepValue(0);
		//remove industries which we could not find suitable spots for
		industry_pro_list.RemoveList(blacklist);
		//if industry list is empty try another cargo
		if(industry_pro_list.IsEmpty()) {
			//AILog.Info("Industry lists are empty");
			continue;
		}
		
		industry_pro_list.Valuate(ProducingIndustryValuator, cargo);
		industry_pro_list.Sort(AIList.SORT_BY_VALUE, false);
		industry_pro_list.RemoveValue(0);
		
		//AILog.Warning("CARGO SPECIFIC INDUSTRIES: " + AICargo.GetCargoLabel(cargo));
		for(local pro = industry_pro_list.Begin(); !industry_pro_list.IsEnd(); pro = industry_pro_list.Next()){
			//create industry pair (with no industry acc, we add this later)
			local industry_pair = IndustryPair(pro, cargo, industry_pro_list.GetValue(pro));
			//AILog.Info(industry_pair.ToString());
			industry_pro_all_cargo_list.push(industry_pair);	
		}	
	}
	
	industry_pro_all_cargo_list.sort();
	/*
	AILog.Warning("ALL INDUSTRIES ");
	foreach(pro in industry_pro_all_cargo_list) {
		AILog.Info("CARGO: " + AICargo.GetCargoLabel(pro._cargo) + " PRO: " + AIIndustry.GetName(pro._industry_pro) + " Value: " + pro._value);
	}
	*/

	//reverse so the best value is at the and (we can push);
	industry_pro_all_cargo_list.reverse();	
	//this is mainly for smaller maps so they can find route too 
	for(local i = 0; this.optimal_railroad_route_length - (i + RailManager.RAILROAD_ROUTE_LENGTH_TOLERANCE) > RailManager.RAILROAD_ROUTE_MINIMUM_LENGTH ; 
		i += RailManager.RAILROAD_ROUTE_LENGTH_TOLERANCE_JUMP) { 
		
		//AILog.Warning("RAILROAD_ROUTE_LENGTH_TOLERANCE_" + (i + RailManager.RAILROAD_ROUTE_LENGTH_TOLERANCE));
		
		//we need to create copy so we dont destroy original data
		local industry_pro_all_cargo_list_copy = array(0)
		foreach(idx, item in industry_pro_all_cargo_list) {
			industry_pro_all_cargo_list_copy.push(industry_pro_all_cargo_list[idx]);
		}
		local industry_acc_list;
		local best_industry_pair;
		do {
		    //AILog.Warning("looping industry_acc_list");
			industry_acc_list = null; //need to reset every loop
			//we used every producing industry in list so we increase tolerance
			if(industry_pro_all_cargo_list_copy.len() == 0) {
				AILog.Warning("pro list empty");
				break; 
			}
			best_industry_pair = industry_pro_all_cargo_list_copy.pop();
			industry_acc_list = AIIndustryList_CargoAccepting(best_industry_pair._cargo);	
			/*
			AILog.Info("PRO: " + AIIndustry.GetName(best_industry_pair._industry_pro));
			for(local acc = industry_acc_list.Begin(); !industry_acc_list.IsEnd(); acc = industry_acc_list.Next()) {
				AILog.Info("Cargo: " + AICargo.GetCargoLabel(best_industry_pair._cargo) + " ACC: " + AIIndustry.GetName(acc));
			}
			*/
			industry_acc_list.Valuate(AIIndustry.IsValidIndustry);
			industry_acc_list.KeepValue(1);	
			industry_acc_list.Valuate(AIIndustry.IsBuiltOnWater);
			industry_acc_list.KeepValue(0);
			//this is general blacklist of routes that failed in process of creating (path not found)
			foreach(item in this._blacklist) {
				if(item.getclass() == IndustryPair) {
					if(item._industry_pro == best_industry_pair._industry_pro) { 
						industry_acc_list.RemoveItem(item._industry_acc);
					}
				}
			}
			//this is blacklist for industries with no suitable spots
			industry_acc_list.RemoveList(blacklist);
			//we do not want overcrowded industries
			industry_acc_list.Valuate(AIIndustry.GetAmountOfStationsAround);
			industry_acc_list.RemoveAboveValue(3);
			//valuate distance difference
			industry_acc_list.Valuate(GetDistanceDiffToleranceValuator, best_industry_pair._industry_pro, 
									  i + RailManager.RAILROAD_ROUTE_LENGTH_TOLERANCE, false);
			industry_acc_list.RemoveValue(-1);
			industry_acc_list.Sort(AIList.SORT_BY_VALUE, true);
			/*
			AILog.Info("AFTERSORT");
			for(local acc = industry_acc_list.Begin(); !industry_acc_list.IsEnd(); acc = industry_acc_list.Next()) {
				AILog.Info("Cargo: " + AICargo.GetCargoLabel(best_industry_pair._cargo) + " ACC: " + AIIndustry.GetName(acc));
			}
			*/
			//if industry acc list is empty we need to try another insutry pair
		} while(industry_acc_list.IsEmpty());
		
		//if incustry acc list is null we used all industry pro and we have to increase tolerance
		if(industry_acc_list == null) {
			AILog.Warning("acc list empty so skip to next round");
			continue;
		}
		
		AILog.Info("Suitable pair found");
	    best_industry_pair.SetIndustryAcc(industry_acc_list.Begin());

		AILog.Info(best_industry_pair.ToString());

		return best_industry_pair;
	}
	
	throw("we did not find any suitable industry pair (should not happen) because it iterates over all industries");
}
function RailManager::FindSuitableTownPairList() 
{
	local town1_list = GeneralManager.GetTownList(false);
	if(town1_list.Count() == 0) return null;
	local town2 = null;
	local town_pair_array = array(0);
	
	//this is mainly for smaller maps so we find routes too
	for(local i = 0; this.optimal_railroad_route_length - (i + RailManager.RAILROAD_ROUTE_LENGTH_TOLERANCE) > RailManager.RAILROAD_ROUTE_MINIMUM_LENGTH; 
		i += RailManager.RAILROAD_ROUTE_LENGTH_TOLERANCE_JUMP) {
		
		//AILog.Info("town1_list count: " + town1_list.Count());
		//for each town find most suitable (distance) other town
		for(local town1 = town1_list.Begin(); !town1_list.IsEnd(); town1 = town1_list.Next()) {		
			local town2_list = GeneralManager.GetTownList(false);
			town2_list.Valuate(AITown.IsValidTown);
			town2_list.KeepValue(1);
			//get distance difference between two towns
			town2_list.Valuate(GetDistanceDiffToleranceValuator, town1, i + RailManager.RAILROAD_ROUTE_LENGTH_TOLERANCE, true);
			//remove towns that are out of range
			town2_list.RemoveValue(-1);
			town2_list.Sort(AIList.SORT_BY_VALUE, true);
			
			//AILog.Info("town2_list count: " + town2_list.Count());
			if(town2_list.Count() == 0) continue; //no suitable towns for town1			
			town2 = town2_list.Begin();		

			//check if this found town pair is not duplicate
			local is_duplicate = false;
			for(local i = 0; i < town_pair_array.len(); i++) {
				if(town_pair_array[i]._town1 == town2 && town_pair_array[i]._town2 == town1) is_duplicate = true;
			}
			
			if(!is_duplicate) {
				local best_town_pair = TownPair(town1, town2);
				town_pair_array.push(best_town_pair);
			}
			town2 = null;
		}
		if(town_pair_array.len() != 0) break; //we found some town pairs, no need to look for more
	}
	//remove town pairs that creating path have not suceeded in the past (path not found)
	foreach(item in this._blacklist) {
		if(item.getclass() == TownPair) {
			foreach(idx, town_pair in town_pair_array)
			if(item._town1 == town_pair._town1 && item._town2 == town_pair._town2) { 
				town_pair_array.remove(idx);
			}
		}
	}
	//sort by population
	town_pair_array.sort();
	return town_pair_array;

}	
function RailManager::FindRoute(station_from, station_to) 
{
	local sources = [];
	local goals = [];
	local ignored = AITileList();
	
	//direction offsets, we put multiplier number in both x, y coordinates. because we use only station direction we get real offset we need
	local to_direction_offset_1 = Direction.GetDirectionOffset(station_to._exit_direction, 1, 1);
	local to_direction_offset_2 = Direction.GetDirectionOffset(station_to._exit_direction, 2, 2);
	local to_direction_offset_3 = Direction.GetDirectionOffset(station_to._exit_direction, 3, 3);
	local to_direction_offset_4 = Direction.GetDirectionOffset(station_to._exit_direction, 4, 4);
	local from_direction_offset_1 = Direction.GetDirectionOffset(station_from._exit_direction, 1, 1);
	local from_direction_offset_2 = Direction.GetDirectionOffset(station_from._exit_direction, 2, 2);
	local from_direction_offset_3 = Direction.GetDirectionOffset(station_from._exit_direction, 3, 3);
	local from_direction_offset_4 = Direction.GetDirectionOffset(station_from._exit_direction, 4, 4);
		
	local from_direction_turn = Direction.GetDirectionTurned(station_from._exit_direction, 6);
	local to_direction_turn = Direction.GetDirectionTurned(station_to._exit_direction, 2);
	local from_direction_turn_offset_1 = Direction.GetDirectionOffset(from_direction_turn, 1, 1);
	local to_direction_turn_offset_1 = Direction.GetDirectionOffset(to_direction_turn, 1, 1);
	ignored.AddRectangle(station_from._exit_tile_pair.right + from_direction_offset_1, station_from._exit_tile_pair.left + from_direction_offset_2);
	sources = [[station_from._exit_tile_pair.right + from_direction_offset_4, station_from._exit_tile_pair.right + from_direction_offset_3]];
	goals = [[station_to._exit_tile_pair.left + to_direction_offset_4, station_to._exit_tile_pair.left + to_direction_offset_3]];
	ignored.AddTile(station_from._exit_tile_pair.left + from_direction_offset_4);
	ignored.AddTile(station_to._exit_tile_pair.right + to_direction_offset_4);
	ignored.AddTile(station_from._exit_tile_pair.left + from_direction_offset_4 + from_direction_turn_offset_1);
	ignored.AddTile(station_to._exit_tile_pair.right + to_direction_offset_4 + to_direction_turn_offset_1);
	ignored.AddRectangle(station_to._exit_tile_pair.right + to_direction_offset_1, station_to._exit_tile_pair.left + to_direction_offset_2);
	
	local pathfinder = RailPathfinder();
	
	pathfinder.InitializePath(sources, goals, Util.ConvertListToArray(ignored));
	
	local startDate = Helper.GetCurrentDateString();
	local path = false;
	local guardian=0;
	local limit = 20; 
	
	//start finding route (one turn takes aproximately little over a month)
	//because sometimes pathfinding takes too long manage other things between turns
	while (path == false) {
		AILog.Info("   Pathfinding ("+guardian+" / " + limit + ") started");
		path = pathfinder.FindPath(2000);
		AILog.Info("   Pathfinding ("+guardian+" / " + limit + ") ended");
		AIController.Sleep(5);
		
		//we need to check inside if aircraft are enabled because we use air manager inside rail manager
		if(!::ai_instance.aircraft_disabled_shown || ::ai_instance.aircraft_max0_shown) {
		//every two turns manage air routes (not rail routes we could remove something we do not want to)
			if(guardian % 2 == 0) {
				//AILog.Info("Managing air routes in pathfinder");
				::ai_instance._air_manager.ManageAirRoutes();
			}
		//every three turns try to build new airport route 
			if(guardian != 0 && guardian % 3 == 0 && ::ai_instance._money_manager.GetAvailableMoney() >= AirManager.MINIMUM_BALANCE_BUILD_AIRPORT + RailManager.MINIMUM_BALANCE_BUILD_RAIL_ROUTE) {
				AILog.Info("Building air route");
				::ai_instance._air_manager.BuildAirportRoute();

			}

		}
		//every turn take care about urgent tasks
		AILog.Info(Helper.GetCurrentDateString() + " --- Urgent Tasks ---");
		::ai_instance._general_manager.ManageUrgentTasks();
		AILog.Info(Helper.GetCurrentDateString() + " --- Urgent Tasks completed ---");
		
		guardian++;
		if(guardian>limit)break;
	}
	
	local stopDate = Helper.GetCurrentDateString();
	if(path) AILog.Info("Path from: " + AIStation.GetName(station_from._stationID) + " to: " + AIStation.GetName(station_to._stationID) + " found");
	AILog.Info("(start: " + startDate + "; stop: " + stopDate + ")");
	return path;
}


function RailManager::RemoveRoute(route_id) 
{ 

	AILog.Info("Removing route " + route_id);
	local route;
	local route_idx;
	//find route by route id
	foreach(idx, item in this._routes) {
		if(item._route_id == route_id) {
			route = item;
			route_idx = idx;
			break;
		}
	}
	//this._rail_demolisher.BuildSignRoute(0);
	//check if station_acc is used by more routes than this one if yes do not demolish this station (for industry only, town do not have that feature)
	local is_station_acc_used_by_more_routes = false;
	local is_depot_pro_used_by_more_routes = false;
	local is_depot_acc_used_by_more_routes = false;
	
	for(local i = 0; i < this._routes.len(); i++) {
		if(!is_station_acc_used_by_more_routes && this._routes[i]._station_accepting._stationID == route._station_accepting._stationID) {
		//check if it is not the same route
			if(this._routes[i]._route_id != route._route_id) {
				is_station_acc_used_by_more_routes = true;
				AILog.Info("ACC Station used by route " + this._routes[i]._route_id);
			}
		}
		if(!is_depot_pro_used_by_more_routes && this._routes[i]._depot_pro == route._depot_pro) {
			if(this._routes[i]._route_id != route._route_id) {
				is_depot_pro_used_by_more_routes = true;
				AILog.Info("Depot PRO used by route " + this._routes[i]._route_id);
			}
		}
		
		if(!is_depot_acc_used_by_more_routes && this._routes[i]._depot_acc == route._depot_acc) {
			if(this._routes[i]._route_id != route._route_id) {
				is_depot_acc_used_by_more_routes = true;
				AILog.Info("Depot ACC used by route " + this._routes[i]._route_id);
			}
		}
	}
	if(!route._routeOK) {
		if(!is_station_acc_used_by_more_routes) RailDemolisher.DemolishStation(route._station_accepting._stationID);
		RailDemolisher.DemolishStation(route._station_producing._stationID);
		if(!is_depot_pro_used_by_more_routes) AITile.DemolishTile(route._depot_pro); 
		if(!is_depot_acc_used_by_more_routes) AITile.DemolishTile(route._depot_acc);
		this._rail_demolisher.DemolishRouteTracks(route_id); 
		this._routes.remove(route_idx);
	}
}

function RailManager::SellTrain(vehicle, sell_reason)
{
	//we have to find from which route this vehicle is
	local i;
	for(i = 0; i < this._routes.len(); i++) {
		if(this._routes[i]._vehicle_list.HasItem(vehicle)) break;
	}
	local veh_name = AIVehicle.GetName(vehicle);

	//AILog.Info("we have found route " + this._routes[i]._route_id + " for " + veh_name);
	// Try to sell the vehicle	
	if (AIVehicle.SellVehicle(vehicle)) {
		AILog.Info("--> Sold " + veh_name + " (id: " + vehicle + ").");
		if(i == this._routes.len()) {
			AILog.Warning("Train "+ veh_name + " is not added in any route");
			return; 
		}
		this._routes[i]._vehicle_list.RemoveItem(vehicle);
		::ai_instance._general_manager._vehicles_to_depot.RemoveItem(vehicle);
		if(sell_reason == 0) {
			this._routes[i].AddNumberOfTrains(1);
		}
	} else AILog.Info("Selling not suceeded " + AIError.GetLastErrorString());
	
}

function RailManager::HandleCrashedTrain(vehicle) 
{
	//we have to find from which route this vehicle is
	local i;
	for(i = 0; i < this._routes.len(); i++) {
		if(this._routes[i]._vehicle_list.HasItem(vehicle)) break;
	}
	//remove vehicle
	this._routes[i]._vehicle_list.RemoveItem(vehicle);
	//add new vehicle
	if(this._routes[i].AddNumberOfTrains(1) != ErrorManager.ERROR_NONE) {
			AILog.Info("Could not build new train");
	}
}

function RailManager::ManageRailRoutes() 
{
	
	AILog.Info(Helper.GetCurrentDateString() + " Managing rail routes");
	//first we check if there is a route without vehicles then if the route is not ok remove the route	
	local routes_len = this._routes.len();
	for(local i = 0; i < routes_len;) {
		if(this._routes[i].GetNumberOfTrains() == 0) this._routes[i]._routeOK = false;
		if(!this._routes[i]._routeOK) {
			this.RemoveRoute(this._routes[i]._route_id);
			routes_len = this._routes.len();
		} else i++;
	}
	//now we adjust number of trains on existing routes
	for(local i = 0; i < this._routes.len(); i++) {
		this._routes[i].AdjustNumberOfTrainsInRoute();
	}
	
	AILog.Info(Helper.GetCurrentDateString() + " Finished managing rail routes");
}

function RailManager::RemoveUnfinishedRoutes()
{
	//if they are some unassigned routes (mostly unfinished because of save/load), remove them.
	AILog.Info("Demolishing unfinished routes if they are any");
	local all_stations_list = AIStationList(AIStation.STATION_TRAIN);
	local all_depots_list = AIDepotList(AITile.TRANSPORT_RAIL);
			
	foreach(route in this._routes) {
		if(all_stations_list.HasItem(route._station_producing._stationID)) all_stations_list.RemoveItem(route._station_producing._stationID);
		if(all_stations_list.HasItem(route._station_accepting._stationID)) all_stations_list.RemoveItem(route._station_accepting._stationID);
		if(all_depots_list.HasItem(route._depot_pro)) all_depots_list.RemoveItem(route._depot_pro);
		if(all_depots_list.HasItem(route._depot_acc)) all_depots_list.RemoveItem(route._depot_acc);
	}
		
	for(local station = all_stations_list.Begin(); !all_stations_list.IsEnd(); station = all_stations_list.Next()) {
		RailDemolisher.DemolishStation(station);
	}
	for(local depot = all_depots_list.Begin(); !all_depots_list.IsEnd(); depot = all_depots_list.Next()) {
		AITile.DemolishTile(depot);
	}
	this._rail_demolisher.DemolishRouteTracks(0);
	AILog.Info("Demolishing unfinished routes finished");

}
function RailManager::Save() 
{
	local routesave = array(0);
	local blacklistsave = array(0);
	for(local i = 0; i < this._routes.len(); i++) {
		routesave.push(this._routes[i].Save());
	}
	for(local i = 0; i < this._blacklist.len(); i++) {
		blacklistsave.push(this._blacklist[i].Save());
	}

	local data = {
		routeid = this.route_id,
		routes = routesave,
		blacklist = blacklistsave,
		optimalrailroadroutelength = optimal_railroad_route_length
	}
	if(this._rail_demolisher != null) data.rawset("raildemolisher", this._rail_demolisher.Save());
	
	return data;

}

function RailManager::Load(data) 
{
	this.route_id = data["routeid"];
	
	if("blacklist" in data) {
		for(local i = 0; i < data.blacklist.len(); i++) {
			if(data.blacklist[i].classname == "IndustryPair") {
				local industry_pair = IndustryPair.Load(data.blacklist[i]);
				this._blacklist.push(industry_pair);
			} else if(data.blacklist[i].classname == "TownPair") {
				local town_pair = TownPair.Load(data.blacklist[i]);
				this._blacklist.push(town_pair);
			}
		}
	}
	
	if("routes" in data) {
		for(local i = 0; i < data.routes.len(); i++) {
			local route = RailRoute.Load(data.routes[i]);
				this._routes.push(route);
		}
	}
	
	if("optimalrailroadroutelength" in data) {
		optimal_railroad_route_length = data["optimalrailroadroutelength"];
	}
	if (data.rawin("raildemolisher")) {
		this._rail_demolisher.Load(data.rawget("raildemolisher"));
	}
}