/**
 * @author Daniela Plachtova
 * @file util.nut
 * @note this is collection of Util functions, most of them are not mine (authors name is included in description)
 */
 
 /**
  * @brief class Util (class created only for info purposes (so we know it is Util function))
  */  
 class Util {}

/**
 * @brief convert decimal number to hexadecimal
 * @param number number to convert
 * @return string of hexadecimal number
 * @author Wormnest (WormAI)
 * @note source: http://forum.iv-multiplayer.com/index.php?topic=914.60
 */
function Util::decToHex(number)
{
	local hexChars = "0123456789ABCDEF";
	local ret = "";
	local quotient = number;
	do
	{
		local remainder = quotient % 16;
		quotient /= 16;
		ret = hexChars[(remainder < 0) ? -remainder : remainder].tochar()+ret;
	}
	while(quotient != 0);
	if(number < 0) return "-"+ret;
	return ret;
}

/**
 * @brief Writes a tile as hexadecimal number
 * @param tile tile to convert
 * @return tile as hexadecimal number
 * @author Wormnest (WormAI)
 */
function Util::WriteTile(tile)
{
	return "0x" + Util.decToHex(tile);
}

/*
 * @brief Add a rectangular area to an AITileList containing tiles that are within /radius/
 *        tiles from the center tile, taking the edges of the map into account.
 * @param list tilelist
 * @param tile base tile
 * @param radius radius
 * @author Rondje
 */  
function Util::SafeAddRectangle(list, tile, radius) 
{
	local x1 = max(0, AIMap.GetTileX(tile) - radius);
	local y1 = max(0, AIMap.GetTileY(tile) - radius);
	
	local x2 = min(AIMap.GetMapSizeX() - 2, AIMap.GetTileX(tile) + radius);
	local y2 = min(AIMap.GetMapSizeY() - 2, AIMap.GetTileY(tile) + radius);
	
	list.AddRectangle(AIMap.GetTileIndex(x1, y1),AIMap.GetTileIndex(x2, y2)); 
}

/**
 * @brief Rough year/month age estimation string where year = 365 days and month = 30 days. 
 * @param AgeInDays age in days
 * @return age string
 * @author Wormnest (WormAI)
 */
function Util::GetAgeString(AgeInDays)
{
	local y = AgeInDays / 365;
	local days = AgeInDays - (y * 365);
	local m = days / 30;
	return y + " years " + m + " months";
}

/**
 * @brief Gets if tile rectangle is flat and buildable
 * @param tile_from tile from
 * @param tile_to tile to
 * @return if tile rectangle is flat and buildable
 */
function Util::IsTileRectagleFlatAndBuildable(tile_from, tile_to)
{
	local list = AITileList();
	list.AddRectangle(tile_from, tile_to);
	if(list.IsEmpty()) {
		//AILog.Warning("tile list is empty");
		return false;
	} 
	list.Valuate(IsTileFlatAndBuildableValuator);
	list.KeepValue(0);

	if(list.Count() == 0) return true;
		else return false;
}

/**
 * @brief Get maximum distance of engine
 * @param engine engine
 * @return maximum distance
 * @author Wormnest (WormAI)
 */
function Util::GetMaximumDistance(engine) 
{
	local max_dist = AIEngine.GetMaximumOrderDistance(engine);
	if (max_dist == 0) {
		/* Unlimited distance. Since we need to be able to keep values above a squared distance
		We set it to a predefined maximum value. Maps can be maximum 2048x2048. Diagonal will
		be more than that. To be safe we compute 10000 * 10000. */
		return 10000 * 10000;
	}
	else {
		return max_dist;
	}
}	

/**
 * @brief Get the slope if a rail is build on certain tile.
 * @return 0 if not sloped,
 * 		   1 if end > start,
 * 	       2 if end < start.
 * @author Thijs Marinussen (AdmiralAI)
 */
function Util::GetSlope(start, middle, end)
{
	local NW = 0; // Set to true if we want to build a rail to / from the north-west
	local NE = 0; // Set to true if we want to build a rail to / from the north-east
	local SW = 0; // Set to true if we want to build a rail to / from the south-west
	local SE = 0; // Set to true if we want to build a rail to / from the south-east

	if (middle - AIMap.GetMapSizeX() == start || middle - AIMap.GetMapSizeX() == end) NW = 1;
	if (middle - 1 == start || middle - 1 == end) NE = 1;
	if (middle + AIMap.GetMapSizeX() == start || middle + AIMap.GetMapSizeX() == end) SE = 1;
	if (middle + 1 == start || middle + 1 == end) SW = 1;

	/* If there is a turn in the current tile, it can't be sloped. */
	if ((NW || SE) && (NE || SW)) return 0;

	local slope = AITile.GetSlope(middle);
	/* A rail on a steep slope is always sloped. */
	if (AITile.IsSteepSlope(slope)) {
		switch (slope) {
			case AITile.SLOPE_STEEP_W: slope = AITile.SLOPE_W; break;
			case AITile.SLOPE_STEEP_S: slope = AITile.SLOPE_S; break;
			case AITile.SLOPE_STEEP_E: slope = AITile.SLOPE_E; break;
			case AITile.SLOPE_STEEP_N: slope = AITile.SLOPE_N; break;
			default: throw("Not reached");
		}
	}

	/* If only one corner is raised, the rail is sloped. */
	switch (slope) {
		case AITile.SLOPE_N: return (end < start) ? 1 : 2;
		case AITile.SLOPE_S: return (end < start) ? 2 : 1;
		case AITile.SLOPE_E:
			if (abs(start - end) == 2) {
				return (end < start) ? 1 : 2;
			} else {
				return (end < start) ? 2 : 1;
			}
		case AITile.SLOPE_W:
			if (abs(start - end) == 2) {
				return (end < start) ? 2 : 1;
			} else {
				return (end < start) ? 1 : 2;
			}
	}

	if (NW && (slope == AITile.SLOPE_NW)) return (end < start) ? 1 : 2;
	if (NW && (slope == AITile.SLOPE_SE)) return (end < start) ? 2 : 1;
	if (NE && (slope == AITile.SLOPE_NE)) return (end < start) ? 1 : 2;
	if (NE && (slope == AITile.SLOPE_SW)) return (end < start) ? 2 : 1;

	return 0;
}

/**
 * @brief Convert list to array
 * @param list list to convert
 * @return converted array
 */
function Util::ConvertListToArray(list)
{
	local array = array(0);
	for(local l = list.Begin(); !list.IsEnd(); l = list.Next()) {
		array.push(l);
	}
	return array;
}