/**
 * Base class to handle town specific action(like fund buikdings or bribe).
 */
class BaseTownActionStrategy extends UniversalAction_AbstractExecutionStrategy
{
/* public */
	/**
	 * BaseTownActionStrategy constructor.
	 */
	constructor()
	{
		::UniversalAction_AbstractExecutionStrategy.constructor();
		this.town_list = AITownList();
	}

	function ConcretizeContext()
	{
		local action_id = this.GetTownActionID();
		local best_rank = APriority.BAD;
		local best_town_id = -1;

		foreach (town_id, dummy in this.town_list) {
			if (!AITown.IsActionAvailable(town_id, action_id)) continue;

			local cost = this.GetActionCost(town_id);

			/*
			 * Cost is 0 when AI failed to perform town action in test mode.
			 * Should prevent bugs when for unknown reason AI will 
			 *  fail to perform some town actions.
			 // - notice that AITown.IsActionAvailable returned true;
			 * AI'll ignore such situations instead of constant attempts to do something
			 */
			if (cost == 0) continue;
			local income = this.GetEstimatedIncome(town_id);
			if (income <= 0) continue;
			local rank = ActionPriority.CalculatePriority(cost, cost, 0, income);
			if (rank > best_rank) {
				best_rank = rank;
				best_town_id = town_id;
			}
		}

		return {priority = best_rank, context = best_town_id};
	}

	function GetActionCost(context)
	{
		local cost = 0;
		local town_id = context;
		if (AITown.IsValidTown(town_id)) {
			local test_mode = AITestMode();
			local accounting_costs = AIAccounting();
			AITown.PerformTownAction(town_id, this.GetTownActionID());
			cost = accounting_costs.GetCosts() + 1000;
		}
		return cost;
	}

	/**
	 * Check if list of target towns is empty.
	 * @return True if list of target towns is empty. False else.
	 */
	function IsStopped()
	{
		return this.town_list.Count() == 0;
	}

	/** Update own list of target towns. */
	function UpdateTownList()
	{
		this.town_list = this.GetNewTargetTownList();
	}

/* protected */
	function Execute(context)
	{
		local town_id = context;
		if (!AITown.IsValidTown(town_id)) {
			this.town_list.Clear();
			return 0;
		}

		local action_id = this.GetTownActionID();
		local result = AITown.PerformTownAction(town_id, action_id);
		if (result == true) CodeUtils.Log(this.GetLogMessage(town_id), 2);

		/* Remove best town from list */
		this.town_list.Valuate(function(t1, t2) { return t1 != t2;}, town_id);
		this.town_list.RemoveValue(0);
		return (result) ? 0 : -1;
	}

	function HandleResult(result, context)
	{
		return result;
	}

	/**
	 * Get new list of target towns.
	 * @return Town list with target town for this strategy.
	 */
	function GetNewTargetTownList();

	/**
	 * Get ID of the town action to handle.
	 */
	function GetTownActionID();

	/**
	 * Get a message to log on success
	 * @param town_id Town ID, where we act.
	 * @return Log string.
	 */
	function GetLogMessage(town_id);

	/**
	 * Get this startegy name.
	 * @return Name, string.
	 */
	function GetStrategyName()
	{
		return "Town action";
	}

	/**
	 * Get profit estimation from this town action.
	 * @return "Month income" from this town action. Since all town actions
	 *  are pretty much useless(in terms of income), this number intended to be
	 *  taken "from the top".
	 */
	function GetEstimatedIncome();

/* private */
	/** List with target towns (towns where this straategy applicable) */
	town_list = null;
}

/**
 * Statue creation process handling.
 */
class StatueBuildStrategy extends BaseTownActionStrategy
{
/* protected */
	function GetNewTargetTownList()
	{
		local t_list = AITownList();

		t_list.Valuate(AITown.IsActionAvailable, AITown.TOWN_ACTION_BUILD_STATUE);
		t_list.RemoveValue(0);

		if (t_list.IsEmpty()) return t_list;

		t_list.Valuate(AITown.GetRating, AICompany.COMPANY_SELF);
		t_list.RemoveValue(AITown.TOWN_RATING_NONE);

		t_list.Valuate(AITown.HasStatue);
		t_list.KeepValue(0);

		t_list.Valuate(AITown.GetPopulation);
		t_list.KeepAboveValue(500);

		return t_list;
	}

	function GetTownActionID()
	{
		return AITown.TOWN_ACTION_BUILD_STATUE;
	}

	function GetStrategyName()
	{
		return "Statue build action";
	}

	function GetEstimatedIncome(town_id)
	{
		local income = 0;
		local town_stations = this.GetTownStations(town_id);
		foreach (id, dummy in town_stations) {
			local list = AIVehicleList_Station(id);

			foreach (v, dummy in list) {
				income += AIVehicle.GetProfitLastYear(v) * 0.1;
			}

			foreach (v, dummy in list) {
				income += AIVehicle.GetProfitThisYear(v) * 0.2;
			}

			/* should be about 10% of last months month income */
			income = income / GameTime.MONTHS_PER_YEAR;
		}
		local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
		if (income != 0) {
			income += 2000;
			if (balance > this.critical_money_amount) {
				income += this.GetActionCost(town_id);
			}
		}
		return income;
	}

	function GetLogMessage(town_id)
	{
		return "My Company Statue built at " + AITown.GetName(town_id) + "!";
	}

/* private */
	/** When exceed we'll force statue construction. */
	critical_money_amount = 1 << 22;

	/**
	 * Get list of AI stations near specified town.
	 * @param town Town id to check.
	 * @return AIStationList object with stations list.
	 */
	function GetTownStations(town)
	{
		local list = AIStationList(AIStation.STATION_ANY);
		local town_location = AITown.GetLocation(town);
		list.Valuate(AIStation.GetDistanceManhattanToTile, town_location);
		list.KeepBelowValue(15);
		return list;
	}
}

/**
 * Town funding handling.
 */
class FundBuildingsStrategy extends BaseTownActionStrategy
{
/* protected */
	function GetNewTargetTownList()
	{
		local t_list = AITownList();

		t_list.Valuate(AITown.IsActionAvailable, AITown.TOWN_ACTION_FUND_BUILDINGS);
		t_list.RemoveValue(0);

		if (t_list.IsEmpty()) return t_list;

		/* Fund only towns with own statues */
		t_list.Valuate(AITown.HasStatue);
		t_list.RemoveValue(0);

		t_list.Valuate(AITown.GetPopulation);
		t_list.KeepBelowValue(7000);

		return t_list; 
	}

	function GetTownActionID()
	{
		return AITown.TOWN_ACTION_FUND_BUILDINGS;
	}

	function GetStrategyName()
	{
		return "Fund buildings action";
	}

	function GetEstimatedIncome(town_id)
	{
		/* Random used to provide funding of all towns in equal measure */
		// Seems random is the simplest way.
		if (!AITown.HasStatue(town_id)) return 0;
		return 5000 + AIBase.RandRange(5000);
	}

	function GetLogMessage(town_id)
	{
		return "New buildings funded [" + AITown.GetName(town_id) + "]";
	}
}

/**
 * Class that handles TOWN_ACTION_BUY_RIGHTS.
 */
class BuyRightsTownAction extends BaseTownActionStrategy
{
/* protected */
	function GetNewTargetTownList()
	{
		local list = AITownList();

		list.Valuate(AITown.IsActionAvailable, AITown.TOWN_ACTION_BUY_RIGHTS);
		list.RemoveValue(0);

		list.Valuate(AITown.HasStatue);
		list.RemoveValue(0);

		return list;
	}

	function GetTownActionID()
	{
		return AITown.TOWN_ACTION_BUY_RIGHTS;
	}

	function GetStrategyName()
	{
		return "Buy rights action";
	}

	function GetEstimatedIncome(town_id)
	{
		if (AITown.GetExclusiveRightsDuration(town_id) > 0) return 0;

		local nodes_repository = NodesRepository.Get();
		local node = nodes_repository.GetNode(NodeTypeID.NT_PAX_TOWN, town_id);
		local other_players_count = node.GetOpponentsCount();
		if (other_players_count == 0) return 0;

		local stations = node.GetAllStations().GetStationsAsArray();
		local income = 0;
		foreach (dummy_id, station in stations) {
			local list = station.GetVehicles();
			foreach (v, dummy in list) {
				income += AIVehicle.GetProfitLastYear(v);
			}
			income = income / GameTime.MONTHS_PER_YEAR;
		}
		income = (income * 0.15 * min(other_players_count, 5)).tointeger();
		local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
		if (balance > this.critical_money_amount) {
			income += this.GetActionCost(town_id);
		}
		return income;
	}

	function GetLogMessage(town_id)
	{
		return "Exclusive transport rights bought [" + AITown.GetName(town_id) + "]";
	}

/* private */
	/** When exceed we'll buy rights more active. */
	critical_money_amount = (1 << 22);
}

/**
 * Class that handles bribing.
 */
class BribeTownStrategy extends BaseTownActionStrategy
{
/* protected */
	function GetNewTargetTownList()
	{
		local t_list = AITownList();
		if (CorporationUtils.month_income.value < 80000) return t_list;

		t_list.Valuate(AITown.IsActionAvailable, AITown.TOWN_ACTION_BRIBE);
		t_list.RemoveValue(0);

		if (t_list.IsEmpty()) return t_list;

		t_list.Valuate(AITown.GetRating, AICompany.COMPANY_SELF);
		t_list.RemoveValue(AITown.TOWN_RATING_NONE);
		t_list.KeepBelowValue(AITown.TOWN_RATING_MEDIOCRE);

		t_list.Valuate(AITown.GetPopulation);
		t_list.KeepAboveValue(1000);

		return t_list;
	}

	function GetTownActionID()
	{
		return AITown.TOWN_ACTION_BRIBE;
	}

	function GetStrategyName()
	{
		return "Bribe action";
	}

	function GetEstimatedIncome(town_id)
	{
		local income = 5000 + AIBase.RandRange(15000);
		local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
		if (balance > this.critical_money_amount) {
			income = this.GetActionCost(town_id);
		}
		return income;
	}

	function GetLogMessage(town_id)
	{
		return "Nothing happening... yeah....";
	}

	/** When exceed we'll bribe more active. */
	critical_money_amount = (1 << 22) + (1 << 21);
}
