/**
 * Class that handles universal stuff handling.
 */
class UniversalAction extends Terron_AbstractAction
{
/* public */
	/**
	 * Creates universal action with polymorphing behaviour to do things.
	 * @param name Desired action name.
	 * @param action_strategy This strategy defines what to do
	 *  and, actually, does it.
	 * @param data_obsolete_time Defines cache re-read frequency.
	 *  e.g. 30 mean that cashed data will be used for 30 days.
	 *  And we cache action_strategy access results.
	 * @param action_obsolete_time Defines how long this action can attempt
	 *  to achieve own goal(goal is result != -2). After exceeding this limit
	 *  the action will be removed from the AI "to-do list".
	 */
	constructor(name, action_strategy, data_obsolete_time, action_obsolete_time)
	{
		::Terron_AbstractAction.constructor();

		this.name = name;
		this.action_obsolete_time = action_obsolete_time;
		this.strategy = action_strategy;
		this.context_info = Property(this.Update.bindenv(this), data_obsolete_time);
		this.context_info.SetName(name + " context info property update");
	}

	function GetName()
	{
		return this.name;
	}

	function GetPriority()
	{
		return this.context_info.Get().priority;
	}

	/**
	 * Triggers this action context update right now.
	 */
	function ForceUpdate()
	{
		this.context_info.ForceUpdate();
	}

	function SelfRemove()
	{
		::Terron_AbstractAction.SelfRemove();
		this.self_remove_sheduled = false;
	}
/* protected */
	function Execute()
	{
		if (!this.self_remove_sheduled) {
			UniversalAction_SelfRemoveTask(this, this.action_obsolete_time);
		}

		local context = this.context_info.Get().context;
		local cost = this.strategy.GetActionCost(context);
		if (!CorporationUtils.GetMoney(cost)) {
			CodeUtils.Log("Not enough funds to start action...", 0);
			return -2;
		}

		return this.strategy.Go(context);
	}

	/**
	 * Update own state and execution context.
	 * @return Instance describing further behaviour and own priority.
	 */
	function Update()
	{
		return this.strategy.ConcretizeContext();
	}

	function HandleResult(result)
	{
		if (result != -2) {
			this.SelfRemove();
			this.ForceUpdate();
		}
		return result;
	}

/* private */
	/** This action's name */
	name = null;

	/** Self obsolete time */
	action_obsolete_time = null;

	/** The strategy defines this action's behaviour */
	strategy = null;

	/** Cached information about "current" execution context */
	context_info = null;

	/** Flag that is used to handle this action replacement. */
	self_remove_sheduled = false;
}
