/**
 * Base class for different production nodes.<p>
 * Node is the structures on game map which can produce/accept cargo.<p>
 * It is obvious at least two node types: towns an industries.<p>
 */
class Node extends Terron_SaveableObject
{
/* public */
	/** This node in-game ID */
	node_id = null;

	/** Type of this node */
	type = null;

	/** ID of the node type */
	type_id = null;

	/** The flag specifying, whether is this node invalid or not */
	is_closed = false;

	/** The flag indicating if this is a town node or not */
	is_town = false;

	/** The flag indicating that this node is built on water(or not) */
	is_built_on_water = false;

	/**
	 * Node constructor.
	 * @param node_id ID unique within this node's type
	 * @param type This node's type.
	 * @param name The node name.
	 * @param location This node location
	 */
	constructor(node_id, type, name, location)
	{
		::Terron_SaveableObject.constructor(null);
		this.node_id = node_id;
		this.name = name;
		this.location = location;
		this.stations = StationsFacade();
		this.type_id = type.type_id;
		this.is_town = (type_id == NodeTypeID.NT_PAX_TOWN || type_id == NodeTypeID.NT_RAW_TOWN);
		this.type = type;
		this.accepting_cargo_list = {};
		this.production_cargo_list = {};
		foreach (c, dummy_value in this.type.consumption) {
			this.accepting_cargo_list[c] <- c;
		}
		foreach (c, dummy_value in this.type.production) {
			this.production_cargo_list[c] <- c;
		}
		this.node_region_id = PlayableRegions.GetMapRegionFromTile(location);

		local n = 1 + DummyCargoVector.Get().max_cargo_id;
		this.current_production_vector = array(n, 0);
		this.opponents_count = Property(this.doGetOpponentsCount.bindenv(this), 3 * GameTime.MONTH);
		this.opponents_count.SetName("Account enemy stations near " + this.name);
		this.free_production = Property(this.doGetFreeProduction.bindenv(this), GameTime.MONTH + 1);
		this.free_production.SetName("Get " + this.name + " production table");
	}

	function GetName()
	{
		return this.name;
	}

	/**
	 * Get the node type.
	 * @return The type of this node.
	 */
	function GetType()
	{
		return this.type;
	}

	/**
	 * Get the node's type ID.
	 * @return The ID of the type of this node.
	 */
	function GetTypeID()
	{
		return this.type_id;
	}

	/**
	 * Get this node acceptance area.
	 * @param radius The radius of the station that will be used.
	 * @param cargo_id Cargo id.
	 * @return list of tiles that will accept cargo for the given node.
	 */
	function AITileList_NodeAccepting(radius, cargo_id);

	/**
	 * Get this node production area.
	 * @param radius The radius of the station you will be using.
	 * @return list of tiles that "are producing cargo from the given node".
	 * @note can use second param - cargo id.
	 */
	function AITileList_NodeProducing(radius, ...);

	/**
	 * Get the list of cargo ID's accepted by this node.
	 * @return Map with cargo ID's (key = ID, and key's value is the same ID).
	 */
	function GetAcceptingCargoList() { return this.accepting_cargo_list;}

	/**
	 * Get the list of cargo ID's produced by this node.
	 * @return Map with cargo ID's (key = ID, and key's value is the same ID).
	 */
	function GetProductionCargoList() { return this.production_cargo_list;}

	/**
	 * Get the number of opponents stations around this node.
	 */
	function GetAmountOfOpponentsStationsAround() { return 0;}

	/** Currently not used */
	function ProductionBuffed(amount)
	{
		if (this.free_production == null) return;
		local p_old = this.free_production.Get();
		this.free_production.ForceUpdate();
		local p = this.free_production.Get();
		foreach (id, dummy in p) {
			p[id] += (max(0, p_old[id] - p[id]) + amount);
		}
	}

	/**
	 * Get "free" node production.
	 * @param cargo_id Cargo id.
	 * @return Amount of cargo that can be transported by AI per month.
	 */
	function GetFreeProduction(cargo_id)
	{
		return this.free_production.Get()[cargo_id];
	}

	/**
	 * Get the node location.
	 */
	function GetLocation() { return this.location;}

	/**
	 * Get the node region.
	 * We split the game map into 512x512 (or not, if map is small) regions.
	 * Each region will receive own ID.
	 * @return Integer number - ID of the node's region.
	 */
	function GetRegion()
	{
		return this.node_region_id;
	}

	/**
	 * Get the node stations.
	 */
	function GetAllStations()
	{
		return this.stations;
	}

	/**
	 * Get the node stations belonging to the specified class.
	 * @param station_class_name Class name of desired stations.
	 * @return Container with stations(if any, otherwise empty container).
	 */
	function GetClassStations(station_class_name)
	{
		return this.stations.GetClassStations(station_class_name);
	}

	/**
	 * Check if this node is built on water.
	 */
	function IsBuiltOnWater()
	{
		return this.is_built_on_water;
	}

	/**
	 * Check if this node represents a town.
	 */
	function IsTown()
	{
		return this.is_town;
	}

	/**
	 * Check if node is accepting specified cargo.
	 * @param c Cargo ID to check.
	 * @return True if node is accepting cargo_id, false else.
	 */
	function IsAccepting(c)
	{
		return c in this.accepting_cargo_list;
	}

	/**
	 * Check if node is producing specified cargo.
	 * @param c Cargo ID to check.
	 * @return True if node is producing cargo_id, false else.
	 */
	function IsProducing(c)
	{
		return c in this.production_cargo_list;
	}

	/**
	 * Check whether this node is valid or not
	 */
	function IsValid()
	{
		return !this.is_closed;
	}

	/**
	 * Get amount of cargo produced by the node last month.
	 * @param c Cargo ID to check.
	 * @return Integer.
	 */
	function GetLastMonthProduction(c);

	/**
	 * Get amount of cargo transported from the node last month.
	 * @param c Cargo ID to check.
	 * @return Integer.
	 */
	function GetLastMonthTransported(c);

	/**
	 * Check if this node is equal to other node.
	 * @param other Node to compare with.
	 * @return True when the nodes are equal, false else.
	 */
	function IsEqual(other)
	{
		return this.type_id == other.type_id && this.location == other.location
			&& this.node_id == other.node_id;
	}

	/**
	 * Get amount of opponents nearby this node.
	 * @return Amount of competitors working with this node.
	 */
	function GetOpponentsCount()
	{
		return this.opponents_count.Get();
	}

/* protected */
	/** List with the cargo IDs accepted by this node */
	accepting_cargo_list = null;

	/** List with the cargo IDs produced by this node */
	production_cargo_list = null;

	/** Node location */
	location = -1;

	/** Node stations */
	</ must_save = true />
	stations = null;

	/**
	 * Get amount of node's production not transported from it last month.
	 * @param c Cargo type, which amount we want to know.
	 * @return Free cargo production.
	 */
	function doGetFreeCargoProduction(c)
	{
		local n = this.opponents_count.Get();

		/* Free production is all production minus all transported */
		local transported = this.GetLastMonthTransported(c);
		local free_production = this.GetLastMonthProduction(c) - transported;

		/*
		 * If we are not only one who is servicing this node =>
		 * => we can bite a little more from opponents
		 */
		if (n > 0) free_production += transported / (n + 1);
		return max(free_production, 0);
	}

/* private */
	/** Container with information about the node's current production. */
	current_production_vector = null;

	/** The name of this node */
	name = null;

	/** Number of competitors working with this node */
	opponents_count = 0;

	/** Node's last month free production */
	free_production = 0;

	/** ID of the map region (defined by node's location) */
	node_region_id = 0;

	/**
	 * Get amount of node production not transported from it last month.
	 * @return Table with pairs [cargo_id, free cargo_id production].
	 */
	function doGetFreeProduction()
	{
		foreach (dummy_id, c in this.production_cargo_list) {
			this.current_production_vector[c] = this.doGetFreeCargoProduction(c);
		}
		return this.current_production_vector;
	}

	/**
	 * Get amount of opponents nearby this node.
	 * @return Amount of competitors working with this node
	 */
	function doGetOpponentsCount()
	{
		local list = AITileList();
		if (this.IsTown()) {
			local c = CorporationUtils.pass_cargo_id.value;
			list = this.AITileList_NodeAccepting(3, c);
		} else {
			TileUtils.AddSimmetricRectangleSafe(list, this.location, 5, 5);
		}
		list.Valuate(AITile.IsStationTile);
		list.RemoveValue(0);
		if (list.IsEmpty()) return 0;

		local tmp_list = {};
		foreach (t, dummy in list) {
			if (AIStation.IsValidStation(AIStation.GetStationID(t))) continue;
			local owner = AITile.GetOwner(t);
			if (!(owner in tmp_list)) tmp_list[owner] <- owner;
		}

		return tmp_list.len();
	}	
}
