/**
 * Container class for all game "nodes".<p>
 * It is needed to keep only one node object(mean no dublicates at the AI scope)
 *  per one game "node", e.g only one TownNode object for one game town.<p>
 * So if one of the AI parts is building station near a node,
 *  other AI parts can know it without any(other) synchronization,
 *  because all of the parts share one node object.
 */
class NodesRepository extends Terron_SaveableObject
{
/* public */
	static function GetClassName()
	{
		return "NodesRepository";
	}

	/**
	 * Get the instance of the NodesRepository singleton.
	 */
	static function Get()
	{
		if (NodesRepository._instance[0] == null) NodesRepository();
		return NodesRepository._instance[0];
	}

	static function Restore(memento)
	{
		local result = NodesRepository();

		result.FillSelf();

		/* Add loaded nodes */
		foreach (dummy_id, node in memento.nodes) {
			result.AddNode(node);
		}
		foreach (dummy_id, node in memento.heap_nodes) {
			result.AddNode(node);
			foreach (dummy_id, secondary_node in node.secondary_nodes) {
				/*
				 * Prevents creation of links from/to these nodes.
				 * When time comes links only to central node will appear.
				 */
				result.RemoveNode(secondary_node.type_id, secondary_node.node_id);
				secondary_node.is_closed = false;
			}
		}

		/* Remove invalid industries(if any), may be needed after long load */
		// (!) i considered to be equal to corresponding node type ID
		/*local types = TransportSchema.Get().node_types;
		foreach (i, dummy in AIIndustryTypeList()) {
			if (!(i in types)) continue;
			foreach (dummy_id, node in result.nodes[i]) {
				if (AIIndustry.IsValidIndustry(node.GetIndustryID())) continue;
				CodeUtils.Log("Found invalid industry!", 2);
				result.RemoveNode(i, node.node_id);
			}
		}*/

		return result;
	}

	function GetMemento()
	{
		local memento = {nodes = [], heap_nodes = []};
		local i = 0;
		foreach (dummy_id, node_type_specific_container in this.nodes) {
			foreach (id, node in node_type_specific_container) {
				/* Save only nodes with stations */
				if (node.GetAllStations().Count() == 0) continue;
				if (node.GetClassName() == HeapNode.GetClassName()) {
					memento.heap_nodes.append(node);
				} else {
					memento.nodes.append(node);
				}
			}
		}
		return memento;
	}

	function GetName()
	{
		return "Nodes Repository";
	}

	/**
	 * Get node with requested parametres from repository(if it there).
	 * @param node_type_id Node type ID
	 * @param node_id Node ID.
	 * @return Node object or null.
	 */
	function GetNode(node_type_id, node_id)
	{
		return node_id in this.nodes[node_type_id] ?
			this.nodes[node_type_id][node_id] : null;
	}

	/**
	 * Insert node into repository.
	 * @param node Node to insert.
	 */
	function AddNode(node)
	{
		this.nodes[node.type_id][node.node_id] <- node;
		local r_id = node.node_region_id;
		if (!(r_id in this.region_grids)) return;

		local grid_cell = this.region_grids[r_id].GetCellObjectsFromTile(node.location);
		local old_nodes = [];
		foreach (dummy_runtime_id, item in grid_cell) {
			if (node.IsEqual(item)) old_nodes.append(item); 
		}
		foreach (dummy_id, item in old_nodes) {
			delete grid_cell[item.GetID()];
		}
		grid_cell[node.GetID()] <- node;
	}

	/**
	 * Remove node from the repository.
	 * @param node_type_id Node to remove type ID.
	 * @param node_type_id Node to remove ID.
	 */
	function RemoveNode(node_type_id, node_id)
	{
		if (!(node_id in this.nodes[node_type_id])) return;
		this.nodes[node_type_id][node_id].is_closed = true;
		local node = delete this.nodes[node_type_id][node_id];

		if (node == null || !(node.node_region_id in this.region_grids)) return;
		this.region_grids[node.node_region_id].RemoveObject(node.GetID(), node.location);
	}

	function FillSelf()
	{
		CodeUtils.Log("Searching new nodes...", 2);

		local tick = AIController.GetTick();
		local types = TransportSchema.Get().node_types;
		/* Get nodes from different AI parts */
		foreach (dummy_id, factory in node_classes_mount_point.factories) { 
			foreach (dummy_node_id, node in factory.CreateNodes(types)) {
				if (!(node.node_id in this.nodes[node.type_id])) {
					this.AddNode(node);
				}
			}
		}

		tick = AIController.GetTick() - tick;
		CodeUtils.Log("... nodes generation complete, ticks: " + tick, 2);

		this.is_filled = true;
	}

	/**
	 * Check whether map scanning was complete or not.
	 */
	function IsFilled()
	{
		return this.is_filled;
	}

/* private */
	/** Single instance of this class */
	static _instance = [null];

	/** Nodes container itself */
	nodes = null;

	/**
	 * Table with pairs [map region id - grid], where
	 *  grid is Terron_ObliqueGrid containing corresponding region's nodes.
	 */
	region_grids = null;

	/**
	 * Shows whether map scanning was complete or not.
	 */
	is_filled = false;

	/**
	 * Creates nodes repository.
	 */
	constructor()
	{
		::Terron_SaveableObject.constructor(null);

		/* Make sure we'll have only one instance of this class */
		if (NodesRepository._instance[0] != null) assert(null);
		NodesRepository._instance[0] = this;

		/* Create separate storage for each node type */
		CodeUtils.Log("Creating nodes repository...", 2);

		local types = TransportSchema.Get().node_types;
		this.nodes = array(NodeTypeID.NT_MAXTYPES);

		this.region_grids = {};
		local regions = PlayableRegions.Get();
		foreach (dummy_id, region_id in regions.map_regions_to_play) {
			local ne_corner = regions.GetRegionBorders(region_id);
			this.region_grids[region_id] <- Terron_ObliqueGrid(
				16, regions.REGION_SIZE, ne_corner.x_min, ne_corner.y_min
			);
		}

		for (local i = 0; i < NodeTypeID.NT_MAXTYPES; i++) {
			local tmp_name = !(i in types) ? "Unused nodes container #" + i :
				"\"" + types[i].type_name + "\" nodes"
			this.nodes[i] = TableContainer.new(tmp_name);
		}

		// works for now, but mb save priority should be raised
		// to save/load this object one of the firsts.
		AISaveEvent.AddListener(this);
	}

/* public */
	/**
	 * Save self.
	 */
	function OnSave(save_array)
	{
		save_array.append(this);
	}
}

Terron_ClassTable.RegisterClass(NodesRepository);
