/**
 * Class that handles given stations as transit nodes.
 */
class TransitNode extends Node
{
/* public */
	static function GetClassName()
	{
		return "TransitNode";
	}

	static function Restore(memento)
	{
		local cargo_allowed = {};
		foreach (dummy_id, c in memento.cargo_ids) {
			cargo_allowed[c] <- c;
		}
		local restored = TransitNode(memento.s_id, memento.transport_types, cargo_allowed);
		restored.stations = memento.stations;
		return restored;
	}

	/** Transit station's ID */
	</ must_save = true />
	s_id = null;

	/** Table with cargo IDs accepted by this node for further transfer. */
	cargo_ids = null;

	/**
	 * Income for delivery of one cargo unit <b>from</b> this node to
	 * somewhere else.
	 */
	cargo_income = 0;

	/**
	 * Table with allowed methods of cargo delivery <b>to</b> the node's station.
	 */
	transport_types_allowed = null;

	/**
	 * Creates a transit node.
	 * @param station_id ID of the transfer station.
	 * @param transit_transport_types_allowed Array with allowed transfer method IDs.
	 * @param cargo_ids Table with cargo IDs that can be can be transfered
	 *  through the node.
	 */
	constructor(station_id, transit_transport_types_allowed, cargo_ids)
	{
		local type = TransportSchema.Get().node_types[NodeTypeID.NT_TRANSIT_DROP];
		local name = AIStation.GetName(station_id);
		local location = AIStation.GetLocation(station_id);  
		::Node.constructor(station_id, type, name, location);

		local is_valid_station = AIStation.IsValidStation(station_id);
		this.s_id = station_id;
		this.transport_types_allowed = {};
		if (is_valid_station) {
			foreach (dummy, type in transit_transport_types_allowed) {
				this.transport_types_allowed[type] <- type ;
			}
		}

		this.accepting_cargo_list.clear();
		if (is_valid_station) {
			foreach (c, dummy in cargo_ids) {
				this.accepting_cargo_list[c] <- c;
			}
		}
		this.cargo_ids = clone this.accepting_cargo_list;

		Terron.UpdateMemento();

		TransitStationClosingEvent.AddListener(this);
		if (is_valid_station) TransitIncomeCalculatedEvent.AddListener(this);
	}

	function GetMemento()
	{
		local memento = ::Node.GetMemento();
		memento.cargo_ids <- CodeUtils.TableToArray(this.cargo_ids);
		memento.transport_types <- CodeUtils.TableToArray(this.transport_types_allowed);
		return memento;
	}

	function AITileList_NodeAccepting(r, c)
	{
		local border = {};
		local ti = AIMap.GetTileIndex;
		local offsets = [ti(0, -1), ti(-1, 0), ti(0, 1), ti(1, 0)];
		local tiles = AITileList_StationType(this.s_id, AIStation.STATION_ANY);
		foreach (t, dummy in tiles) {
			foreach (dummy_id, offset in offsets) {
				local x = t - offset;
				if (!AITile.IsStationTile(x)) border[x] <- 1;
			}
		} 

		local list = AITileList();
		foreach (t, dummy in border) {
			list.AddTile(t);
		}
		return list;
	}

	function AITileList_NodeProducing(r, ...)
	{
		return AITileList();
	}

	/**
	 * Get the host station ID.
	 */
	function GetStationID() 
	{
		return this.s_id;
	}

	function GetLastMonthProduction(c)
	{
		return 0;
	}

	function GetLastMonthTransported(c)
	{
		return 0;
	}

	function GetAmountOfOpponentsStationsAround()
	{
		return 0;
	}

	function GetAdditionalTransitIncome(c)
	{
		return this.cargo_income;
	}

	/**
	 * Close self when host station is closing.
	 * @param station_id ID of the closing transit station.
	 */
	function OnTransitStationClose(station_id)
	{
		if (station_id != this.s_id) return;
		local map = GlobalTransportMap.Get();
		foreach (class_name, dummy in map.structured_transport_nodes) {
			map.RemoveTransportNodeEx(class_name, this.type_id, station_id);
		}
		foreach (dummy_id, station in this.stations.GetStationsAsArray()) {
			this.stations.RemoveStation(station);
			CorporationUtils.CloseSystem(station);
		}
	}

	/**
	 * Update information about income from transfer operations.
	 * @param data Table with station_id and cargo_income fields.
	 */
	function OnTransitIncomeCalculated(data)
	{
		if (data.station_id != this.s_id) return;
		local i = data.cargo_income + 250;
		this.cargo_income = this.cargo_income < i ? i : (this.cargo_income + i) / 2;
	}
}

Terron_ClassTable.RegisterClass(TransitNode);
