/**
 * Class that describes map parts where AI should be active.
 */
class PlayableRegions extends Terron_Settings
{
/* public */
	/**
	 * Size of the maximal game area that AI will see as whole entity.
	 * Bigger maps will be splited into several such areas, regions.
	 * @note Must be 2^n;
	 */
	static REGION_SIZE = 512;
	//static REGION_SIZE = 128;

	static function GetClassName()
	{
		return "PlayableRegions";
	}

	static function Restore(memento)
	{
		local restored = PlayableRegions();
		restored.map_regions_to_play = memento.map_regions_to_play;
		return restored;
	}

	/**
	 * Get ID of the given tile's region.
	 * @param t Tile whose region ID wanted.
	 * @return Region ID of the given tile.
	 */
	static function GetMapRegionFromTile(t)
	{
		local map_size_x = AIMap.GetMapSizeX();
		local region_size = PlayableRegions.REGION_SIZE;

		local x = AIMap.GetTileX(t) / region_size;
		local y = AIMap.GetTileY(t) / region_size;

		local tmp = map_size_x / region_size + min(map_size_x % region_size, 1);
		return y * tmp + x;
	}

	/** Array with region ids */
	</ must_save = true />
	map_regions_to_play = null;

	/**
	 * Get borders of the given region.
	 * @param region_number Number of region. 
	 * @return Table with fields [x_min, y_min], or null if the given number is
	 *  not valid.
	 * @note Regions length is known, => no need to return another corner.
	 */
	function GetRegionBorders(region_number)
	{
		local region_size = PlayableRegions.REGION_SIZE;
		for (local i = 0, y = 0; y < this.map_size_y; y += region_size) {
			for (local x = 0; x < this.map_size_x; x += region_size) {
				if (i == region_number) return {x_min = x, y_min = y};
				i++;
			}
		}
		return null;
	}

/* protected */
	/**
	 * Randomly(if not loading) choose map areas where to act.
	 * @note Not choosed areas will be ignored.
	 */
	constructor()
	{
		::Terron_Settings.constructor();

		this.map_size_x = AIMap.GetMapSizeX();
		this.map_size_y = AIMap.GetMapSizeY();

		if (SaveLoadUtils.IsLoading()) return;

	 	local map_regions_all = [];
	 	local n = this.GetMapRegionsCount();
	 	for (local i = 0; i < n; i++) {
	 		map_regions_all.append(i);
	 	}

		this.map_regions_to_play = [];
		n = min(MagicNumbers.regions_to_play_count, n);
		for (local i = 0; i < n; i++) {
			local index = AIBase.RandRange(map_regions_all.len());
			this.map_regions_to_play.append(map_regions_all[index]);
			map_regions_all.remove(index);
		}
	}

/* private */
	/** Map x size. */
	map_size_x = null;

	/** Map y size. */
	map_size_y = null;

	/**
	 * Get number of map regions.
	 * @param region_size_x Region x size. 
	 * @param region_size_y Region y size.
	 * @return Number of map regions of given size.
	 */
	function GetMapRegionsCount()
	{
		local region_size = PlayableRegions.REGION_SIZE;
		local map_size_x = AIMap.GetMapSizeX();
		local map_size_y = AIMap.GetMapSizeY();

		local morsel_x = map_size_x % region_size;
		local morsel_y = map_size_y % region_size;

		local intact_regions_count_x = (map_size_x - morsel_x) / region_size;
		local intact_regions_count_y = (map_size_y - morsel_y) / region_size;

		if (morsel_x != 0) morsel_x = 1;
		if (morsel_y != 0) morsel_y = 1;

		return (intact_regions_count_x + morsel_x) * (intact_regions_count_y + morsel_y);
	}
}

Terron_ClassTable.RegisterClass(PlayableRegions);
