/**
 * Class that extends simple Node to handle between-nodes links.
 */
class TransportNode extends AbstractTransportSystem
{
/* public */
	/**
	 * Creates a new TransportNode.
	 * @param node Base node that should serve as production/acceptance core.
	 * @param transport_politics Object that should provide control for this
	 *  node's behavior.
	 */
	constructor(node, transport_politics)
	{
		::AbstractTransportSystem.constructor();
		this.name = node.GetName() + " transport node"; // default name
		this.node = node;
		this.links_in = {};
		this.links_out = {};
		this.transport_politics = transport_politics;
	}

	function GetName()
	{
		return this.name;
	}

	/**
	 * Add new link ending up in this transport node.
	 * @param link Link to add.
	 */
	function AddLink_In(link)
	{
		this.links_in[link.from_id] <- link;
		this.SpreadNewLinkFact();
	}

	/**
	 * Remove link ending up in this transport node.
	 * @param link Link to remove.
	 */
	function RemoveLink_In(link)
	{
		this.links_in.rawdelete(link.from_id);
		this.SpreadLostLinkFact();
	}

	/**
	 * Remove link begining in this transport node.
	 * @param link Link to remove.
	 */
	function RemoveLink_Out(link)
	{
		this.links_out.rawdelete(link.to_id);
	}

	/**
	 * ignore
	 */
	function SpreadNewLinkFact()
	{
		if (this.can_be_root == true) return;

		this.ClearCache();

		local global = GlobalTransportMap.Get();
		foreach (dummy_link_id, link in this.links_out) {
			local x = global.GetTransportNodeByID(link.to_id);
			if (link.aux) {
				x.ClearCache();
			} else if (x.can_be_root == false) {
				x.SpreadNewLinkFact();
			}
		}
	}

	/**
	 * ignore
	 */
	function SpreadLostLinkFact()
	{
		if (this.is_weak == true) return;

		this.ClearCache();

		local global = GlobalTransportMap.Get();
		foreach (dummy_link_id, link in this.links_out) {
			local x = global.GetTransportNodeByID(link.to_id);
			if (link.aux) {
				x.ClearCache();
			} else if (x.is_weak == false) {
				x.SpreadLostLinkFact();
			}
		}
	}

	/**
	 * ignore
	 */
	function ClearCache()
	{
		this.can_be_root = null;
		this.is_weak = null;
	}

	/**
	 * Get the base node.
	 * @return Base node for this transport node.
	 */
	function GetNode()
	{
		return this.node;
	}

	function Close()
	{
		foreach (dummy_id, link in (clone this.links_out)) {
			GlobalTransportMap.Link.Terminate(link);
		}
		foreach (dummy_id, link in (clone this.links_in)) {
			GlobalTransportMap.Link.Terminate(link);
		}
		local map = GlobalTransportMap.Get().transport_nodes_map;
		if (this.GetID() in map) delete map[this.GetID()];

		//this.node = null;
		this.name = "CLOSED " + this.name;
		return true;
	}

/* protected */
	/** The name of this transport node */
	name = null;

	/** The base(raw) node for this transport node */
	node = null;

	/**
	 * Map with links leading into this node.
	 * Value - link object representing connection between some source node and this node.
	 * Key - ID of the source node.
	 */
	links_in = null;

	/**
	 * Map with links starting out of this node.
	 * Value - link object representing connection between this node and some endpoint node.
	 * Key - ID of the endpoint node.
	 */
	links_out = null;

	/**
	 * 'Politics' provide/control this node's connection with other nodes.
	 */
	transport_politics = null;

	/**
	 * Can this node be a root for supply chain?
	 */
	can_be_root = null;

	/**
	 * Is this node supplies is too weak to produce anything?
	 */
	is_weak = null;

	/**
	 * Checks whether this node can be root for cargo delivery tree.
	 */
	function CanBeRoot()
	{
		if (this.can_be_root != null) return this.can_be_root;

		local node_type = this.node.GetType();

		if (this.IsWeak()) {
			this.can_be_root = false;
		} else if (node_type.raw_production) {
			this.can_be_root = true;
		} else {
			local global = GlobalTransportMap.Get();
			local c_in = {};
			foreach (dummy_id, link in this.links_in) {
				local supplier = global.GetTransportNodeByID(link.from_id);
				if (link.aux && supplier.IsWeak() == false) {
					foreach (dummy_route_id, route in link.routes) {
						c_in[route.GetCargoID()] <- 1;
					}
				} else if (link.aux == false && supplier.CanBeRoot()) {
					foreach (dummy_route_id, route in link.routes) {
						c_in[route.GetCargoID()] <- 1;
					}
				}
			}
			foreach (dummy_id, link in this.links_out) {
				if (link.one_way_link) continue;
				local supplier = global.GetTransportNodeByID(link.to_id);
				if (link.aux && supplier.IsWeak() == false) {
					foreach (dummy_route_id, route in link.routes) {
						c_in[route.GetCargoID()] <- 1;
					}
				} else if (link.aux == false && supplier.can_be_root) {
					foreach (dummy_route_id, route in link.routes) {
						c_in[route.GetCargoID()] <- 1;
					}
				}
			}
			this.can_be_root = node_type.GetOutput(c_in).has_output;
		}

		return this.can_be_root;
	}

	/**
	 * Checks whether this node is poorly supplied.
	 */
	function IsWeak()
	{
		if (this.is_weak != null) return this.is_weak;

		local node_type = this.node.GetType()
		if (node_type.raw_consumption) {
			if (this.node.IsTown() || node_type.production.len() == 0) {
				this.is_weak = false;
				return false;
			}
		}

		local c_in = {};
		foreach (dummy_link_id, link in this.links_in) {
			foreach (dummy_route_id, route in link.routes) {
				c_in[route.GetCargoID()] <- 1;
			}
		}
		foreach (dummy_link_id, link in this.links_out) {
			if (link.one_way_link) continue;
			foreach (dummy_route_id, route in link.routes) {
				c_in[route.GetCargoID()] <- 1;
			}
		}
		this.is_weak = (node_type.GetOutput(c_in).has_output == false);
		return this.is_weak;
	}

	/**
	 * Make necessary actions when a link in the "out" container changes color.
	 * @param link Link object that just changed color.
	 */
	function OnOutLinkColorChange(link, previous_color) {};

	/**
	 * Make necessary actions when a link in the "in" container changes color.
	 * @param link Link object that just changed color.
	 */
	function OnInLinkColorChange(link, previous_color) {};
}
