/**
 * Base class to centralize transport nodes and routes creation.
 */
class TransportPolitics extends Terron_Object
{
/* public */
	/**
	 * TransportPolitics class constructor.
	 */
	constructor()
	{
		::Terron_Object.constructor(null);
	}

	/**
	 * Create a TransportNode object from the given Node object.
	 * @param node Some node.
	 * @return Transport node built upon the given node.
	 * @note Newly created object will be inserted into the global container.
	 */
	function EnableTransportNode(node)
	{
		local new_transport_node = this.GetTransportNode(node);
		if (new_transport_node != null) {
			local global = GlobalTransportMap.Get();
			local c_name = new_transport_node.GetClassName();
			local old = global.GetTransportNode(c_name, new_transport_node.node);
			if (old != null) return old;

			global.AddTransportNode(new_transport_node);
			this.IntegrateNode(new_transport_node);
		}
		return new_transport_node;
	}

	/**
	 * Get all available routes between the given transport nodes.
	 * @param from_transport_node Desired routes start point.
	 * @param to_transport_node Desired routes end point.
	 * @param cargo_ids Table. Cargo ids(must be keys in table) to transport.
	 * @return Table with pairs routeID - route.
	 */
	function GetRoutes(from_transport_node, to_transport_node, cargo_ids);

/* protected */
	/**
	 * Check if trade allowed between the given transport nodes.
	 * @param from_transport_node Possible routes start point.
	 * @param to_transport_node Possible routes end point.
	 * @return False if connection is not wanted, true otherwise.
	 */
	function CanRoute(from_transport_node, to_transport_node);

	/**
	 * Custom transport node class constructor call should here.
	 * @param node Some node.
	 * @return Custom type transport node, or null trade with given node
	 *  impossible(or not wanted).
	 */
	function GetTransportNode(node);

	/**
	 * Make custom upon the just created transport node.
	 */
	function FinalizeIntegration(transport_node, transport_schema) {};

	/**
	 * Create links between the given transport node and other(previously
	 *  created) transport nodes.
	 * @param transport_node Well, transport node to integrate.
	 * @return None.
	 */
	/*
	 * Testing results for 512x512:
	 *  most of the time(about 2/3) takes GetRoutes function in links creation,
	 *  thus current dumb - O(N^2) - links search can be considered good enough;
	 *  sweep line even in naive realisation greatly(-60% and more) decreases
	 *  amount of CanRoute checks but it practically does not change overall
	 *  performance and thus omitted
	 */
	function IntegrateNode(transport_node)
	{
		local schema = TransportSchema.Get();
		local t_class = transport_node.GetClassName();
		local node_type_id = transport_node.node.type_id;
		local same_class_nodes = GlobalTransportMap.Get().structured_transport_nodes[t_class];
		local LINK = GlobalTransportMap.Link;
		foreach (j, j_type in schema.node_types) {
			local shipments_from = schema.GetShipmentsList(node_type_id, j);
			if (shipments_from.len() > 0) {
				foreach (dummy_id, e in same_class_nodes[j]) {
					if (!this.CanRoute(transport_node, e)) continue;

					if (j == node_type_id) {
						LINK.Create(transport_node, e, this, 0, shipments_from);
						continue;
					}

					local shipments_into = schema.GetShipmentsList(j, node_type_id);
					LINK.Create(transport_node, e, this, 1, shipments_from);
					if (shipments_into.len() > 0) {
						LINK.Create(e, transport_node, this, 1, shipments_into);
					}
				}
			} else {
				local shipments_into = schema.GetShipmentsList(j, node_type_id);
				if (shipments_into.len() == 0) continue;

				if (transport_node.node.type_id == NodeTypeID.NT_TRANSIT_DROP) {
					shipments_into = clone shipments_into;
					foreach (c, dummy in (clone shipments_into)) {
						if (!(c in transport_node.node.cargo_ids)) delete shipments_into[c];
					}
				}
				foreach (dummy_id, e in same_class_nodes[j]) {
					if (this.CanRoute(e, transport_node)) {
						LINK.Create(e, transport_node, this, 1, shipments_into);
					}
				}
			}
		}
		this.FinalizeIntegration(transport_node, schema);
	}

}