/**
 * Class that describes AI "route".
 */
class AIAbstractRoute extends Terron_Object
{
/* public */
	/**
	 * AIAbstractRoute class constructor.
	 */
	constructor(name)
	{
		::Terron_Object.constructor(null);

		this.name = name;
		this.trade_routes = TableContainer.new("Trade Routes");
	}

	/**
	 * Get estimated cost of all vehicles needed for route.
	 */
	function GetVehiclesCost();

	/**
	 * Get estimated income from all vehicles needed for route. 
	 */
	function GetEstimatedIncome();

	/**
	 * Check if route is fully upgraded and can't be used more than it is now.
	 */
	function IsCompletelyUpgraded()
	{
		return this.trade_routes.len() > this.GetMaxRouteLevel();
	}

	/**
	 * Check if route is "nefarious".
	 * AI should ignore such routes for a better times
	 * (until route become "normal" again).
	 */
	function IsNefarious() { return this.is_nefarious;}

	/**
	 * Mark this route as "nefarious".
	 * @param time Time to "sleep". Negative values mean infinite sleep time.
	 */
	function MakeNefarious(time)
	{
		if (time < 0) {
			this.is_nefarious = true;
			return;
		}

		local date = AIDate.GetCurrentDate() + time;
		if (date > this.nefarious_end_date) {
			this.nefarious_end_date = date;
			this.is_nefarious = true;
			CancelNefariousStatusTask(this, date);
		}
	}

	/**
	 * Remove "nefarious" status from this route.
	 */
	function UnmakeNefarious()
	{
		this.is_nefarious = false;
	}

	function GetName()
	{
		return this.name;
	}

	/**
	 * Get max route level.
	 */
	function GetMaxRouteLevel() { return 0;}

	/**
	 * Get current route level.
	 */
	function GetCurrentRouteLevel() { return this.trade_routes.len();}

/* protected */
	/** Name of the route */
	name = null;

/* private */
	/** Flag, showing if this route is nefarious route and should not be used */
	is_nefarious = false;

	/**
	 * Date when this route will drop it "nefarious" status and become "normal".
	 * When this is lesser than current game date => route is "normal".
	 */
	nefarious_end_date = 0;

	/** Trade routes having this route as foundation. */
	trade_routes = null;
}

/**
 * Class that handles routes "enabling".
 */
class AIAbstractRoute.CancelNefariousStatusTask extends Terron_AbstractTask
{
/* public */
	/**
	 * Creates a new task.
	 * @param route Some route.
	 * @param nefarious_status_drop_date Date when the route must become normal.
	 */
	constructor(route, nefarious_status_drop_date)
	{
		::Terron_AbstractTask.constructor(null);
		this.next_run_date = nefarious_status_drop_date;
		this.route = route;
		this.SetName("Activate route \"" + route.GetName() + "\"");
		_ai_clock.AddTask(this);
	}

	function Run(current_date)
	{
		if (!this.route.is_nefarious) return;
		this.route.is_nefarious = this.route.nefarious_end_date > current_date;
	}

/* private */
	/** Some route */
	route = null;
}
