/**
 * Class that represents air trade route - two airports(and cargo to transport).
 */
class AirTradeRoute extends TradeRoute
{
/* public */
	/** Special name for ships assigned to the cruise lines. */
	static special_air_name_prefix = "Aircraft ";

	static function GetClassName()
	{
		return "AirTradeRoute";
	}

	constructor(air_route, s_airport, e_airport)
	{
		this.name = s_airport.GetName() + " <-> " + e_airport.GetName();
		::TradeRoute.constructor([s_airport, e_airport]);

		air_route.trade_routes.AddItem(this);

		this.air_route = air_route;
		this.s_airport = s_airport;
		this.e_airport = e_airport;
		this.s_terminal = s_airport.airport_terminal;
		this.e_terminal = e_airport.airport_terminal;

		local t1 = s_airport.location;
		local t2 = e_airport.location;
		local dx = abs(AIMap.GetTileX(t1) - AIMap.GetTileX(t2));
		local dy = abs(AIMap.GetTileY(t1) - AIMap.GetTileY(t2));
		this.length = (min(dx, dy) * 1.41).tointeger() + abs(dx - dy);
		this.length_manhattan = dx + dy;

		local best_engine = this.GetBestEngineToBuy();
		if (!AIEngine.IsBuildable(best_engine)) {
			this.vehicles_required = 0;
		} else {
			this.vehicles_required = air_route.GetEstimatedVehicles();
			local n = 1 + this.GetID() % 2;
			if (this.s_airport.GetVehicles().Count() > n) this.vehicles_required = 0;
			if (this.e_airport.GetVehicles().Count() > n) this.vehicles_required = 0;

			if (s_terminal.GetLoadingVehicles() > 1) this.vehicles_required = 0;
			if (e_terminal.GetLoadingVehicles() > 1) this.vehicles_required = 0;
		}

		Terron_Event.AircraftCrashed.AddListener(this);
		Terron_Event.AircraftDestTooFar.AddListener(this);
		Terron_Event.AircraftUnprofitable.AddListener(this);
		AircraftEchoEvent.AddListener(this);
		UsedPlaneAvailabeForFreeEvent.AddListener(this);

		this.actual_profit = this.air_route.vehicle_profit;

		local s_list = s_airport.GetVehicles();
		foreach (v, dummy in e_airport.GetVehicles()) {
			if (AIVehicle.IsValidVehicle(v) && s_list.HasItem(v)) {
				if (AIVehicle.GetState(v) == AIVehicle.VS_CRASHED) continue;
				local e = AIVehicle.GetEngineType(v);
				local t = AirlineUtils.CalculateAlmostFullTravelTime(e, this.length);
				local f = 1.0 / (t + AIEngine.GetCapacity(e) / 50);

				this.s_terminal.AddVehicle(v, f);
				this.e_terminal.AddVehicle(v, f);
				this.vehicles.AddItem(v, v);
			}
		}

		this.vehicles_update_date = AIDate.GetCurrentDate();
	} 

	function GetCargoID()
	{
		return this.air_route.cargo_id;
	}

	function GetOneVehicleProfit()
	{
		return this.actual_profit;
	}

	function IsSaturated()
	{
		local vehicles_count = this.GetVehicles().Count();
		return (this.vehicles_required - vehicles_count <= 0);
	}

	/**
	 * Handler for aircraft crash situation.
	 */
	function OnAircraftCrash(v)
	{
		if (!this.vehicles.HasItem(v)) return;
		this.vehicles.RemoveItem(v);
		this.s_terminal.RemoveVehicle(v);
		this.e_terminal.RemoveVehicle(v);
	}

	function OnAircraftEcho(info)
	{
		if (!info.response_received) {
			if (this.vehicles.HasItem(info.v)) info.response_received = true;
		}
	}

	function OnAircraftDestTooFar(v)
	{
		this.OnAircraftCrash(v);
	}

	function OnAircraftUnprofitable(v)
	{
		if (!this.vehicles.HasItem(v)) return;
		this.SendSpecifiedVehiclesToSell([v]);
	}

	/**
	 * Try to assign plane to another trade route instead of selling it.
	 * @param info Table with 4 fields:<p>
	 *  old_host - previous plane's assignment route<p>
	 *  new_host - new plane's work route(when not null)<p>
	 *  plane_id - plane's vehicle ID<p>
	 *  engine_id - plane's engine ID<p>
	 */
	function OnFreeUsedPlane(info)
	{
		if (info.new_host != null || !this.IsEnabled()) return;
		if (info.old_host.GetID() == this.GetID()) return;
		if (this.GetBestEngineToBuy() != info.engine_id) return;

		local c = this.GetCargoID();
		local m = AirConstants.low_airport_rating_border;
		if (AIStation.GetCargoRating(this.s_airport.station_id, c) > m) return;
		if (AIStation.GetCargoRating(this.e_airport.station_id, c) > m) return;

		local f = this.GetFrequency();
		local n = min(this.s_airport.GetFreePlanes(f), this.e_airport.GetFreePlanes(f));
		if (n < 1) return;

		info.new_host = this;
		this.AssignPlane(info.plane_id);
		this.vehicles.AddItem(info.plane_id, info.plane_id);
		AIOrder.SkipToOrder(info.plane_id, 2);
	}

/* protected */
	function doBuyVehicle()
	{
		local start_point = (this.vehicles.Count() % 2 == 0) ?
			this.s_airport.location : this.e_airport.location;
		local hangar = AIAirport.GetHangarOfAirport(start_point);
		local id = AIVehicle.BuildVehicle(hangar, this.GetBestEngineToBuy());

		if (AIVehicle.IsValidVehicle(id)) this.AssignPlane(id);

		local e = AIVehicle.GetEngineType(id);
		local roster = PlaneEnginesRoster.Get();
		if (roster.GetMailCapacity(e) == -1) {
			local capacity = AIVehicle.GetCapacity(id, CorporationUtils.mail_cargo_id.value);
			roster.SetMailCapacity(e, capacity);
		}
		return id;
	}

	function doSellVehicle(v)
	{
		local info = {
			engine_id = AIVehicle.GetEngineType(v),
			plane_id = v,
			old_host = this,
			new_host = null
		};

		if (AIVehicle.IsValidVehicle(v)) {
			UsedPlaneAvailabeForFreeEvent.Fire(info);
		}

		if (info.new_host == null) {
			::TradeRoute.doSellVehicle(v);
		} else {
			this.vehicles.RemoveItem(v);
		}
		this.s_terminal.RemoveVehicle(v);
		this.e_terminal.RemoveVehicle(v);
	}

	function Close()
	{
		if (!(::TradeRoute.Close())) return false;
		Terron_Event.AircraftCrashed.RemoveListener(this);
		UsedPlaneAvailabeForFreeEvent.RemoveListener(this);
		this.air_route.trade_routes.RemoveItem(this.GetID());
		return true;
	}

	function doGetEngine()
	{
		local e = this.air_route.doGetEngine();
		if (AIEngine.IsBuildable(e)) {
			local c = this.GetCargoID();
			local t = AirlineUtils.CalculateAlmostFullTravelTime(e, this.length);
			t += AIEngine.GetCapacity(e) / 50;
			local ci = AICargo.GetCargoIncome(c, this.length_manhattan, t.tointeger() / 2);
			if (this.s_airport.IsTransit()) {
				local id = this.s_airport.GetStationID();
				TransitIncomeCalculatedEvent.Fire({station_id = id, cargo_id = c, cargo_income = ci});
			}
			if (this.e_airport.IsTransit()) {
				local id = this.e_airport.GetStationID();
				TransitIncomeCalculatedEvent.Fire({station_id = id, cargo_id = c, cargo_income = ci});
			}
			this.frequency = 1.0 / t;
		} else {
			this.frequency = 1.0; // value doesn't matter, just not null is good
			this.Disable(GameTime.YEAR + GameTime.DAY);
		}
		return e;
	}

	function GetFrequency()
	{
		return this.frequency;
	}

/* private */
	/** Improved value for profit per vehicle */
	actual_profit = 0;

	/** Base air route for this trade route. */
	air_route = null;

	/** First airport. */
	s_airport = null;

	/** First airport's terminal. */
	s_terminal = null;

	/** Second airport. */
	e_airport = null;

	/** Second airport's terminal. */
	e_terminal = null;

	/** Vehicles arrival frequency. */
	frequency = 1.0;

	/** Length. */
	length = null;

	/** Distance manhattan between the airports. */
	length_manhattan = 0;

	/** Amount of cargo transported by one vehicle per month. */
	transport_per_vehicle = 0;

	/** Optimal number of airplanes required by this trade route. */
	vehicles_required = 0;

	/** Date of last vehicle number check. */
	vehicles_update_date = null;

	/**
	 * Assign plane to this trade route.
	 * @param v Vehicle ID.
	 */
	function AssignPlane(v)
	{
		for (local i = AIOrder.GetOrderCount(v) - 1; i >= 0; i--) {
			AIOrder.RemoveOrder(v, i);
		}

		local s = this.s_airport.location;
		local e = this.e_airport.location;

		local load_flag = (this.vehicles.Count() % 2 == 0);

		AIOrder.AppendOrder(v, s, AIOrder.OF_FULL_LOAD_ANY);
		AIOrder.AppendOrder(v, AIAirport.GetHangarOfAirport(s), AIOrder.OF_NONE);
		AIOrder.AppendOrder(v, e, AIOrder.OF_FULL_LOAD_ANY);
		AIOrder.AppendOrder(v, AIAirport.GetHangarOfAirport(e), AIOrder.OF_NONE);
		if (!load_flag) AIOrder.SkipToOrder(v, 2);

		local f = this.GetFrequency();
		this.s_terminal.AddVehicle(v, f);
		this.e_terminal.AddVehicle(v, f);

		AIVehicle.SetName(v, (AirTradeRoute.special_air_name_prefix + v));
	}
}

function AirTradeRoute::EstimateVehiclesNeeds()
{
	local current_date = AIDate.GetCurrentDate();
	local vehicles_count = this.vehicles.Count();
	local dt = current_date - this.vehicles_update_date;
	local n1 = max(this.vehicles_required - vehicles_count, 0);

	if (dt < 3 * GameTime.MONTH / 2 && n1 == 0) return 0;

	local e = this.GetBestEngineToBuy();
	if (!AIEngine.IsBuildable(e)) return 0;

	local f = this.GetFrequency();
	local n2 = min(this.s_airport.GetFreePlanes(f), this.e_airport.GetFreePlanes(f));
	if (dt < 3 * GameTime.MONTH / 2) return max(min(n1, n2), 0);

	this.vehicles_update_date = current_date;

	/* Calculate more accurate profit for vehicles attached to this route */
	this.actual_profit = 0;
	vehicles.Valuate(AIVehicle.GetProfitLastYear);
	foreach (v, profit_last_year in vehicles) {
		if (AIVehicle.GetAge(v) < 1 * GameTime.YEAR) continue;
		if (AIVehicle.GetEngineType(v) == e) this.actual_profit += profit_last_year;
	}
	vehicles.Valuate(AIVehicle.GetProfitThisYear);
	foreach (v, profit_this_year in vehicles) {
		if (AIVehicle.GetAge(v) < 1 * GameTime.YEAR) continue;
		if (AIVehicle.GetEngineType(v) == e) this.actual_profit += profit_this_year;
	}
	if (this.actual_profit != 0) {
		local t = 12 + AIDate.GetMonth(current_date);
		this.actual_profit = this.actual_profit / vehicles.Count();
		this.actual_profit = (this.air_route.vehicle_profit + this.actual_profit / t) / 2;
		if (this.actual_profit < 0) this.actual_profit = 0;
		PlaneEnginesRoster.Get().EngineReport(e, this.air_route.vehicle_profit, this.actual_profit);
	} else {
		this.actual_profit = this.air_route.vehicle_profit;
	}

	if (n2 <= -1) return (vehicles_count == 0) ? 0 : -1;

	/* Sell obsolete vehicles */
	if (AIDate.GetYear(current_date) % 2 == this.GetID() % 2) {
		local to_sell = [];
		local invalid = [];
		this.vehicles.Valuate(AIVehicle.GetAge);
		local money = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
		money += (AICompany.GetMaxLoanAmount() - AICompany.GetLoanAmount());
		money += CorporationUtils.month_income.value / 2;
		foreach (v, age in this.vehicles) {
			if ((!AIVehicle.IsValidVehicle(v))
			|| AIVehicle.GetName(v) != (AirTradeRoute.special_air_name_prefix + v)) {
				invalid.append(v);
			} else if (money > AIEngine.GetPrice(e) && (age >= 12 * GameTime.YEAR
			|| (age >= 7 * GameTime.YEAR && e != AIVehicle.GetEngineType(v)))) {
				to_sell.append(v);
			}
		}

		// strange but happens
		// maybe crashed that was not processed because of some delay
		// or with airports replacement
		foreach (dummy_id, v in invalid) {
			CodeUtils.Log("Invalid vehicle #" + v + " in " + this.GetName(), 2);
			this.vehicles.RemoveItem(v);
			this.s_terminal.RemoveVehicle(v);
			this.e_terminal.RemoveVehicle(v);
		}

		if (to_sell.len() > 0) {
			this.SendSpecifiedVehiclesToSell(to_sell);
			vehicles_count = this.vehicles.Count();
			this.vehicles_required = vehicles_count + 1;
			return 1;
		}
	}

	local buy_limit = 200;
	if (AirSettings.Get().greedy_strategy) {
		local planes_per_airport_limit = 1 + this.GetID() % 2;

		local list = this.s_airport.GetVehicles();
		list.Valuate(AIVehicle.GetVehicleType);
		list.KeepValue(AIVehicle.VT_AIR);
		buy_limit = max(0, planes_per_airport_limit - list.Count());

		list = this.e_airport.GetVehicles();
		list.Valuate(AIVehicle.GetVehicleType);
		list.KeepValue(AIVehicle.VT_AIR);
		buy_limit = min(buy_limit, max(0, planes_per_airport_limit - list.Count()));
	}

	local transport_rate = this.air_route.transport_rate;
	local s_need = this.s_terminal.EstimateVehiclesNeeds(transport_rate);
	local e_need = s_need < 0 ? 0 :
		this.e_terminal.EstimateVehiclesNeeds(transport_rate);

	/* Don't buy more than 3 planes at one time */
	local additional_vehicles = min(min(s_need, e_need), 3);

	if (additional_vehicles < 0) {
		additional_vehicles = -min(-additional_vehicles, vehicles_count);
		if (additional_vehicles < -1) additional_vehicles = -1;
	}

	// second time can be 0 because of 0 vehicles_count
	if (additional_vehicles < 0) {

		local current_month = AIDate.GetMonth(current_date);

		local vehicles = this.GetVehicles();
		vehicles.Valuate(AIVehicle.GetEngineType);

		foreach (v, engine in vehicles) {
			if (AIVehicle.GetAge(v) < 2 * GameTime.YEAR) continue;

			local month_profit = AIVehicle.GetProfitLastYear(v);
			month_profit += AIVehicle.GetProfitThisYear(v);
			month_profit = month_profit / (12 + current_month);

			if (AIEngine.GetPrice(engine) > 5 * month_profit * GameTime.MONTHS_PER_YEAR) {
				this.vehicles_required = vehicles_count - 1;
				this.SendSpecifiedVehiclesToSell([v]);
				return 0;
			}
		}

		local t = 5 * GameTime.MONTH;

		/* Try to increase pass rating if needed in airports */
		local c = this.GetCargoID();

		local r = 5 + AirConstants.low_airport_rating_border;
		local s_id = this.s_airport.GetStationID();
		local adv_success = false;
		if (AIStation.GetCargoWaiting(s_id, c) < 50 && AIStation.GetCargoRating(s_id, c) > r) {
			adv_success = this.air_route.GetStart().TryAdvertiseCampaign(t);
		}

		local e_id = this.e_airport.GetStationID();
		if (AIStation.GetCargoWaiting(e_id, c) < 50 && AIStation.GetCargoRating(e_id, c) > r) {
			adv_success = adv_success || this.air_route.GetEnd().TryAdvertiseCampaign(t);
		}

		if (!adv_success) {
			local s_waiting = AIStation.GetCargoWaiting(s_id, c);
			local e_waiting = AIStation.GetCargoWaiting(e_id, c);
			if (s_waiting < 50 || e_waiting < 50) {
				this.vehicles_required = vehicles_count - 1;
				return -1;
			}
		}
		return 0;
	}

	if (n2 < additional_vehicles) additional_vehicles = n2;
	if (buy_limit < additional_vehicles) additional_vehicles = buy_limit;

	this.vehicles_required = vehicles_count + additional_vehicles;
	return additional_vehicles; 
}
