/**
 * Class that handles airports building.
 */
class AirportBuilder
{
/* public */
	/**
	 * Build an airport.
	 * @param node Node where we must build airport.
	 * @param c Cargo to transport. 
	 * @param desired_airport_types Array with airport types.
	 * [0] - best type, function will try to build it first, then [1], e.t.c.
	 * @return New airport station ID or -1 if function fails.
	 */
	static function BuildAirport(node, c, desired_airport_types)
	{
		foreach (dummy_id, type in desired_airport_types) {
			if (!AIAirport.IsValidAirportType(type)) continue;
			AICompany.SetLoanAmount(AICompany.GetMaxLoanAmount());
			local id = AirportBuilder.doBuildAirport(node, c, type);
			if (id == -2 || AIStation.IsValidStation(id)) return id;
		}
		return -1;
	}

	/*static function DebugBuild(town_name)
	{
		foreach (t, dummy in AITownList()) {
			if (AITown.GetName(t) == town_name) {
				local node = TownNode(t, TransportSchema.Get().node_types[NodeTypeID.NT_PAX_TOWN]);
				AirportBuilder.BuildAirport(node, CorporationUtils.pass_cargo_id.value, AirSettings.Get().allowed_airport_types)
			}
		}
		assert(null);
	}*/

/* private */
	/**
	 * Build an airport.
	 * @param node Node where to build airport.
	 * @param cargo_id Cargo to transport.
	 * @param airport_type Type of airport to build. 
	 * @return New airport id or -1 if function fails (-2 if not enough money).
	 */
	static function doBuildAirport(node, cargo_id, airport_type)
	{
		if (airport_type == AIAirport.AT_INVALID) return -1;

		CodeUtils.Log("Trying to build airport near " + node.GetName() + "...", 2);

		local a_x = AIAirport.GetAirportWidth(airport_type);
		local a_y = AIAirport.GetAirportHeight(airport_type); 
		local r = AIAirport.GetAirportCoverageRadius(airport_type);
		local acc = AirConstants.Get().min_acceptance_for_airport;
		local list = TileUtils.GetProductionNodeFreeStationLocations(node, acc, a_x, a_y, r, cargo_id);

		if (node.IsTown()) {
			local pop = node.GetPopulation();
			if (pop > 1000) TileUtils.CutRoads(list, 3);
		}

		list.Valuate(AITile.IsBuildableRectangle, a_x , a_y);
		list.RemoveValue(0);

		if (AirConstants.Get().noise_matters) {
			list.Valuate(AIAirport.GetNoiseLevelIncrease, airport_type);
			list.KeepBelowValue(AITown.GetAllowedNoise(node.town_id) + 1);
		}

		if (list.Count() == 0) {
			CodeUtils.Log("...can't find enough flat land around." , 2);
			return -1;
		} 

		local failures = 0;
		local airport_center = AIMap.GetTileIndex((a_x - 1) / 2, (a_y - 1)/ 2);
		list.Valuate(AIMap.DistanceSquare, node.location - airport_center);
		list.Sort(AIList.SORT_BY_VALUE, true);

		foreach (t, dummy in list) {
			if (!TileUtils.MakeStationFoundation(t, false, a_x, a_y, true)) continue;
			if (!TileUtils.MakeStationFoundation(t, false, a_x, a_y, false)) continue;

			/* Second iteration provides "double terraforming" */
			/* It does nothing when land is good enough for airport,
			 * but when first attempt failed, it will terraform already changed area.
			 * Such approach should handle difference in test and exec modes.
			 * (if test run says ok, while exec mode failes, but do part of work) */
			// still something wrong
			foreach (t2, dummy in list) {
				AITile.PlantTree(t);
				AITile.PlantTree(t2);
				if (!TileUtils.MakeStationFoundation(t2, false, a_x, a_y, true)) continue;
				if (!TileUtils.MakeStationFoundation(t2, false, a_x, a_y, false)) continue;
				if (AIAirport.BuildAirport(t2, airport_type, AIStation.STATION_NEW)) {
					CodeUtils.Log("...built" , 2);
					return AIStation.GetStationID(t2);
				}

				CodeUtils.Log("Failed to build airport. Reason: " + AIError.GetLastErrorString(), 1);
				failures++;
				if (failures > AirConstants.airport_build_attempts_limit) {
					CodeUtils.Log("...failed too many times. Aborting." , 2);
					return -1;
				}

				local last_error = AIError.GetLastError();
				if (last_error == AIError.ERR_LOCAL_AUTHORITY_REFUSES) {
					AITile.PlantTreeRectangle(t, a_x, a_y);
					AITile.PlantTreeRectangle(t, a_x, a_y);
					AITile.PlantTreeRectangle(t, a_x, a_y);
					AITile.PlantTreeRectangle(t2, a_x, a_y);
					AITile.PlantTreeRectangle(t2, a_x, a_y);
					AITile.PlantTreeRectangle(t2, a_x, a_y);
					if (AIAirport.BuildAirport(t2, airport_type, AIStation.STATION_NEW)) {
						CodeUtils.Log("...built" , 2);
						return AIStation.GetStationID(t2);
					}
					CodeUtils.Log("...locals disallow to build!" , 2);
					return -1;
				} else if (last_error == AIStation.ERR_STATION_TOO_MANY_STATIONS_IN_TOWN) {
					CodeUtils.Log("...too many airports in town already. Aborting." , 2);
					return -1;
				}

				if (last_error == AIError.ERR_NOT_ENOUGH_CASH) {
					CodeUtils.Log("...not enough cash" , 2);
					return -2;
				}

				if (last_error == AIError.ERR_STATION_TOO_SPREAD_OUT ) {
					CodeUtils.Log("...too large airport for this world" , 2);
					return -1;
				}
				if (t == t2) break;
			}
		}

		CodeUtils.Log("...failed to build in every tile where was trying" , 2);
		return -1;
	}
}
