/**
 * Town as air hub.
 */
class AirHub extends Hub
{
/* public */
	static function GetClassName()
	{
		return "AirHub";
	}

	/**
	 * Builds airport at the given town.
	 * @param town_node Town - air hub.
	 * @return 0 for success, or -1 if function fails (-2 if not enough money).
	 */
	static function BuildAirport(town_node)
	{
		local c = CorporationUtils.pass_cargo_id.value;
		local airport_types = AirSettings.Get().allowed_airport_types;
		local id = AirportBuilder.BuildAirport(town_node, c, airport_types);
		if (!AIStation.IsValidStation(id)) return id;

		local airport = Airport(id, town_node.node_id, town_node.GetTypeID(), c);
		town_node.GetAllStations().AddStation(airport);
		NewAirportBuiltEvent.Fire(airport);
		return 0;
	}

	constructor(node, transport_politics)
	{
		local k = min(3, this.GetMaxAirlinesForTown(node));
		::Hub.constructor(node, transport_politics, k);
		this.name = node.GetName() + " air hub";

		local a_name = "Building airport at " + node.GetName();
		local work_strategy = AirHub_AirportBuildStrategy(this);
		local t = GameTime.MONTH;
		this.best_action = UniversalAction(a_name, work_strategy, 3 * t, t);

		NewAirportBuiltEvent.AddListener(this);
	}

	/**
	 * Safely removes airport from this hub.
	 */
	function CloseAirport()
	{
		local airport = Airport.GetAirport(this.node);
		if (airport == null) return false;

		CodeUtils.Log("Must close: " + airport.GetName(), 2);
		this.k = min(3, this.GetMaxAirlinesForTown(node));
		this.has_airport = false;

		CorporationUtils.CloseSystem(airport);

		foreach (dummy_id, link in this.links_in) {
			foreach (dummy_route_id, route in link.routes) {
				foreach (dummy_trade_id, trade in route.trade_routes) {
					trade.Close();
				}
			}
		}
		foreach (dummy_id, link in this.links_out) {
			foreach (dummy_route_id, route in link.routes) {
				foreach (dummy_trade_id, trade in route.trade_routes) {
					trade.Close();
				}
			}
		}

		return true;
	}

	/**
	 * Checks whether airport has already been built here.
	 * @return True if and only if the town already has airport.
	 */
	function HasAirport()
	{
		return (this.node.GetClassStations("Airport").len() != 0);
	}

/* protected */
	function GetRating(hub1, hub2, link)
	{
		local result = 0;
		foreach (dummy_route_id, route in link.routes) {
			if (route.IsNefarious()) continue;
			local c = CorporationUtils.pass_cargo_id.value;
			local p1 = hub1.node.GetFreeProduction(c);
			local p2 = hub2.node.GetFreeProduction(c);

			local m = route.length.tofloat() / route.length_manhattan;
			result = min(p1, p2) * route.length * m * m;
			switch (route.GetFreePlanes(16)) {
				case 0 : result = result / 5; break;
				case 1 : result = result / 3; break;
				case 2 : result = result / 2; break;
				case 3 : result = 2 * result / 3; break;
				default : break;
			}
			break;
		}

		if (hub1.has_airport != hub2.has_airport) result = result / 2;
		return result;
	}

	function doGetBestAction()
	{
		return this.best_action;
	}

/* private */
	/** True if and only if the hub already has airport. */
	has_airport = false;

	/** Best action available for this hub. */
	best_action = null;

	/**
	 * Establish trade with neighbour airports.
	 */
	function CreateBestAirTrade()
	{
		foreach (dummy_id, link_info in this.best_links) {
			if (!link_info.hub.has_airport) continue;
			foreach (dummy_route_id, route in link_info.link.routes) {
				if (route.IsCompletelyUpgraded()) continue;
				local s = Airport.GetAirport(route.GetStart());
				local e = Airport.GetAirport(route.GetEnd());
				if (s == null || e == null) continue;
				AirTradeRoute(route, s, e);
			}
		}
	}

	/**
	 * Get max number of airlines allowed for the given town.<p>
	 * This number based on town population.
	 */
	function GetMaxAirlinesForTown(town_node)
	{
		local min_size = AirConstants.lowest_population_limit_for_towns;
		return 2 + town_node.GetPopulation() / min_size;
	}

/* public */
	/**
	 * New airports handling function.
	 * @param airport Newly built airport.
	 */
	function OnNewAirport(airport)
	{
		if (airport.node_id == this.node.node_id) {
			if (!this.has_airport) this.has_airport = true;
		}
		if (!this.has_airport) return;

		this.CreateBestAirTrade();
		if (this.k < 8) {
			this.k = 8;
			this.Reshake();
			this.CreateBestAirTrade();
		}
	}
}

/**
 * Class that handles airports construction.
 */
class AirHub_AirportBuildStrategy extends UniversalAction_AbstractExecutionStrategy
{
/* public */
	/**
	 * Creates AirHub_AirportBuildStrategy object.
	 * @param air_hub Air hub to handle.
	 */
	constructor(air_hub)
	{
		::UniversalAction_AbstractExecutionStrategy.constructor();
		this.air_hub = air_hub;
	}

	function ConcretizeContext()
	{
		if (this.air_hub.node.IsTown() &&
			this.air_hub.node.GetPopulation() < AirConstants.tiny_town_pop) {
				this.air_hub.Disable(15 * GameTime.MONTH);
				return {priority = APriority.BAD, context = null};
		}

		local best_route = null;
		local best_route_rank = APriority.BAD;
		foreach (dummy_id, link_info in this.air_hub.best_links) {
			if (link_info.rating <= 0) continue;
			foreach (dummy_route_id, route in link_info.link.routes) {
				local r = this.RankAirRoute(route);
				if (r > best_route_rank) {
					best_route = route;
					best_route_rank = r;
				}
			}
		}

		return {priority = best_route_rank, context = best_route};
	}

	function GetActionCost(air_route)
	{
		local plane_price = air_route.best_engine_price;
		plane_price = max(1, plane_price - CorporationUtils.month_income.value);

		local n = (Airport.GetAirport(air_route.GetStart()) == null) ? 1 : 0;
		if (Airport.GetAirport(air_route.GetEnd()) == null) n++;
		if (n == 0) return 1;

		local cfg = AirSettings.Get();
		local best_airport_type = cfg.GetBestAirportType();
		local factor = AIAirport.GetMaintenanceCostFactor(best_airport_type);
		local maintenance = cfg.airport_base_maintenance_cost * factor * n;
		maintenance = maintenance * CorporationUtils.GetInflation()

		/* at least 2 planes for each new airport */
		local planes_price = 2 * n * air_route.best_engine_price;
		planes_price = planes_price + 6 * maintenance;
		local corp_income = CorporationUtils.month_income.value - maintenance;
		planes_price = max(1, (planes_price - corp_income).tointeger());

		local build_cost = n * cfg.GetAirportCost(best_airport_type);
		return max(1, build_cost.tointeger()) + planes_price;
	}

/* protected */
	function Execute(air_route)
	{
		local s_airport = Airport.GetAirport(air_route.GetStart());
		if (s_airport == null) {
			local result = AirHub.BuildAirport(air_route.GetStart());
			if (result != 0) return result;
		}

		local e_airport = Airport.GetAirport(air_route.GetEnd());
		if (e_airport == null) {
			local result = AirHub.BuildAirport(air_route.GetEnd());
			if (result != 0) return result;
		}

		return 0;
	}

	function HandleResult(result, route)
	{
		if (result == -1) {
			route.MakeNefarious(2 * GameTime.YEAR);
			local town = this.air_hub.node;
			if (town.GetID() == route.GetStart().GetID()) {
				if (Airport.GetAirport(town) == null) {
					// failed to build at this hub
					this.air_hub.Disable(GameTime.YEAR + GameTime.HALF_YEAR);
				}
			} else {
				if (Airport.GetAirport(route.GetStart()) != null) {
					// has airport at other end, but failed to build at this hub
					this.air_hub.Disable(GameTime.YEAR + GameTime.HALF_YEAR);
				}
			}
			return -1;
		}

		route.MakeNefarious(GameTime.HALF_YEAR);
		return 0;
	}

/* private */
	/** Host */
	air_hub = null;

	/**
	 * Returns air route rank.
	 * @param route Air route which rank we want to know.
	 * @return Number indicating air route upgrade desireability.
	 */
	function RankAirRoute(route)
	{
		if (route.IsNefarious() || route.IsCompletelyUpgraded()) return APriority.BAD;

		local n = (Airport.GetAirport(route.GetStart()) == null) ? 1 : 0;
		if (Airport.GetAirport(route.GetEnd()) == null) n++; 

		local cfg = AirSettings.Get();
		local best_airport_type = cfg.GetBestAirportType();
		if (best_airport_type == AIAirport.AT_INVALID) return APriority.BAD;

		local factor = AIAirport.GetMaintenanceCostFactor(best_airport_type);
		local maintenance = cfg.airport_base_maintenance_cost * factor * n;
		maintenance = maintenance * CorporationUtils.GetInflation();

		local build_cost = n * cfg.GetAirportCost(best_airport_type);

		local income = route.GetEstimatedIncome() - maintenance;
		local full_cost = build_cost + route.GetVehiclesCost();

		/* Need at least this number of planes to be profitable*/
		local m = (maintenance / route.vehicle_profit).tointeger() + 1;
		/* And their cost is... */
		local planes_price = m * route.best_engine_price;
		/* But cost can be ignored if company month income is large enough */
		local corp_income = CorporationUtils.month_income.value - maintenance;
		build_cost += max(1, (planes_price - corp_income).tointeger());

		return ActionPriority.CalculatePriority(build_cost, full_cost, 0, income); 
	}

	/**
	 * Create new trade route and add it to the this subsystems.
	 * @param air_route Air route with built airports.
	 */
	function TryNewTradeFromAirRoute(air_route)
	{
		local s = air_route.GetAirport(air_route.GetStart());
		local e = air_route.GetAirport(air_route.GetEnd());
		if (s != null && e != null) {
			local new_trade_route = AirTradeRoute(air_route, s, e);
		}
	}
}

Terron_ClassTable.RegisterClass(AirHub);
