/**
 * Class that handles obsolete airports destruction.
 */
class ReplaceAirportAction extends TriggerAction
{
/* public */
	constructor()
	{
		::TriggerAction.constructor();

		this.popup_task = AIManageVehiclesAction.DisableTask(this);
		this.popup_task.DisableAction(3 * GameTime.MONTH + GameTime.DAY);

		TransportSystemClosedEvent.AddListener(this); 
	}

	function GetName()
	{
		return "Airport destruction";
	}

	function IsReady()
	{
		/* Replace airports one by one, not all at the same time */
		if (this.closing_airport_id != -1) {
			this.popup_task.DisableAction(GameTime.MONTH);
			return false;
		}

		/* Don't close last airports */
		local vehicles = AIVehicleList();
		vehicles.Valuate(AIVehicle.GetVehicleType);
		vehicles.KeepValue(AIVehicle.VT_AIR);
		if (vehicles.Count() < 16) {
			this.popup_task.DisableAction(GameTime.MONTH);
			return false;
		}

		local cfg = AirSettings.Get();
		local best_airport_type = cfg.GetBestAirportType();
		local cost = cfg.GetAirportCost(best_airport_type);
		local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
		if (cost > balance + AICompany.GetMaxLoanAmount() - AICompany.GetLoanAmount()) {
			this.popup_task.DisableAction(GameTime.MONTH);
			return false;
		}

		return true;
	}

	/**
	 * Check if closed system is the airport we wanted.
	 */
	function OnTransportSystemClose(closed_system)
	{
		if (closed_system.GetID() != this.closing_airport_id) return;

		this.closing_airport_id = -1;
		this.last_call_date = -1000;

		local node = closed_system.GetStationNode();
		if (node.GetClassStations(Airport.GetClassName()).len() == 0) {
			AirHub.BuildAirport(node);
		}
	}

/* protected */
	function Execute()
	{
		this.popup_task.DisableAction(3 * GameTime.MONTH + GameTime.DAY);

		local current_date = AIDate.GetCurrentDate();
		this.last_call_date = current_date;

		local cfg = AirSettings.Get();
		local global = GlobalTransportMap.Get().structured_transport_nodes;
		local class_name = Airport.GetClassName();
		local best_airport_type = cfg.GetBestAirportType();

		foreach (dummy_type_id, type_nodes in global[AirHub.GetClassName()]) {
			foreach (dummy_hub_id, air_hub in type_nodes) {
				local airports = air_hub.node.GetClassStations(class_name);
				foreach (dummy_airport_id, airport in airports) {
					local airport_type = airport.GetAirportType();
					/* No need to replace */
					if ((!airport.is_obsoletable) || (!airport.IsEnabled())) continue;
					if (airport_type == best_airport_type) continue;

					/* Don't demolish, airport just built */
					if (current_date < airport.airport_terminal.build_date + 5 * GameTime.YEAR) continue;

					/* Don't demolish, or we can fail with the replacement */
					if (air_hub.node.GetMyRating() <= AITown.TOWN_RATING_MEDIOCRE) continue;

					if (AirConstants.Get().noise_matters) {
						local free_noise = AITown.GetAllowedNoise(air_hub.node.town_id);
						local next_noise = 100;
						local t = airport.GetLocation();
						local n = cfg.allowed_airport_types.len();
						foreach (id, next_type in cfg.allowed_airport_types) {
							if (!AIAirport.IsValidAirportType(next_type)) continue;
							if (id < n - 1 && cfg.allowed_airport_types[id + 1] == airport_type) {
								next_noise = AIAirport.GetNoiseLevelIncrease(t, next_type);
								break;
							}
						}
						local current_noise = AIAirport.GetNoiseLevelIncrease(t, airport_type);
						/* Don't demolish, next airport type is to loud */
						if (next_noise >= free_noise + current_noise) continue;
					}

					this.closing_airport_id = airport.GetID();

					if (air_hub.CloseAirport()) {
						this.closing_airport_id = -1;
						return 0;
					}
				}
			}
		}

		return -1;
	}

/* private */
	/** Id of airport to demolish */
	closing_airport_id = -1;

	/** Date of last action run */
	last_call_date = -1000;

	/** This task should add/remove this action to/from action repository */
	popup_task = null;
}
