/**
 * Hub class.
 * This class is written to speed up best town-to-town route search.<p>
 * Problem with a lot of towns is that number of possible connections between
 *  them grows quickly enough to noticeably slow down the AI. Good news,
 *  however, is that most of this possible connections don't really have
 *  any chance to be, well, actually connected.
 *  For example: when big town X located near two other big towns(Y, Z) and a
 *  dozen of very small villages, than the only rational choice with X for AI
 *  is XY or XZ - dozen of other connections can be safely ignored.
 *  So, objects of this class(hubs) after creation will track several "best"
 *  neighbour hubs, and AI can then forget about others.<p>
 *  It must be said, that ALL neighbours are remembered and "not best" can be
 *  promoted at any time(there is a Reshake function for this).
 */
class Hub extends TransportNode
{
/* public */
	static function GetClassName()
	{
		return "Hub";
	}

	/**
	 * Hub constructor.
	 * @param town_node Base town node.
	 * @param transport_politics Politics to handle town node.
	 * @param max_connections
	 */
	constructor(town_node, transport_politics, max_connections)
	{
		::TransportNode.constructor(town_node, transport_politics);
		this.best_links = {};
		this.k = max_connections;
	}

	function Close()
	{
		foreach (id, link in this.best_links) {
			this.RemoveLink(id);
		}
		local nodes_full = GlobalTransportMap.Get().transport_nodes_map;
		foreach (id, link in this.links_in) {
			local neighbour = nodes_full[link.from_id];
			neighbour.RemoveLink(id);
		}

		return ::TransportNode.Close();
	}

	function CanBeRoot()
	{
		return true;
	}

	/**
	 * Try to find better than current transport option among neighbours.
	 * @return None.
	 */
	function Reshake()
	{
		local nodes_full = GlobalTransportMap.Get().transport_nodes_map;
		foreach (dummy_id, link in this.links_in) {
			this.TryHandshake(nodes_full[link.from_id], link, 2);
			//this.TryHandshake(nodes_full[link.from_id], link, this.k);
		}
	}

/* protected */
	/**
	 * Container with links to currently best neighbour bus hubs.
	 */
	best_links = null;

	/**
	 * Get rating of hubs connection.
	 * @param hub1 First hub.
	 * @param hub2 Second hub.
	 * @param link Link between the hubs.
	 * @return Number. Greater number mean better connection.
	 */
	function GetRating(hub1, hub2, link)
	{
		local c = CorporationUtils.pass_cargo_id.value;
		local p1 = hub1.node.GetFreeProduction(c);
		local p2 = hub2.node.GetFreeProduction(c);
		local d = AIMap.DistanceManhattan(hub1.node.location, hub2.node.location);
		return min(p1, p2) * d;
	}

/* private */
	/**
	 * Table prototype with information about neighbourhoods.<p>
	 */
	static dummy_neighbour_info = {
		/** Neighbour hub. */
		hub = null,
		/** Link to the neighbour hub. */
		link = null,
		/** Rating. More mean better. */
		rating = 0
	};

	/**
	 * Max allowed number of neighbours to connect to.
	 */
	k = 2; // max best_links len

	/**
	 * Try to find better than current transport option among neighbours.
	 * @param n Limit for recurrent calls.
	 */
	function LazyReshake(n)
	{
		local nodes_full = GlobalTransportMap.Get().transport_nodes_map;
		foreach (dummy_id, link in this.links_in) {
			this.TryHandshake(nodes_full[link.from_id], link, n);
		}
	}

	/**
	 * Get the connection with lower rating than given one.
	 * @param new_rating Target rating.
	 * @return First neighbour info table found
	 *  with lower rating than new_rating.
	 */
	function GetLinkToReplace(new_rating)
	{
		foreach (dummy_id, info in this.best_links) {
			if (new_rating >= info.rating) return info;
		}
		return null;
	}

	/**
	 * Safely removes given link from the memorized best links.
	 * @param link ID of the link to remove.
	 * @return None.
	 */
	function RemoveLink(link_id)
	{
		if (!(link_id in this.best_links)) return;
		local info = delete this.best_links[link_id];
		if (link_id in info.hub.best_links) delete info.hub.best_links[link_id];
	}

	/**
	 * Replace one of the "best" connections with new one, if it is even better.
	 * @param another_hub Hub object, "best" neighbour candidate.
	 * @param link Link to another_hub.
	 * @param n Limit for recurrent calls.
	 * @return None.
	 */
	function TryHandshake(another_hub, link, n)
	{
		local link_id = link.GetID();
		if (link_id in this.best_links) return;

		local to_replace_a = null, to_replace_b = null;
		local r = this.GetRating(this, another_hub, link);
		if (this.best_links.len() >= this.k) {
			to_replace_a = this.GetLinkToReplace(r);
			if (to_replace_a == null) return;
		}

		if (another_hub.best_links.len() >= another_hub.k) {
			to_replace_b = another_hub.GetLinkToReplace(r);
			if (to_replace_b == null) return;
		}

		if (to_replace_a != null) this.RemoveLink(to_replace_a.link.GetID());
		if (to_replace_b != null) another_hub.RemoveLink(to_replace_b.link.GetID());
		this.best_links[link_id] <- {hub = another_hub, link = link, rating = r};
		another_hub.best_links[link_id] <- {hub = this, link = link, rating = r};

		if (n > 0) {
			if (to_replace_a != null) to_replace_a.hub.LazyReshake(n - 1);
			if (to_replace_b != null) to_replace_b.hub.LazyReshake(n - 1);
		}
	}
}
