/**
 * Class that fixes damaged roads.
 */
class FixRoadAction extends InstantAction
{
/* public */
	/**
	 * Creates instance to fix specified road trade route.
	 * @param road_trade_route Road trade route to fix.
	 */
	constructor(road_trade_route)
	{
		::InstantAction.constructor();
		this.road_trade_route = road_trade_route;
		road_trade_route.Disable(GameTime.MONTH);
	}

	function GetName()
	{
		return "Fixing road";
	}

/* protected */
	function Execute()
	{
		AILog.Warning("Trying to fix: " + this.road_trade_route.GetName());
		AICompany.SetLoanAmount(AICompany.GetMaxLoanAmount());
		if (AICompany.GetBankBalance(AICompany.COMPANY_SELF) < 10000) return -2;

		local old_road_type = RoadUtils.SetRoadType(AIRoad.ROADTYPE_ROAD)

		local s = this.road_trade_route.s_stop;
		local e = this.road_trade_route.e_stop;
		/* Fix depots if needed */
		if ((!this.RebuildDepots(s)) || (!this.RebuildDepots(e))) this.fails++;

		/* Fix road */
		local s_arr = s.GetEntrances();
		local e_arr = e.GetExits();

		AIRoad.SetCurrentRoadType(old_road_type);

		if (s_arr.len() == 0 || e_arr.len() == 0) return -1;
		local pf = RoadUtils.GetAccuratePF();
		if (pf.AreTileSetsConnected(s_arr, e_arr, AIRoad.ROADTYPE_ROAD)) {
			return this.fails > 0 ? -1  : 0;
		}
		pf = RoadUtils.GetAccuratePF();
		local stations_fixed = pf.ConnectTiles(s_arr[0], e_arr[0], AIRoad.ROADTYPE_ROAD);
		return (stations_fixed != 0 || this.fails > 0) ? -1 : 0;
	}

/* protected */
	function HandleResult(result)
	{
		if (result == 0 || result == -2) return result;

		/* Too many fails => can't fix => close trade */
		this.road_trade_route.road_route.connection.state = ConnectionState.CS_ACCURATE_COST_UNKNOWN;
		CorporationUtils.CloseSystem(this.road_trade_route);
		Terron.UpdateMemento();
		return 0;
	}

/* private */
	/** Failed fix attempts counter */
	fails = 0;

	/** Road trade route to fix */
	road_trade_route = null;

	function RebuildDepots(road_station)
	{
		foreach (dummy, terminal in road_station.station_plan.GetLoadingTerminals()) {
			if (!this.RebuildDepot(terminal.depot)) return false;
		}

		foreach (dummy, terminal in road_station.station_plan.GetCargoDropTerminals()) {
			if (!this.RebuildDepot(terminal.depot)) return false;
		}

		return true;
	}

	/**
	 * Repair corrupted depot(+road when needed) of the given station.
	 * @note Do nothing when all is good.
	 * @param station_depot The depot which needs to be fixed.
	 * @return false if depot was not repaired, true if depot is ok.
	 */
	function RebuildDepot(station_depot)
	{
		/* If depot is good => do nothing */
		if (station_depot.Check()) return true;

		local depot_tile = station_depot.location;
		/* If there is no depot => build depot */
		if (!AIRoad.IsRoadDepotTile(depot_tile)) {
			return (station_depot.Build() == 0);
		}

		/* Check if depot front tile is actually connected with depot */
		local depot_front = AIRoad.GetRoadDepotFrontTile(depot_tile);
		if (!AIRoad.AreRoadTilesConnected(depot_tile, depot_front)) {
			local v_list = this.road_trade_route.GetVehicles();
			foreach (v, dummy_value in v_list) {
				ForceSellRoadVehicleEvent.Fire(v);
			}
			FixRoadAction.DemolishDepotTask(depot_tile);
			if (AIRoad.BuildRoad(depot_tile, depot_front)) {
				AIRoad.BuildRoadDepot(depot_tile, depot_front);
			} else {
				station_depot.Build();
			}
			if (!AIRoad.IsRoadDepotTile(station_depot.location)) return false;
		}

		depot_front = AIRoad.GetRoadDepotFrontTile(station_depot.location);
		/* Check if depot front tile is connected with station */
		local entrances = station_depot.terminal_core.GetEntrances();
		if (entrances.len() == 0) return false;
		local pf = RoadUtils.GetAccuratePF();
		if (pf.AreTileSetsConnected([depot_front], entrances, AIRoad.ROADTYPE_ROAD)) return true;
		return pf.ConnectTiles(depot_front, entrances[0], AIRoad.ROADTYPE_ROAD) == 0;
	}
}

/**
 * Class that handles accurate road depot demolishing.
 */
class FixRoadAction.DemolishDepotTask extends Terron_Task
{
/* public */
	/**
	 * Creates a task to demolish road depot.
	 * @param depot_location Tile ID to demolish.
	 */
	constructor(depot_location)
	{
		this.depot_location = depot_location;
		if (this.Execute()) return;
		local date = AIDate.GetCurrentDate();
		::Terron_Task.constructor(date + 4 * GameTime.DAY, 10 * GameTime.DAY);
	}

/* protected */
	function Execute()
	{
		AITile.DemolishTile(this.depot_location);
		if (AIRoad.IsRoadDepotTile(this.depot_location)) return false;
		this.period = 0;
		return true;
	}

/* private */
	/** Demolition target depot location */
	depot_location = -1;
}
