/**
 * Class that builds road + stations + depots.
 */
class RoadRouteBuilder
{
/* public */
	/**
	 * Builds road infrastructure required for road trade.
	 * @param road_route Road route to upgrade into trade route.
	 * @param pf Preferred road pathfinder for roads construction.
	 * @return Zero if succeed, negative error code else.
	 */
	static function BuildRoute(road_route, pf)
	{
		CodeUtils.Log("Build command requested for " + road_route.GetName(), 1);
		/* Need to know stuff */
		local c = road_route.cargo_id;
		local l = road_route.GetCurrentRouteLevel();
		local s_node = road_route.GetStart();
		local e_node = road_route.GetEnd();

		local result = {err_code = -3, s_station = null, e_station = null};
		/* Nodes became invalid, don't build anything */
		// actual because of cached access:
		//  while data is cleared from general places, it still may be in cache
		//  => we can access something that points to garbage/null
		if (s_node.is_closed || e_node.is_closed) {
			return {err_code = -3, s_station = null, e_station = null};
		}

		/* Find/build start station */
		local s = road_route.GetStartStation(l);
		if (s == null) {
			CodeUtils.Log("...can't find suitable station near " + s_node.GetName(), 1);
			local id = RoadRouteBuilder.BuildStation(s_node, e_node, c, l, true);
			if (id < 0) return {err_code = id, s_station = null, e_station = null};

			foreach (station_id, station in road_route.GetStartStations(l)) {
				if (station_id == id) {
					s = station;
					break;
				}
			}

			if (s == null) return {err_code = -1, s_station = null, e_station = null};
		} else {
			CodeUtils.Log("...good station found: " + s.GetName(), 1);
		}

		/* Find/build end station */
		local e = road_route.GetEndStation(l);
		if (e == null) {
			CodeUtils.Log("...can't find suitable station near " + e_node.GetName(), 1);
			local id = RoadRouteBuilder.BuildStation(e_node, s_node, c, l, !road_route.IsOneWay());
			if (id < 0) return {err_code = id, s_station = null, e_station = null};

			foreach (station_id, station in road_route.GetEndStations(l)) {
				if (station_id == id) {
					e = station;
					break;
				}
			}

			if (e == null) return {err_code = -1, s_station = null, e_station = null};
		} else {
			CodeUtils.Log("...good station found: " + e.GetName(), 1);
		}

		/* Connect stations, if needed */
		local roadmap = RoadDepartment.Get().GetStationsRoadMap();
		if (!roadmap.HasRoad(s.GetStationID(), e.GetStationID())) {
			local tmp = RoadRouteBuilder.ConnectStations(s, e, pf, AIRoad.ROADTYPE_ROAD);
			if (tmp != 0) {
				CodeUtils.Log("Failed to build road, error code: " + tmp, 1);
				return {err_code = tmp, s_station = null, e_station = null};
			} else {
				roadmap.RegisterRoad(s.GetStationID(), e.GetStationID());
				RoadDepartment.Get().UpdateConnectionsRoadMapWhenRoadBuilt(s_node, e_node);
			}
		}

		return (s.CheckDepots() && e.CheckDepots()) ?
			{err_code = 0, s_station = s, e_station = e} :
			{err_code = -1, s_station = null, e_station = null};
	}

	/**
	 * Builds new road station.
	 * @param node Node where to build.
	 * @param opposite_node Another node (need for proper station orientation).
	 * @param c Cargo to handle.
	 * @param level Desired road station level.
	 * @param must_load Flag, defining if new station must accept or supply cargo.
	 */
	static function BuildStation(node, opposite_node, c, level, must_load)
	{
		CodeUtils.Log("Building a station near " + node.GetName() + "...", 1);
		/* Actually, build a station */

		if (node.GetClassName() == HeapNode.GetClassName()) {
			local plan = RandomTransitStationPlan();
			local build_result = plan.GetConstructionPlan().Execute(node, c);
			if (build_result.err_code != 0) {
				CodeUtils.Log("...failed", 1);
				return build_result.err_code;
			}

			local id = build_result.build_result_info.station_id;
			local new_station = plan.MakeRoadStation(build_result.build_result_info, c, node.node_id, node.GetTypeID());
			new_station.station_level = level;
			CodeUtils.Log("Station \"" + AIStation.GetName(id) + "\" placed", 1);

			node.GetAllStations().AddStation(new_station);

			Terron.UpdateMemento();
			return id;
		}

		local id;
		if (node.type_id == NodeTypeID.NT_TRANSIT_DROP) {
			id = c == CorporationUtils.pass_cargo_id.value ?
				TownStationBuilder.BuildStation(node, c) :
				IndustryStationBuilder.BuildStation(node, c, opposite_node.location, must_load);
		} else {
			id = node.IsTown() ?
				TownStationBuilder.BuildStation(node, c) :
				IndustryStationBuilder.BuildStation(node, c, opposite_node.location, must_load);
		}
		if (!AIStation.IsValidStation(id)) {
			local msg = "   ...failed: ";
			if (id == -2) msg += "not enough money";
			if (id == -3) msg += "nowhere to build, or locals are angry";
			CodeUtils.Log(msg, 1);
			return id;
		}

		local station_type = RoadUtils.GetStationType(c);
		local plan = null;
		if (must_load) {
			if (node.type_id == NodeTypeID.NT_PAX_TOWN) {
				plan = FreeStationPlan.MakePaxAndMailStationPlan(id, station_type, c);
			} else {
				plan = FreeStationPlan.MakeLoadStationPlan(id, station_type, c);
			}
		} else {
			local c_list = clone node.accepting_cargo_list;
			/* Freight(e.g. goods) drop in town => disallow pass/mail */
			if (node.IsTown() && !(c in node.production_cargo_list)) {
				foreach (x, dummy in c_list) {
					if (x in node.production_cargo_list) delete c_list[x];
				}
			}
			/* Pass/mail station in town => disallow freight drop */
			if (node.IsTown() && (c in node.production_cargo_list)) {
				foreach (x, dummy in c_list) {
					if (!(x in node.production_cargo_list)) delete c_list[x];
				}
			}
			plan = FreeStationPlan.MakeCargoDropStationPlan(id, station_type, c_list);
		}

		local new_station = RoadStation(id, level, node.node_id, node.GetTypeID(), plan);
		CodeUtils.Log("Station \"" + AIStation.GetName(id) + "\" placed", 1);

		node.GetAllStations().AddStation(new_station);

		Terron.UpdateMemento();
		return id;
	}

	/**
	 * Connects two road stations with road.
	 * @param start_station First road station.
	 * @param end_station Second road station.
	 * @param pf Preferred road pathfinder for roads construction.
	 * @param road_type Road type to build.
	 * @return Zero if succeed, negative error code else.
	 */
	static function ConnectStations(start_station, end_station, pf, road_type)
	{
		local s_enter = start_station.GetEntrances();
		local e_exit = end_station.GetExits();

		return pf.ConnectTileSets(s_enter, e_exit, road_type);
	}
}
