/**
 * Class that handles road stations building inside towns.
 */
class TownStationBuilder
{
/* public */
	/**
	 * Build a new station.
	 * @param node Town node.
	 * @param c Cargo ID the station must handle.
	 * @return ID of new station, or error code if fails:
	 *  -1 Unknown error
	 *  -2 Not enough money
	 *  -3 Impossible to build(local authority refuses, low acceptance, e.t.c)
	 */
	static function BuildStation(node, c)
	{
		local list = TownStationBuilder.GetTilesWhereCanBuild(node, c);

		local old_road_type = RoadUtils.SetRoadType(AIRoad.ROADTYPE_ROAD);
		local is_transit = node.type_id == NodeTypeID.NT_TRANSIT_DROP;
		local id = TownStationBuilder.TryBuild_TownStation(list, c, is_transit);
		RoadUtils.SetRoadType(old_road_type);

		return id;
	}
}

/**
 * Get list with tiles where station can be built.
 * @param node Town node.
 * @param c Cargo ID the station must handle.
 * @return Sorted tile list (first - best).
 */
function TownStationBuilder::GetTilesWhereCanBuild(node, c)
{
	if (node.type_id == NodeTypeID.NT_TRANSIT_DROP) {
		/* Halt construction if company rating is low */
		local town = AIStation.GetNearestTown(node.GetStationID());
		local rating = AITown.GetRating(town, AICompany.COMPANY_SELF);
		if (rating == AITown.TOWN_RATING_NONE) rating = AITown.TOWN_RATING_VERY_GOOD;
		if (rating <= AITown.TOWN_RATING_POOR) return AITileList();

		local list = node.AITileList_NodeAccepting(1, c);
		list.Valuate(TileUtils.GetAmountOfFlatTilesAround);
		list.Sort(AIList.SORT_BY_VALUE, false);
		return list;
	}

	local station_type = RoadUtils.GetStationType(c);
	local r = AIStation.GetCoverageRadius(station_type);
	local shift_1 = r * AIMap.GetTileIndex(-1, -1);
	local shift_2 = r * AIMap.GetTileIndex(1, 1);

	/* This will be the result - list with possible locations for new station */
	/* Further code mainly just sort it in "right" order */
	local list = node.AITileList_NodeAccepting(r, c);
	/* Remove water tiles */
	list.Valuate(AITile.IsWaterTile);
	list.KeepValue(0);

	/* Helper to obtain desireability for each tile in list */
	local tiles = {};
	list.Valuate(AITile.GetCargoAcceptance, c, 1, 1, 0);
	foreach (t, value in list) {
		tiles[t] <- {acceptance = value, penalty = 0, coverage_area = []};
	}

	/* Add "penalty" for tiles near stations */
	list.Valuate(AITile.IsStationTile);

	foreach (t, is_station in list) {
		local penalty = 0;
		if (is_station) {
			penalty = AICompany.IsMine(AITile.GetOwner(t)) ? 0.5 : 0.7;
			if (AIRoad.IsDriveThroughRoadStationTile(t)) penalty += 0.2;
		}

		local tile_info = tiles[t];
		local coverage_area = AITileList();
		coverage_area.AddRectangle(t + shift_1, t + shift_2)

		foreach (x, dummy in coverage_area) {
			if (t == x || !(x in tiles)) continue;
			tile_info.coverage_area.append(x);
			tiles[x].penalty = tiles[x].penalty + penalty - tiles[x].penalty * penalty;
		}
	}

	/* Remove tiles extremely close to own stations */
	local tmp_list_copy = AITileList();
	tmp_list_copy.AddList(list);
	foreach (t1, dummy in tmp_list_copy) {
		if (AITile.IsStationTile(t1) && AICompany.IsMine(AITile.GetOwner(t1))) {
			foreach (t2, dummy_too in tmp_list_copy) {
				if (AIMap.DistanceManhattan(t1, t2) <= 2) list.RemoveTile(t2);
			}
		}
	}

	/* Remove station tiles - can't build new station there */
	list.KeepValue(0);

	/* Remove not clearable tiles and not flat roads */
	list.Valuate(AITile.IsBuildable);
	{
		local test_mode = AITestMode();
		foreach (t, is_buildable in list) {
			if (is_buildable) continue;
			if (AIRoad.IsRoadTile(t) && AITile.GetSlope(t) == AITile.SLOPE_FLAT) continue;
			if (!AITile.DemolishTile(t)) list.RemoveTile(t);
		}
	}

	/* Set "desireability"(more - better) for each tile */
	foreach (tile_id, tile_info in tiles) {
		tile_info.desireability <- 0.0;
		foreach (dummy_id, t in tile_info.coverage_area) {
			local z = tiles[t];
			local penalty = (z.acceptance * z.penalty).tofloat();
			tile_info.desireability += (z.acceptance - penalty);
		}
		tile_info.desireability = tile_info.desireability.tointeger();
	}

	/* Cut "poor" tiles and sort list according to desireability */
	list.Valuate(function(t) : (tiles) {return tiles[t].desireability;});
	list.RemoveBelowValue(8);
	list.Sort(AIList.SORT_BY_VALUE, false);

	return list;
}

/**
 * Actually build a new station.
 * @param tiles TileList with possible station locations
 * @param c Cargo ID the station must handle.
 * @param is_transit Must be true if this station is transit point.
 * @return ID of new station, or error code if fails:
 *  -1 Unknown error
 *  -2 Not enough money
 *  -3 Impossible to build(local authority refuses, low acceptance, e.t.c)
 */
function TownStationBuilder::TryBuild_TownStation(tiles, cargo_id, is_transit)
{
	if (tiles.Count() == 0) return -3;

	local err_auth = AIError.ERR_LOCAL_AUTHORITY_REFUSES;
	local err_cash = AIError.ERR_NOT_ENOUGH_CASH;
	local offsets = [AIMap.GetTileIndex(0, -1), AIMap.GetTileIndex(-1, 0),
						AIMap.GetTileIndex(0, 1), AIMap.GetTileIndex(1, 0)];
	local s_type = is_transit ? AIStation.STATION_JOIN_ADJACENT : AIStation.STATION_NEW;
	local v_type = AIRoad.GetRoadVehicleTypeForCargo(cargo_id);
	local truck_station = (v_type == AIRoad.ROADVEHTYPE_TRUCK);

	foreach (t, dummy in tiles) {
		if (AIRoad.IsRoadTile(t) && AIRoad.GetNeighbourRoadCount(t) > 1) continue;

		local bad_tile = false;
		foreach (dummy_id, offset in offsets) {
			local z = t - offset;
			if (AIRoad.IsRoadStationTile(z)) {
				if (AIRoad.GetRoadStationFrontTile(z) == t) bad_tile = true;
			}
		}
		if (bad_tile) continue;

		foreach (dummy_id, offset in offsets) {
			local exit = t - offset;
			if (!AITile.IsBuildable(exit) && !AIRoad.IsRoadTile(exit)) continue;

			local exit_slope = AITile.GetSlope(exit);
			if (!TileUtils.IsSlopeFlatOrWithThreeCornersRaised(exit_slope)) continue;

			/* Do not build station if it front tile leads to a "dead end" */
			local blocks = 0;
			foreach (dummy_id, o in offsets) {
				local z = exit - o;
				if (AITile.IsStationTile(z)) blocks = 3;
				if  ( (!TileUtils.IsRoad(z)) && (!AITile.IsBuildable(z)) ) blocks++;
			}
			if (blocks >= 3) continue;

			if (AIRoad.IsRoadTile(t) && AICargo.IsFreight(cargo_id)) {
				local enter = t + offset;
				if (!AIRoad.AreRoadTilesConnected(t, exit)) continue;
				if (!AIRoad.AreRoadTilesConnected(t, enter)) continue;
				if (!AIRoad.BuildDriveThroughRoadStation(t, exit, v_type, s_type)) continue;
				return AIStation.GetStationID(t); 
			}

			{
				local test_mode = AITestMode();
				local accounting = AIAccounting();
				AITile.DemolishTile(t);
				if (accounting.GetCosts() > 10000) continue;
			}

			if (!AIRoad.IsRoadTile(t)) {
				if (!AITile.DemolishTile(t)) continue;
			}

			if (!AIRoad.BuildRoad(t, exit)) continue;
			/*if (!AIRoad.BuildDriveThroughRoadStation(t, exit, v_type, s_type)) {
				return AIStation.GetStationID(t)
			}*/
			if (AIRoad.BuildRoadStation(t, exit, v_type, s_type)) return AIStation.GetStationID(t);
			local err = AIError.GetLastError();
			if (err == err_auth || err == err_cash) return err == err_cash ? -2 : -3;
		}
	}
	return -1;
}
