class RoadSiteConstructionPlan extends Terron_Object
{
/* public */
	constructor(construction_commands)
	{
		::Terron_Object.constructor(null);

		this.construction_commands = construction_commands;

		foreach (dummy_id, command in construction_commands) { 
			local x = 1 + (command.code == ConstructionCommand.CC_ROAD_X ?
				command.x + command.extension : command.x);
			if (x > construction_site_x_size) this.construction_site_x_size = x;

			local y = 1 + (command.code == ConstructionCommand.CC_ROAD_Y ?
				command.y + command.extension : command.y);
			if (y > construction_site_y_size) this.construction_site_y_size = y;
		}

		local rnd = AIBase.RandRange(4); 
		this.construction_commands = this.Rotate(construction_commands, rnd);
	}

	function Rotate(commands, angle_k90)
	{
		local transformations = [];
		switch (angle_k90 % 4) {
			case 0 :
				this.dx = AIMap.GetTileIndex(1, 0);
				this.dy = AIMap.GetTileIndex(0, 1);

				this.construction_site_x_len = this.construction_site_x_size;
				this.construction_site_y_len = this.construction_site_y_size;

				return commands;
			case 1 :
				this.dx = AIMap.GetTileIndex(0, -1);
				this.dy = AIMap.GetTileIndex(1, 0);

				this.construction_site_x_len = this.construction_site_y_size;
				this.construction_site_y_len = -this.construction_site_x_size;

				transformations.append(RoadSiteConstructionCommand.ROTATE_COMMAND);
				break;
			case 2 :
				this.dx = AIMap.GetTileIndex(-1, 0);
				this.dy = AIMap.GetTileIndex(0, -1);

				this.construction_site_x_len = -this.construction_site_x_size;
				this.construction_site_y_len = -this.construction_site_y_size;

				transformations.append(RoadSiteConstructionCommand.POINT_REFLECT_COMMAND);
				break;
			case 3 :
				this.dx = AIMap.GetTileIndex(0, 1);
				this.dy = AIMap.GetTileIndex(-1, 0);

				this.construction_site_x_len = -this.construction_site_y_size;
				this.construction_site_y_len = this.construction_site_x_size;

				transformations.append(RoadSiteConstructionCommand.ROTATE_COMMAND);
				transformations.append(RoadSiteConstructionCommand.POINT_REFLECT_COMMAND);
				break;
		}

		local result = [];
		foreach (dummy_dummy_id, command in commands) {
			local new_command = command;
			foreach (dummy_id, transformation in transformations) { 
				new_command = transformation(new_command);
			}
			result.append(new_command);
		}

		return result;
	}

	construction_site_x_size = 0;
	construction_site_x_len = 0;

	construction_site_y_size = 0;
	construction_site_y_len = 0;

	function GetBuildAreaXSize()
	{
		return this.construction_site_x_size;
	}

	function GetBuildAreaYSize()
	{
		return this.construction_site_y_size;
	}

	function Execute(node, cargo_id)
	{
		local nx = this.GetBuildAreaXSize();
		local ny = this.GetBuildAreaYSize(); 

		local station_type = RoadUtils.GetStationType(cargo_id);
		local r = AIStation.GetCoverageRadius(station_type) - 1;
		local list = TileUtils.GetProductionNodeFreeStationLocations(node, -1, nx, ny, r, cargo_id);

		local production_tiles = node.IsTown() ?
			node.AITileList_NodeAccepting(r + 1, cargo_id) :
			node.AITileList_NodeProducing(r + 1, cargo_id);

		production_tiles.Valuate(AITile.GetCargoProduction, cargo_id, nx, ny, r + 1);
		production_tiles.RemoveBelowValue(1);

		local valuator = RoadSiteConstructionPlan.IsGoodConstructionSite.bindenv(this);
		list.Valuate(valuator);
		list.RemoveValue(0);

		if (list.Count() == 0) {
			CodeUtils.Log("...can't find enough flat land around." , 2);
			return {err_code = -1, build_result_info = null};
		} 

		local nnx = construction_site_x_len - construction_site_x_len / nx;
		local nny = construction_site_y_len - construction_site_y_len / ny;
		local station_center = AIMap.GetTileIndex(nnx / 2, nny/ 2);
		list.Valuate(AIMap.DistanceSquare, node.location - station_center);
		list.Sort(AIList.SORT_BY_VALUE, true);

		nnx = construction_site_x_len;
		nny = construction_site_y_len;

		local station_tiles_offsets = [];
		foreach (dummy_dummy_id, command in this.construction_commands) {
			switch (command.code) {
				case ConstructionCommand.CC_BUS_STATION :
				case ConstructionCommand.CC_TRUCK_STATION :
					station_tiles_offsets.append(command.x * dx + command.y * dy);
					break;
			}
		}

		foreach (t, dummy in list) {
			local is_in_production_area = false;
			foreach (dummy_id, offset in station_tiles_offsets) {
				local s = t + offset;
				if (production_tiles.HasItem(s)) {
					is_in_production_area = true;
					break;
				}
			}
			if (!is_in_production_area) continue;

			if (!TileUtils.MakeStationFoundation(t, true, nnx, nny, true)) continue;
			if (!TileUtils.MakeStationFoundation(t, true, nnx, nny, false)) continue;

			local result = this.BuildSelf(t);
			switch (result.err_code) {
				case 0:
					return result;
				case 2:
					CodeUtils.Log("...not enough money" , 2);
					return result;
				case -3:
					CodeUtils.Log("...local authority refuses" , 2);
					result.err_code = -1;
					return result;
				default:
					break;
			}

			this.ClearBuildArea(t);
		}

		return {err_code = -1, build_result_info = null};
	}

	function BuildSelf(t)
	{
		local result = {err_code = 0, build_result_info = null};
		local old_road_type = RoadUtils.SetRoadType(AIRoad.ROADTYPE_ROAD);

		AICompany.SetLoanAmount(AICompany.GetMaxLoanAmount());

		this.s_id = -1;
		local n = this.construction_commands.len();
		for (local i = 0; i < n; i++) {
			if (this.IsComponentExist(t, i)) continue;
			if (!this.doBuildComponent(t, i)) {
				local err = AIError.GetLastError();
				result.err_code = err == AIError.ERR_NOT_ENOUGH_CASH ? -2 : -1;
				if (err == AIError.ERR_LOCAL_AUTHORITY_REFUSES) result.err_code = -3;

				for (local j = i; j >= 0; j--) {
					this.doEraseComponent(t, j);
				}
				break;
			}
		}

		if (result.err_code == 0) {
			result.build_result_info = this.GetReturnValueOnSuccess(t);
		}

		RoadUtils.SetRoadType(old_road_type);
		return result;
	}

	function ClearBuildArea(area_ne_corner)
	{
		for (local i = 0; i < this.GetBuildAreaXSize(); i++) {
			for (local j = 0; j < this.GetBuildAreaXSize(); j++) {
				local t = area_ne_corner + i * dx + j * dy
				if (!AITile.IsBuildable(t) || AIRoad.IsRoadTile(t)) {
					if (AITile.DemolishTile(t)) continue
					if (AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH) return -2;
				}
			}
		}

		return this.IsBuildAreaClear(area_ne_corner) ? 0 : -1;
	}

	function IsBuildAreaClear(area_ne_corner)
	{
		for (local i = 0; i < this.GetBuildAreaXSize(); i++) {
			for (local j = 0; j < this.GetBuildAreaXSize(); j++) {
				local t = area_ne_corner + i * dx + j * dy
				if (!AITile.IsBuildable(t) || AIRoad.IsRoadTile(t)) return false;
			}
		}

		return true;
	}

	s_id = -1;

/* protected */
	dx = -1;

	dy = -1;

	function GetReturnValueOnSuccess(construction_site_ne_corner)
	{
		local result = {
			station_id = this.s_id,
			cargo_load_terminal_tiles = [],
			cargo_drop_terminal_tiles = [],
		}

		foreach (dummy_id, command in this.construction_commands) { 
			if (command.code == ConstructionCommand.CC_TRUCK_STATION) {
				local target = command.extension.drop_only ?
					result.cargo_drop_terminal_tiles :
					result.cargo_load_terminal_tiles;

				target.append(construction_site_ne_corner + command.x * dx + command.y * dy);
			}
		}

		return result;
	}

	function doBuildComponent(construction_site_ne_corner, command_number)
	{
		local v_type = AIRoad.ROADVEHTYPE_TRUCK;
		local command = this.construction_commands[command_number];
		local s = construction_site_ne_corner + command.x * dx + command.y * dy;
		switch (command.code) {
			case ConstructionCommand.CC_ROAD_X :
				local n = command.extension;
				return AIRoad.BuildRoad(s, s + n * dx);
			case ConstructionCommand.CC_ROAD_Y :
				local n = command.extension;
				return AIRoad.BuildRoad(s, s + n * dy);
			case ConstructionCommand.CC_ROAD_DEPOT :
				local offset = command.extension;
				AITile.DemolishTile(s);
				AIRoad.BuildRoad(s, s + offset);
				return AIRoad.BuildRoadDepot(s, s + offset);
			case ConstructionCommand.CC_BUS_STATION :
				v_type = AIRoad.ROADVEHTYPE_BUS;
			case ConstructionCommand.CC_TRUCK_STATION :
				local offset = command.extension.enter_offset;
				AITile.DemolishTile(s);
				AIRoad.BuildRoad(s, s + offset);
				if (this.s_id == -1) {
					if (!AIRoad.BuildDriveThroughRoadStation(s, s + offset, v_type, AIStation.STATION_NEW)) return false;
					this.s_id = AIStation.GetStationID(s);
					return true;
				} else {
					return AIRoad.BuildDriveThroughRoadStation(s, s + offset, v_type, this.s_id);
				}
		}

		return false;
	}

	function doEraseComponent(construction_site_ne_corner, command_number)
	{
		local command = this.construction_commands[command_number];
		local s = construction_site_ne_corner + command.x * dx + command.y * dy;
		switch (command.code) {
			case ConstructionCommand.CC_ROAD_X :
				for (local i = 0; i < command.extension; i++) {
					AITile.DemolishTile(s + i * dx);
				}
				break;
			case ConstructionCommand.CC_ROAD_Y :
				for (local i = 0; i < command.extension; i++) {
					AITile.DemolishTile(s + i * dy);
				}
				break;
			case ConstructionCommand.CC_ROAD_DEPOT :
			case ConstructionCommand.CC_BUS_STATION :
			case ConstructionCommand.CC_TRUCK_STATION :
				AITile.DemolishTile(s);
		}
	}

	function IsComponentExist(construction_site_ne_corner, command_number)
	{
		local command = this.construction_commands[command_number];
		local s = construction_site_ne_corner + command.x * dx + command.y * dy;
		switch (command.code) {
			case ConstructionCommand.CC_ROAD_X :
				return this.IsRoadExist(s, dx, command.extension);
			case ConstructionCommand.CC_ROAD_Y :
				return this.IsRoadExist(s, dy, command.extension);
			case ConstructionCommand.CC_ROAD_DEPOT :
				return this.IsDepotExist(s, s + command.extension);
			case ConstructionCommand.CC_BUS_STATION :
			case ConstructionCommand.CC_TRUCK_STATION :
				return this.IsStationExist(s, s + command.extension.enter_offset);
		}

		return false;
	}

/* private */
	construction_commands = null;

	station_or_depot_tiles = null;

	function IsRoadExist(start_tile, dir_offset, len)
	{
		for (local i = 0, t = start_tile; i < len; i++, t += dir_offset) {
			if (!AIRoad.AreRoadTilesConnected(t, t + dir_offset)) return false;
		}

		return true;
	}

	function IsStationExist(station_tile, exit_tile)
	{
		if (!AIRoad.IsRoadStationTile(station_tile)) return false;
		if (!AIRoad.IsRoadTile(exit_tile)) return false;

		return AIRoad.AreRoadTilesConnected(station_tile, exit_tile);
	}

	function IsDepotExist(depot_tile, exit_tile)
	{
		if (!AIRoad.IsRoadDepotTile(depot_tile)) return false;
		if (!AIRoad.IsRoadTile(exit_tile)) return false;

		return AIRoad.AreRoadTilesConnected(depot_tile, exit_tile);
	}
}

function RoadSiteConstructionPlan::IsGoodConstructionSite(area_ne_corner)
{ 
	local depot_tiles = {};
	local station_tiles = {};
	foreach (dummy_id, command in this.construction_commands) {
		local t = command.x * dx + command.y * dy;
		if (command.code == ConstructionCommand.CC_ROAD_DEPOT) {
			depot_tiles[area_ne_corner + t] <- command.extension;
		}

		if (command.code == ConstructionCommand.CC_TRUCK_STATION) {
			station_tiles[area_ne_corner + t] <- command.extension.enter_offset;
		}
	}

	local mine_id = AICompany.ResolveCompanyID(AICompany.COMPANY_SELF);
	local IS_FLAT = TileUtils.IsTileFlatOrWithThreeCornersRaised;
	for (local i = 0; i < this.GetBuildAreaXSize(); i++) {
		for (local j = 0; j < this.GetBuildAreaYSize(); j++) {
			local t = area_ne_corner + i * dx + j * dy;

			if (!AITile.IsBuildable(t) && !AIRoad.IsRoadTile(t)) return false;
			if (AITile.IsStationTile(t)) return false;

			local id = AICompany.ResolveCompanyID(AITile.GetOwner(t));
			if (id != mine_id && id != AICompany.COMPANY_INVALID) return false;

			if (AIRoad.IsRoadTile(t)) {
				if (t in depot_tiles) return false;
				if (t in station_tiles) {
					if (!IS_FLAT(t) || !IS_FLAT(t + station_tiles[t])) return false;
				}					
			}
		}
	}

	return true;
}
