/**
 * Base class to handle road routes building process.
 */
class DefaultRoadConstructionStrategy extends UniversalAction_AbstractExecutionStrategy
{
/* public */
	function ConcretizeContext()
	{
		return this.SelectObjectToBuild();
	}

	function GetActionCost(context)
	{
		return this.GetConstructionCost(context);
	}

	function Execute(context)
	{
		return this.Build(context);
	}

	/**
	 * Calculate road route priority.
	 * @param route Road route which rank we want to know.
	 * @return Number indicating road route upgrade desireability.
	 */
	function RankRoadRoute(route)
	{
		if (route.IsNefarious() || route.IsCompletelyUpgraded()) {
			return APriority.BAD;
		}

		local build_cost = 2 * RoadSettings.Get().GetRoadStationCost();
		if (route.connection.state != ConnectionState.CS_ROAD_BUILT) {
			build_cost += route.connection.cost;
		} else {
			build_cost += route.best_engine_price;
		}
		local income = route.GetEstimatedIncome();
		local full_cost = build_cost + route.GetVehiclesCost();
		build_cost += 2 * route.best_engine_price;

		return ActionPriority.CalculatePriority(build_cost, full_cost, 0, income);
	}

	/**
	 * Calculate united road routes priority.
	 * @param road_routes Container with road routes that will probably share
	 *  one road when built.
	 * @return Number indicating upgrade desireability for all of the given
	 *  routes, as if it was one.
	 */
	function RankRoadRoutes(road_routes)
	{
		local build_cost = 1, income = 0, full_cost = 1;
		foreach (dummy_id, route in road_routes) {
			if (route.IsNefarious() || route.IsCompletelyUpgraded()) continue;

			if (route.connection.state != ConnectionState.CS_ROAD_BUILT) {
				if (build_cost > 1) continue;
				build_cost = this.GetConstructionCost(route);
			}
			income += route.GetEstimatedIncome();
			full_cost += route.GetVehiclesCost();
		}

		full_cost += build_cost;
		return ActionPriority.CalculatePriority(build_cost, full_cost, 0, income);
	}

/* protected */
	/**
	 * Syntactic sugar/replacement for the ConcretizeContext method.
	 */
	function SelectObjectToBuild();

	/**
	 * Syntactic sugar/replacement for the GetActionCost method.
	 */
	function GetConstructionCost(road_route)
	{
		local cost = 2 * RoadSettings.Get().GetRoadStationCost();
		cost += road_route.connection.cost + road_route.best_engine_price;
		if (road_route.GetStart().GetClassName() == "HeapNode") {
			cost = 6 * cost / 5;
		}
		return cost;
	}

	/**
	 * Syntactic sugar/replacement for the Execute method.
	 */
	function Build(road_route);

/* protected */
	/**
	 * Calculate road construction cost for the given route.
	 * @param route Road route to check.
	 * @return None (route.connection will be updated).
	 */
	function CalculateConnectionCost(route)
	{
		local connection = route.connection;
		if (connection.state != ConnectionState.CS_ACCURATE_COST_UNKNOWN) return;

		CodeUtils.Log("Calculating road cost for " + route.GetName(), 1);
		local c = route.GetCargoID();
		local s_list = RoadUtils.GetPossibleStationLocations(route.GetStart(), c, true);
		local e_list = RoadUtils.GetPossibleStationLocations(route.GetEnd(), c, false);
		if (s_list.Count() == 0 || e_list.Count() == 0) {
			route.MakeNefarious(3 * GameTime.YEAR);
			CodeUtils.Log("... unable to start pathfinding", 1);
			connection.cost = 2 * connection.cost;
			connection.state = ConnectionState.CS_ACCURATE_COST_CALCULATED;
			Terron.UpdateMemento();
			return;
		}

		local pf = this.GetDefaultPathfinder();
		local cost = pf.GetConnectionCost(s_list.Begin(), e_list.Begin(), AIRoad.ROADTYPE_ROAD);

		local msg = cost == -1 ? "pathfinder error" : "cost = " + cost;
		CodeUtils.Log("... " + msg, 1);

		connection.cost = cost != -1 ? cost : 3 * connection.cost / 2;
		if (cost == -1 && route.connection.state == ConnectionState.CS_ROAD_BUILT) {
			connection.cost = RoadSettings.Get().GetRoadCost(route.length);
		}
		connection.state = ConnectionState.CS_ACCURATE_COST_CALCULATED;
		Terron.UpdateMemento();
	}

	/**
	 * Get pathfinder to be used for road building.
	 */
	function GetDefaultPathfinder()
	{
		return SimplePF_Adapter();
	}

	/**
	 * Create new trade route from the given road route.
	 * @param road_route Road route object.
	 * @param start_station First trade station.
	 * @param end_station Second trade station.
	 * @return New trade route or null if fails.
	 */
	function FoundNewTradeRoute(road_route, start_station, end_station)
	{
		CodeUtils.Log("Creating new road trade route... ", 1);
		//AILog.Info(start_station.GetName())
		//AILog.Info(end_station.GetName())
		/*local l = road_route.GetCurrentRouteLevel();
		local s = road_route.GetStartStation(l);
		local e = road_route.GetEndStation(l);
		if (s == null || e == null) {
			CodeUtils.Log("   ... suitable stations not found aborting", 1);
			return null;
		}*/
		local result = RoadTradeRoute(road_route, start_station, end_station);
		CodeUtils.Log(result.GetName() + " founded", 2);
		return result;
	}
}
