/**
 * Main goal of this class is to provide const-time access to pathfinding nodes,
 *  instead of "default" log-time.
 * This is in fact a 2D array.
 */
class SimpleSearchRectangle
{
/* public */
	/**
	 * Creates SimpleSearchRectangle object.
	 * @param s Array with pathfinder "start" tiles.
	 * @param e Array with pathfinder "end" tiles.
	 */
	constructor(s, e)
	{
		local t = AIController.GetTick();
		CodeUtils.Log("Creating local search rectangle...", 0);
		this.pf_data = RoadPFDataStructures.Get();

		local shift = 9;
		local big = max(AIMap.GetMapSizeX(), AIMap.GetMapSizeY()) + 1;
		local s_min_x = big, s_max_x = -1, s_min_y = big, s_max_y = -1;
		local e_min_x = big, e_max_x = -1, e_min_y = big, e_max_y = -1;
		foreach (dummy_id, t in s) {
			local x = AIMap.GetTileX(t);
			local y = AIMap.GetTileY(t);
			if (x < s_min_x) s_min_x = x;
			if (x > s_max_x) s_max_x = x;
			if (y < s_min_y) s_min_y = y;
			if (y > s_max_y) s_max_y = y;
		}
		foreach (dummy_id, t in e) {
			local x = AIMap.GetTileX(t);
			local y = AIMap.GetTileY(t);
			if (x < e_min_x) e_min_x = x;
			if (x > e_max_x) e_max_x = x;
			if (y < e_min_y) e_min_y = y;
			if (y > e_max_y) e_max_y = y;
		}

		this.x_min = (s_min_x < e_min_x ? s_min_x : e_min_x) - shift;
		this.x_min = max(this.x_min, 1);
		this.x_max = (s_max_x > e_max_x ? s_max_x : e_max_x) + shift;
		this.x_max = min(this.x_max, AIMap.GetMapSizeX() - 1);

		this.y_min = (s_min_y < e_min_y ? s_min_y : e_min_y) - shift;
		this.y_min = max(this.y_min, 1);
		this.y_max = (s_max_y > e_max_y ? s_max_y : e_max_y) + shift;
		this.y_max = min(this.y_max, AIMap.GetMapSizeY() - 1);

		local x_len = this.x_max - this.x_min;
		local y_len = this.y_max - this.y_min;
		CodeUtils.Log("...bounds determined: ", 0);
		CodeUtils.Log("   from [" + x_min + ", " + y_min + "] - to - [" + x_max + ", " + y_max + "]", 0);
		local d = AIMap.DistanceManhattan(s[0], e[0]);
		CodeUtils.Log("   distance manhattan between start tiles" + d, 0);
		CodeUtils.Log("   up to " + x_len * y_len + " possible nodes inside", 0);

		CodeUtils.Log("...allocating memory...", 0);
		this.nodes_array = array(x_len);
		for (local i = 0; i < x_len; i++) {
			this.nodes_array[i] = array(y_len);
		}
	}

	/**
	 * Save a patfinding node inside this object.
	 * @param x The node tile X coordinate.
	 * @param y The node tile Y coordinate.
	 * @param node The node to store.
	 * @return Not used.
	 * @note ALWAYS use IsInRect function BEFORE calling this one.
	 */
	function SetNode(x, y, node)
	{
		this.nodes_array[x - this.x_min][y - this.y_min] = node;
	}

	/**
	 * Recover a patfinding node from this object.
	 * @param x The node tile X coordinate.
	 * @param y The node tile Y coordinate.
	 * @return The node with [x, y] coordinates.
	 * @note Function include "lazy" initialization:
	 *  at the begining all nodes records are null, and if desired node
	 *  was not added yet => it will be created within this function.
	 * @note ALWAYS use IsInRect function BEFORE calling this one.
	 */
	function GetNode(x, y)
	{
		local node = this.nodes_array[x - this.x_min][y - this.y_min];
		if (node == null) {
			node = pf_data.GetPrimitiveNodeXY(x, y);
			this.nodes_array[x - this.x_min][y - this.y_min] = node;
		}
		return node;
	}

	/**
	 * Recover a patfinding node from this object.
	 * @param x The node tile X coordinate.
	 * @param y The node tile Y coordinate.
	 * @return The node with [x, y] coordinates, or null if node was not stored
	 *  into this object before.
	 * @note ALWAYS use IsInRect function BEFORE calling this one.
	 */
	function GetRawNode(x, y)
	{
		return this.nodes_array[x - this.x_min][y - this.y_min];
	}

	/**
	 * Check whether the given coordinates pair lies within this rectangle.
	 * @param x X coordinate.
	 * @param y Y coordinate.
	 * @return true If this object can contain [x, y] node, false otherwise.
	 */
	function IsInRect(x, y)
	{
		return (x_min < x && x < x_max && y_min < y && y < y_max);
	}

/* private */
	/** "Left" border (seems right in game map, though it doesn't matter) */
	x_min = null;

	/** "Right" border (seems left in game map, though it doesn't matter) */
	x_max = null;

	/** "Bottom" border (seems top in game map, though it doesn't matter) */
	y_min = null;

	/** "Top" border (seems bottom in game map, though it doesn't matter) */
	y_max = null;

	/** 2D nodes array representing rectangle itself */
	nodes_array = null;

	/** Helper with pathfinding related data */
	pf_data = null;
}
