/**
 * Main class for all road related actions.
 */
class RoadDepartment extends AbstractTransportSystem
{
/* public */
	static function GetClassName()
	{
		return "RoadDepartment";
	}

	/**
	 * Get the instance of the RoadDepartment singleton.
	 */
	static function Get()
	{
		if (RoadDepartment._instance[0] == null) {
			RoadDepartment(RoadMap(), RoadMap());
		}
		return RoadDepartment._instance[0];
	}

	static function Restore(memento)
	{
		local dep = RoadDepartment(memento.nodes_roadmap, memento.stations_roadmap);
		if ("progress_sign_id" in memento) {
			dep.progress_sign_id = memento.progress_sign_id;
		}
		return dep;
	}

	/**
	 * Get the container with trade routes whether needs to be repaired.
	 */
	function GetDamagedTradeRoutes() { return this.damaged_trade_routes;}

	/**
	 * Get the road section progress.
	 * @return Float number indicating progress in %.
	 */
	function GetDevelopmentProgress() { return this.development_progress.Get();}

	function GetName() { return "Road Section";}

	/**
	 * Get the between-nodes-roads graph.
	 * @return Roadmap object with information about roads between nodes.
	 */
	function GetNodesRoadMap() { return this.nodes_roadmap;}

	/**
	 * Get the between-station-roads graph.
	 * @return Roadmap object with information about roads between stations. 
	 */
	function GetStationsRoadMap() {	return this.stations_roadmap;}

/* protected */
	function onDisable(time)
	{
		foreach (id, sys in this.GetSubsystems()) {
			sys.Disable(time);
		}
	}

/* private */
	/** Single instance of this class */
	static _instance = [null];

	/** Between-nodes-roads graph */
	</ must_save = true />
	nodes_roadmap = null;

	/** Between-stations-roads graph */
	</ must_save = true />
	stations_roadmap = null;

	/**
	 * "Development progress" property.<p>
	 * "Development progress" actually mean
	 *  [current number of vehicles] / [max number of vehicles allowed to build]
	 */
	development_progress = null;

	/** Trade routes where road can be damaged and need repairs */
	damaged_trade_routes = null;

	/**
	 * Container with tiles prohibited for further station construction attempts.
	 */
	dangerous_station_locations = null;

	/** ID of the sign with road progress (will be near HQ) */
	</ must_save = true />
	progress_sign_id = -1;

	/**
	 * Used to create and maintain all lower level road logic stuff but
	 *  prevent any construction. Should be useful when cars are going to be
	 *  only secondary transport type and provide easy explicit roads building
	 *  delegation to other AI parts.
	 */
	dummy_modules = null;

	/**
	 * Private constructor for RoadDepartment singleton.
	 */
	constructor(nodes_roadmap, stations_roadmap)
	{
		::AbstractTransportSystem.constructor();
		if (RoadDepartment._instance[0] != null) assert(null);
		RoadDepartment._instance[0] = this;

		this.nodes_roadmap = nodes_roadmap;
		this.stations_roadmap = stations_roadmap;
		this.development_progress = Property(this.doGetProgress.bindenv(this), GameTime.MONTH);
		this.development_progress.SetName("Road development progress update");
		this.damaged_trade_routes = TableContainer.new("DamagedRoads");
		this.dangerous_station_locations = {};
		this.dummy_modules = [null, null];

		local cfg = RoadSettings.Get();		
		if (cfg.max_vehicles != 0) {
			this.AddSubsystem(RoadTransitHandler.Get());

			local dumb_control_section = null;
			switch (AIController.GetSetting("TruckModule")) {
				case 1: dumb_control_section = DumbTruckControlSection(false); break;
				case 2: this.AddSubsystem(RoadNetworksCoordinator(false)); break;
				case 3: this.AddSubsystem(RoadNetworksCoordinator(true)); break;
				default: break;
			}

			if (cfg.dummy_truck_nodes_initialization_required && dumb_control_section == null) {
				this.dummy_modules[1] = DumbTruckControlSection(false);
			} else if (dumb_control_section != null) {
				this.AddSubsystem(dumb_control_section);
			}

			local bus_setting = AIController.GetSetting("BusModule");
			if (bus_setting == 1 || bus_setting == 3) {
				this.AddSubsystem(BusHubControlSection());
			}
			if (bus_setting >= 2) {
				this.AddSubsystem(PaxTransitControlSection());
			}

			SearchRoadPlugsTask();
			SelfControlTask();

			Terron_Event.EnginePreview.AddListener(this);
			DepartmentRequestsPriorityEvent.AddListener(this);
		}
	}

	/**
	 * Places a sign with road progress near HQ.
	 * @param road_progress Float between 0 and 1 - percentage.
	 */
	function ShowProgress(road_progress)
	{
		if (this.progress_sign_id != -1) {
			AISign.RemoveSign(this.progress_sign_id);
		}
		local t = AICompany.GetCompanyHQ(AICompany.COMPANY_SELF);
		if (AIMap.IsValidTile(t)) {
			local p = (road_progress * 100).tointeger();
			local text = road_progress >= 1.0 ? "Road expansion complete!" :
				"Road expansion: " + p + "%";
			local id = AISign.BuildSign(t, text);
			if (AISign.IsValidSign(id)) this.progress_sign_id = id
		}
	}

/* public */
	/**
	 * Function that actually calculates road progress.
	 */
	function doGetProgress()
	{
		local vehicles = AIVehicleList();
		vehicles.Valuate(AIVehicle.GetVehicleType);
		vehicles.KeepValue(AIVehicle.VT_ROAD);

		local cfg = RoadSettings.Get();
		local p = 1.0 * vehicles.Count() / (cfg.max_vehicles * (1.0 - cfg.reserved_for_transit) + 1);
		this.ShowProgress(p)
		return p;
	}

	/** Disable self so other department can build. */
	function OnDepartmentPriorityRequest(department)
	{
		if (department.GetClassName() == AirDepartment.GetClassName()) {
			this.Disable(GameTime.YEAR);
		}
	}

	/**
	 * Accept all new road engines on preview.
	 */
	function OnEnginePreview(ec)
	{
		if (ec.GetVehicleType() == AIVehicle.VT_ROAD) ec.AcceptPreview();
	}

	/**
	 * When we've built a new road, all AI parts must know about it.<p>
	 * This is the function to "spread the knowledge about new road" inside AI.
	 * @param s_node Node, where the road begins.
	 * @param e_node Node, where the road ends.
	 */
	function UpdateConnectionsRoadMapWhenRoadBuilt(s_node, e_node)
	{
		local t1 = s_node.GetLocation();
		local t2 = e_node.GetLocation();

		/* Write information about new road (if needed) */
		local c = this.nodes_roadmap.GetConnection(t1, t2);
		if (c != null) {
			c.state = ConnectionState.CS_ROAD_BUILT;
			c.cost = 0;
			Terron.UpdateMemento();
		}

		local d = 12 + AIMap.DistanceManhattan(t1, t2) / 16;
		/* Find industries located near the start */
		local i1_list = AIIndustryList();
		i1_list.Valuate(AIIndustry.GetDistanceManhattanToTile, t1);
		i1_list.KeepBelowValue(d);

		/* Find industries located near the end */
		local i2_list = AIIndustryList();
		i2_list.Valuate(AIIndustry.GetDistanceManhattanToTile, t2);
		i2_list.KeepBelowValue(d);

		/* Write information about new road for nearby industries */
		local rep = NodesRepository.Get();
		foreach (i1_id, dummy in i1_list) {
			local i1_node = rep.GetNode(AIIndustry.GetIndustryType(i1_id), i1_id);
			if (i1_node == null) continue; // can happen when industry is closing
			foreach (i2_id, dummy in i2_list) {
				local i2_node = rep.GetNode(AIIndustry.GetIndustryType(i2_id), i2_id);
				if (i2_node == null) continue; // can happen when industry is closing
				local c = this.nodes_roadmap.GetConnection(i1_node.location, i2_node.location);
				if (c == null) continue;
				c.state = ConnectionState.CS_ROAD_BUILT;
				c.cost = 0;
				Terron.UpdateMemento();
			}
		}
	}
}

/**
 * Stop road expansion when amount of road vehicles is closing to the
 *  allowed upper limit.<p> Also enable/disable accurate construction mode.
 */
class RoadDepartment.SelfControlTask extends Terron_Task
{
/* public */
	constructor()
	{
		local date = AIDate.GetCurrentDate() + GameTime.DAY;
		::Terron_Task.constructor(date, GameTime.HALF_YEAR);
	}

/* protected */
	function Execute()
	{
		local cfg = RoadSettings.Get();
		local dep = RoadDepartment.Get();
		local road_progress = dep.GetDevelopmentProgress();
		if (road_progress > 1.0) {
			dep.Disable(2 * GameTime.YEAR);
		} else {
			if (road_progress > 1.0 - cfg.reserved_for_transit) {
				local correct_name = PaxTransitControlSection.GetClassName();
				foreach (dummy_id, item in dep.GetSubsystems()) {
					if (item.GetClassName() != correct_name) item.Disable(GameTime.YEAR);
				}
			}
		}

		if (road_progress > cfg.safe_development_percentage) {
			if (cfg.accurate_construction) cfg.accurate_construction = false;
		} else {
			local id_self = AICompany.COMPANY_SELF;
			local t = AICompany.CURRENT_QUARTER;
			local income = AICompany.GetQuarterlyIncome(id_self, t);
			income = income - AICompany.GetQuarterlyExpenses(id_self, t);
			cfg.accurate_construction = income.tointeger() <= 125000;
		}
	}
}

Terron_ClassTable.RegisterClass(RoadDepartment);
