/**
 * Class that provides fast access to best road engine for each cargo ID.
 */
class RoadEngineRoster extends Terron_Settings
{
/* public */
	static function GetClassName()
	{
		return "RoadEngineRoster";
	}

	static function Restore(memento)
	{
		return RoadEngineRoster();
	}

	/**
	 * Get "best" road engine for specified cargo ID.
	 * @param cargo_id ID of the desired cargo to transport.
	 * @return Engine ID considered to be the best.
	 */
	function GetBestEngine(cargo_id, ignore_articulated)
	{
		return ignore_articulated ?
			this.engines[cargo_id][1].Get() :
			this.engines[cargo_id][0].Get();
	}

	/**
	 * Mark given engine as deprecated and never use it.
	 * @param e Engine ID.
	 */
	function ProhibitEngine(e)
	{
		this.deprecated_engines[e] <- AIEngine.GetName(e);
		foreach (dummy_dummy_id, variants in this.engines) {
			foreach (dummy_id, engine in variants) {
				engine.ForceUpdate();
			}
		}
		this.last_update_date = AIDate.GetCurrentDate();
	}

	/**
	 * Re-check engine list in free time.
	 */
	function OnFreeTime(dummy)
	{
		local date = AIDate.GetCurrentDate();
		if (date - this.last_update_date < 3 * GameTime.MONTH) return;

		foreach (dummy_dummy_id, variants in this.engines) {
			foreach (dummy_id, engine in variants) {
				engine.ForceUpdate();
			}
		}
		this.last_update_date = AIDate.GetCurrentDate();
	}

/* protected */
	constructor()
	{
		::Terron_Settings.constructor();
		this.engines = {};
		this.deprecated_engines = {};
		local bad_engines = this.deprecated_engines;
		foreach (c, dummy_value in AICargoList()) {
			local ignore_articulated = function():(c, bad_engines) {
				return RoadEngineRoster.doGetEngine(c, bad_engines, true);
			}
			local allow_articulated = function():(c, bad_engines) {
				return RoadEngineRoster.doGetEngine(c, bad_engines, false);
			}

			local l = AICargo.GetCargoLabel(c);
			this.engines[c] <- array(2);
			this.engines[c][0] = Property(allow_articulated, GameTime.ENGINE_REVIEW_INTERVAL - 1);
			this.engines[c][0].SetName("Articulated road engines for " + l + " update");
			this.engines[c][1] = Property(ignore_articulated, GameTime.ENGINE_REVIEW_INTERVAL - 1);
			this.engines[c][1].SetName("Non-articulated road engines for " + l + " update");
		}

		last_update_date = AIDate.GetCurrentDate();

		FreeTimeEvent.AddListener(this);
		AISettingsSaveEvent.RemoveListener(this);
	}

/* private */
	/**
	 * Get "best" road engine for specified cargo ID.
	 * @param cargo_id ID of the desired cargo to transport.
	 * @return Engine ID considered to be the best.
	 */
	static function doGetEngine(cargo_id, bad_engines, ignore_articulated)
	{
		/* Scan engine list and remove futile engines */
		local list = AIEngineList(AIVehicle.VT_ROAD);
		/* Can't handle route cargo => futile */
		list.Valuate(AIEngine.CanRefitCargo, cargo_id);
		list.RemoveValue(0);
		/* Remove trams */
		list.Valuate(AIEngine.GetRoadType);
		list.KeepValue(AIRoad.ROADTYPE_ROAD);
		/* Breaks often => futile */
		list.Valuate(AIEngine.GetReliability);
		list.KeepAboveValue(65);
		/* Low capacity => futile */
		list.Valuate(AIEngine.GetCapacity);
		list.RemoveBelowValue(10);
		/* Remove unbuildable*/
		list.Valuate(AIEngine.IsBuildable);
		list.RemoveValue(0);

		/* Ignore articulated */
		if (ignore_articulated) {
			list.Valuate(AIEngine.IsArticulated);
			list.KeepValue(0);
		}
		/* Ignore deprecated */
		foreach (e, e_name in bad_engines) {
			if (list.HasItem(e) && AIEngine.GetName(e) == e_name) {
				list.RemoveItem(e);
			}
		}

		local all_engines_stats = EnginesInfo.stats;
		foreach (e, dummy in list) {
			if (!(e in all_engines_stats)) EnginesInfo.CalculateEngineStats(e);
		}

		local current_year = AIDate.GetYear(AIDate.GetCurrentDate());
		/* Select engine with max (capacity * speed) */
		local f = function (e) : (current_year, all_engines_stats) {
			local stats = all_engines_stats[e];
			local design_age = current_year - stats[EngineStat.ES_DESIGN_YEAR];
			local rl = stats[EngineStat.ES_RELIABILITY] + max(12 - 2 * design_age, -4);
			rl = (min(100, rl)).tofloat() / 100;
			return rl * stats[EngineStat.ES_CAPACITY] * stats[EngineStat.ES_MAX_SPEED];
		}

		return CorporationUtils.ChooseBestEngineFromList(list, f);
	}

	/** Map, where key is cargo ID, and value is best engine for such cargo. */
	engines = null;

	/** Table with deprecated engines */
	deprecated_engines = null;

	/** Date of last update */
	last_update_date = -1000;
}

Terron_ClassTable.RegisterClass(RoadEngineRoster);
