/**
 * Class that handles road trade.
 */
class RoadTradeRoute extends TradeRoute
{
/* public */
	static function GetClassName()
	{
		return "RoadTradeRoute";
	}

	constructor(road_route, s_stop, e_stop)
	{
		this.name = s_stop.GetName() + " -> " + e_stop.GetName();
		this.name += " (" + AICargo.GetCargoLabel(road_route.cargo_id) + ")";
		::TradeRoute.constructor([s_stop, e_stop]);
 		road_route.trade_routes.AddItem(this);
 
 		local c = road_route.cargo_id;

		this.len = AIMap.DistanceManhattan(s_stop.location, e_stop.location);
		this.s_stop = s_stop;
		this.e_stop = e_stop;
		this.road_route = road_route;

		foreach (dummy_id, terminal in s_stop.station_plan.GetLoadingTerminals()) {
			if (terminal.cargo_id == c) {
				this.s_terminal = terminal;
				break;
			}
		}

		foreach (dummy_id, terminal in e_stop.station_plan.GetCargoDropTerminals()) {
			if (c in terminal.cargo_list) {
				this.e_terminal = terminal;
				break;
			}
		}
		if (this.e_terminal == null) {
			foreach (dummy_id, terminal in e_stop.station_plan.GetLoadingTerminals()) {
				if (terminal.cargo_id == c) {
					this.e_terminal = terminal;
					break;
				}
			}
		}

		if (!(this.s_terminal.core.drive_through && this.e_terminal.core.drive_through)) {
			this.use_articulated = false;
		}

		this.station_type = RoadUtils.GetStationType(c);
		this.vehicles_required = min(4 * road_route.GetEstimatedVehicles() / 5, 10);
		if (road_route.GetStart().GetClassName() == HeapNode.GetClassName()) {
			this.vehicles_required = this.vehicles_required / 2;
		}
		if (this.s_terminal.core.tiles.len() == 0) {
			Corporation.Get().AddAction(AIExpandRoadStationAction(this.s_terminal));
			this.vehicles_required = 0;
		}
		if (this.e_terminal.core.tiles.len() == 0) {
			Corporation.Get().AddAction(AIExpandRoadStationAction(this.e_terminal));
			this.vehicles_required = 0;
		}
		this.integrity_check_date = AIDate.GetCurrentDate();
		this.ReloadOwnVehicles();

		Terron_Event.RoadVehicleCrashed.AddListener(this);
		Terron_Event.RoadVehicleUnprofitable.AddListener(this);
		ForceSellRoadVehicleEvent.AddListener(this);
		RoadVehicleStuckEvent.AddListener(this);

		local road_department = RoadDepartment.Get();
		if (road_department.IsEnabled()) {
			/* Stop new road stations building when have near max vehicles allowed */
			local road_vehicles = AIVehicleList();
			road_vehicles.Valuate(AIVehicle.GetVehicleType);
			road_vehicles.KeepValue(AIVehicle.VT_ROAD); 
			if (road_vehicles.Count() > RoadSettings.Get().max_vehicles) {
				road_department.Disable(2 * GameTime.YEAR);
			}
		}

		local init_engine = this.GetBestEngineToBuy();
		NewRoadTradeRouteEvent.Fire(this);
	}

	function Close()
	{
		if (!(::TradeRoute.Close())) return false;
		this.road_route.trade_routes.RemoveItem(this.GetID());
		return true;
	}

	function GetCargoID()
	{
		return this.road_route.cargo_id;
	}

	function GetManageInterval(n)
	{
		return GameTime.MONTH * (n == 0 ? 4 : 2);
	}

	function GetOneVehicleProfit()
	{
		return this.road_route.vehicle_profit;
	}

	function IsSaturated()
	{
		local vehicles_count = this.GetVehicles().Count();
		if (this.vehicles_required - vehicles_count <= 1) return true;
		return !(this.s_terminal.GetFreeSheduleSlots(this.GetFrequency()) <= 2);
	}

/* protected */
	function doBuyVehicle()
	{
		local old_road_type = RoadUtils.SetRoadType(AIRoad.ROADTYPE_ROAD);
		local s_depot = this.s_terminal.depot.location;
		local e_depot = this.e_terminal.depot.location;
		if (!AIRoad.IsRoadDepotTile(s_depot) || !AIRoad.IsRoadDepotTile(e_depot)) {
			Corporation.Get().AddAction(FixRoadAction(this));
			return -1;
		}

		local e = this.GetBestEngineToBuy();
		local n = this.vehicles.Count();
		local is_one_way = this.road_route.IsOneWay();
		local load_flag = is_one_way || n % 2 == 0;
		local depot_tile = load_flag ? s_depot : e_depot;
		local v = AIVehicle.BuildVehicle(depot_tile, e);
		if (!AIVehicle.IsValidVehicle(v)) {
			if (AIError.GetLastError() == AIVehicle.ERR_VEHICLE_TOO_MANY) {
				RoadSettings.Get().max_vehicles--;
				Terron.UpdateMemento();
			}
			return -1;
		}

		if (AIVehicle.GetLength(v) > 32) {
			RoadEngineRoster.Get().ProhibitEngine(e);
			this.engine.ForceUpdate();
			return -2;
		}

		local unload_flag = !this.road_route.IsTransit() ? AIOrder.OF_UNLOAD :
			AIOrder.OF_TRANSFER;

		local s_load_strategy = load_flag ? AIOrder.OF_FULL_LOAD_ANY : AIOrder.OF_NONE;
		local e_load_strategy = !load_flag ? AIOrder.OF_FULL_LOAD_ANY :
			(is_one_way ? unload_flag | AIOrder.OF_NO_LOAD : AIOrder.OF_NONE);
		e_load_strategy = e_load_strategy | AIOrder.OF_NON_STOP_INTERMEDIATE;
		s_load_strategy = s_load_strategy | AIOrder.OF_NON_STOP_INTERMEDIATE;

		if (n > 0) {
			AIOrder.CopyOrders(v, this.vehicles.Begin());
			if (!load_flag) {
				AIOrder.SetOrderFlags(v, 0, s_load_strategy);
				AIOrder.SetOrderFlags(v, 2, e_load_strategy);
			}
			if (n % 2 == 0 && !is_one_way) {
				AIOrder.SetOrderFlags(v, 0, AIOrder.OF_FULL_LOAD_ANY);
				AIOrder.SetOrderFlags(v, 2, AIOrder.OF_NONE);
			}
		} else {
			AIOrder.AppendOrder(v, this.s_stop.location, s_load_strategy);
			if (is_one_way && this.len < 50) {
				AIOrder.AppendOrder(v, s_depot, AIOrder.OF_NON_STOP_INTERMEDIATE);
			} else {
				AIOrder.AppendOrder(v, s_depot, AIOrder.OF_NON_STOP_INTERMEDIATE);
			}
			AIOrder.AppendOrder(v, this.e_stop.location, e_load_strategy);
			AIOrder.AppendOrder(v, e_depot, AIOrder.OF_NON_STOP_INTERMEDIATE);
		}
		if (!load_flag) AIOrder.SkipToOrder(v, 2); 

		local f = this.GetFrequency();
		this.s_terminal.AddVehicle(v, f);
		this.e_terminal.AddVehicle(v, f);
		RoadUtils.SetRoadType(old_road_type);
		return v;
	}

	function doSellVehicle(v)
	{
		::TradeRoute.doSellVehicle(v);

		this.s_terminal.RemoveVehicle(v);
		this.e_terminal.RemoveVehicle(v);
	}

	/**
	 * Get most suitable engine for this trade route.
	 * @return Currently best engine id for this trade route.
	 */
	function doGetEngine()
	{
		local c = this.GetCargoID();
		local e = RoadEngineRoster.Get().GetBestEngine(c, !this.use_articulated);
		if (AIEngine.IsBuildable(e)) {
			this.frequency = RoadUtils.GetFrequency(e, this.len);
		} else {
			this.frequency = 1; // value doesn't matter, just not null is good
			this.Disable(GameTime.YEAR + GameTime.DAY);
		}

		return e;
	}

	function GetFrequency()
	{
		return this.frequency;
	}

/* private */
	/** Base road route for this trade route */
	road_route = null;

	/** Either AIStation.STATION_TRUCK_STOP or AIStation.STATION_BUS_STOP */
	station_type = null;

	/** First road staion stop - trade route "begin" station */
	s_stop = null;

	/** Second road station stop - trade route "end" station */
	e_stop = null;

	/** "Begin" station's terminal that handles this trade route */
	s_terminal = null;

	/** "End" station's terminal that handles this trade route */
	e_terminal = null;

	/** Flag, defining if this trade route can use articulated vehicles */
	use_articulated = true;

	/** The arrival frequency for this trade route best engine */
	frequency = 1;

	/** Length */
	len = null;

	/** Optimal number of vehicles required by this trade route */
	vehicles_required = 0;

	/** Date of last vehicle number check */
	vehicles_update_date = null;

	/** Helper date to avoid checking the same thing for nth times in a row */
	integrity_check_date = null;

	/**
	 * Find vehicles belonging to this trade route and properly save information
	 *  about each one.<p>
	 * Needed after loading.
	 */
	function ReloadOwnVehicles()
	{
		local c = this.road_route.GetCargoID();
		local s_list = this.s_stop.GetVehicles();
		local e_list = this.e_stop.GetVehicles();
		local sell_name = SellLostVehiclesTask.special_sell_name;
		local n = sell_name.len();
		foreach (v, dummy in e_list) {
			if (s_list.HasItem(v) && AIVehicle.IsValidVehicle(v)) {
				if (AIVehicle.GetState(v) == AIVehicle.VS_CRASHED) continue;
				if (AIVehicle.GetCapacity(v, c) == 0) continue;
				local v_name = AIVehicle.GetName(v); 
				if (v_name.len() >= n) {
					v_name = v_name.slice(0, n);
					if (v_name == sell_name) continue;
				}

				local e = AIVehicle.GetEngineType(v);
				local f = RoadUtils.GetFrequency(e, this.len);

				this.s_terminal.AddVehicle(v, f);
				this.e_terminal.AddVehicle(v, f);
				this.vehicles.AddItem(v, v);
			}
		}
		this.vehicles_update_date = AIDate.GetCurrentDate();
	}

/* public */
	/** Crashed vehicles handling */
	function OnRoadVehicleCrash(v)
	{
		if (!this.vehicles.HasItem(v)) return;
		this.vehicles.RemoveItem(v);
		this.s_terminal.RemoveVehicle(v);
		this.e_terminal.RemoveVehicle(v);
	}

	/** "Force" sell vehicle */
	function OnForceSellRoadVehicle(v)
	{
		if (!this.vehicles.HasItem(v)) return;
		this.SendSpecifiedVehiclesToSell([v]);
	}

	/** Unprofitable vehicles handling */
	function OnRoadVehicleUnprofitable(v)
	{
		if (!this.vehicles.HasItem(v)) return;
		if (AIVehicle.GetAge(v) <= 3 * GameTime.YEAR / 2) return;
		if (!this.CheckIntegrity()) {
			Corporation.Get().AddAction(FixRoadAction(this));
		}
		this.SendSpecifiedVehiclesToSell([v]);
	}

	/** "Stuck" vehicles handling */
	function OnRoadVehicleStuck(v)
	{
		if (!this.vehicles.HasItem(v)) return;
		if (!this.CheckIntegrity()) {
			Corporation.Get().AddAction(FixRoadAction(this));
		}
	}
}

function RoadTradeRoute::SendVehiclesToSell(n)
{
	if (n >= this.vehicles.Count() || !this.road_route.IsOneWay()) {
		return ::TradeRoute.SendVehiclesToSell(n);
	}

	if (n > 1) {
		CodeUtils.Log("Selling " + n + " vehicles [" + this.name + "]", 2);
	} else if (n == 1) {
		CodeUtils.Log("Selling a vehicle [" + this.name + "]", 2);
	} else {
		return 0 //n <= 0
	}

	this.vehicles.Valuate(AIVehicle.GetCargoLoad, this.road_route.cargo_id);
	local to_sell = [];
	foreach (v, load in this.vehicles) {
		if (n <= 0) break;
		if (load == 0) {
			to_sell.append(v);
			n--;
		}
	}

	foreach (dummy_id, v in to_sell) {
		this.doSellVehicle(v);
	}

	return 0;
}

/** Check if stations are connected, all roads ok, and depots in place */
function RoadTradeRoute::CheckIntegrity()
{
	local current_date = AIDate.GetCurrentDate();
	if (current_date - this.integrity_check_date <= GameTime.MONTH) return true;
	this.integrity_check_date = current_date;
	CodeUtils.Log("Checking " + this.GetName() + " integrity...", 1);

	local s_arr = this.s_terminal.core.GetEntrances();
	local e_arr = this.e_terminal.core.GetEntrances();
	local pf = RoadUtils.GetAccuratePF();
	if (!pf.AreTileSetsConnected(s_arr, e_arr, AIRoad.ROADTYPE_ROAD)) {
		CodeUtils.Log("Stations connection damaged! (" + this.GetName() + ")", 2);
		return false;
	}
	if (this.s_stop.CheckDepots() && this.e_stop.CheckDepots()) {
		CodeUtils.Log("...everything seems normal.", 1);
		return true;
	}

	CodeUtils.Log("Road to depot is damaged! (" + this.GetName() + ")", 2);
	return false;
}

function RoadTradeRoute::EstimateVehiclesNeeds()
{
	local vehicles_count = this.GetVehicles().Count();
	local current_date = AIDate.GetCurrentDate();
	local dt = current_date - this.vehicles_update_date;
	if (dt < GameTime.MONTH) {
		return max(this.vehicles_required - vehicles_count, 0);
	}

	local s = this.s_terminal;
	local e = this.e_terminal;
	local c = this.GetCargoID();

	/* handle situation when cargo do not disappear in delivery station */
	local waiting_at_end = AIStation.GetCargoWaiting(e.station_id, c);
	if (this.road_route.one_way && waiting_at_end > 10) {
		local should_stop = true;
		if (this.road_route.IsTransit()) {
			local rating = AIStation.GetCargoRating(e.station_id, c);
			if (waiting_at_end < 800 && rating > 40) should_stop = false;
		} else {
			foreach (dummy_id, t in e.core.tiles) {
				if (AITile.GetCargoAcceptance(t, c, 1, 1, 3) >= 8) should_stop = false;
			}
		}
		if (should_stop) {
			this.Disable(4 * GameTime.MONTH);
			return -vehicles_count;
		}
	}

	local f = this.GetFrequency();
	local t_rate = this.road_route.transport_rate;
	local s_free = s.GetFreeSheduleSlots(f);
	local e_free = e.GetFreeSheduleSlots(f);
	local n = max(-vehicles_count, max(-5, min(s_free, e_free)));

	this.vehicles_update_date = current_date;
	local engine_id = this.GetBestEngineToBuy();
	if (!AIEngine.IsBuildable(engine_id)) return min(n, 0);

	/* Sell old, or obsolete vehicles */
	if (AIDate.GetYear(current_date) % 3 == this.GetID() % 3) {
		local to_sell = [];

		this.vehicles.Valuate(AIVehicle.GetAgeLeft);
		foreach (v, age_left in this.vehicles) {
			if ((!AIVehicle.IsValidVehicle(v)) || age_left < 3 * GameTime.YEAR) {
				to_sell.append(v);
			}
		}
		if (to_sell.len() > 0) {
			this.SendSpecifiedVehiclesToSell(to_sell);
			this.vehicles_required = this.vehicles.Count();
			return 0;
		}

		this.vehicles.Valuate(AIVehicle.GetAge);
		foreach (v, age in this.vehicles) {
			if (age < 3 * GameTime.YEAR) continue;
			if (age > 9 * GameTime.YEAR ||
			(engine_id != AIVehicle.GetEngineType(v) && age > 7 * GameTime.YEAR)) {
				to_sell.append(v);
			}
		}
		if (to_sell.len() >= 2) {
			this.SendSpecifiedVehiclesToSell(to_sell);
			this.vehicles_required = this.GetVehicles().Count() + to_sell.len() / 2;
			return to_sell.len() / 2;
		}
	}

	local can_expand_s = s.CanBeExpanded(c);
	local can_expand_e = e.CanBeExpanded(c);

	if (s_free < 4 && can_expand_s) {
		Corporation.Get().AddAction(AIExpandRoadStationAction(s));
	}
	if (e_free < 4 && can_expand_e) {
		Corporation.Get().AddAction(AIExpandRoadStationAction(e));
	}

	if ((s_free < 4 || e_free < 4) && ((!can_expand_s) || (!can_expand_e))) {
		if (this.road_route.max_level < 1 && !this.road_route.IsTransit()) {
			/*
			 * will cause dublicate trade route construction
			 *  except transit routes
			 */
			this.road_route.max_level++;
			AdditionalRoadTradeWantedEvent.Fire(this.road_route);
			return 0; 
		}
	}

	n = min(n, s.EstimateVehiclesNeeds(t_rate));
	if (n >= 0 && !this.road_route.IsOneWay()) {
		n = min(n, e.EstimateVehiclesNeeds(t_rate));
	} else if (n < 0 && (-n > vehicles_count)) {
		this.vehicles_required = 0;
		return -vehicles_count;
	} else if (n < 0) {
		local at_start = AIStation.GetCargoWaiting(s.station_id, c);
		if (at_start > 300 && (this.road_route.one_way || waiting_at_end > 300)) {
			n = 0;
		}
	}

	local n = max(min(n, 5), -3);
	this.vehicles_required = vehicles_count + n;
	return n; 
}
