/**
 * Class that handles road transit routes construction.
 * Main purpose is to make road transit automatic and trivial task for
 *  other AI parts.
 */
class RoadTransitHandler extends AbstractTransportSystem
{
/* public */
	static function GetClassName()
	{
		return "RoadTransitHandler";
	}

	/**
	 * Get the instance of the RoadDepartment singleton.
	 */
	static function Get()
	{
		if (RoadTransitHandler._instance[0] == null) {
			RoadTransitHandler();
		}
		return RoadTransitHandler._instance[0];
	}

	static function Restore(memento)
	{
		return RoadTransitHandler.Get();
	}

	function GetName()
	{
		return "Road Transit Controller";
	}

	/**
	 * Recheck transit possibilities when new road trade route is spotted.
	 */
	function OnNewRoadTradeRoute(trade_route)
	{
		local s = trade_route.road_route.GetStart();
		if (s.GetClassName() != HeapNode.GetClassName()) return;

		local c = trade_route.GetCargoID();
		local global = GlobalTransportMap.Get();
		foreach (class_name, class_nodes in global.structured_transport_nodes) {
			if (!(s.type_id in class_nodes)) continue;
			local type_nodes = class_nodes[s.type_id];
			if (!(s.node_id in type_nodes)) continue;
			foreach (dummy_link_id, link in type_nodes[s.node_id].links_in) {
				foreach (dummy_route_id, route in link.routes) {
					if (route.GetCargoID() == c && route.IsNefarious()) {
						route.UnmakeNefarious();
					}
				}
			}
		}

		this.best_action.ForceUpdate();
	}

	/**
	 * Allows transit trade construction to gather resources "scattered" within
	 *  given heap node. 
	 * @param transport_node Heap node.
	 * @param station_id ID of the station which must gather cargo from all
	 *  sub nodes.
	 * @param cargo_id ID of the cargo to gather.
	 */
	function EnableHeapNodeGathering(transport_node, station_id, cargo_id)
	{
		local raw = transport_node.node;
		if (raw.GetClassName() != HeapNode.GetClassName()) return;

		raw.CreateSubRoutes(transport_node, station_id, cargo_id);

		this.to_feed_transport_nodes[transport_node.GetID()] <- transport_node;
	}

/* protected */
	function doGetBestAction()
	{
		return this.best_action;
	}

/* private */
	/** Single instance of this class */
	static _instance = [null];

	/** Best action for the RoadTransitHandler */
	best_action = null;

	/**
	 * Table with transport nodes which need to be supplied by transit routes.
	 */
	to_feed_transport_nodes = null;

	/**
	 * Private constructor for RoadTransitHandler singleton.
	 */
	constructor()
	{
		::AbstractTransportSystem.constructor();
		if (RoadTransitHandler._instance[0] != null) assert(null);
		RoadTransitHandler._instance[0] = this;

		this.to_feed_transport_nodes = {};

		local a_name = "Building road transit route..."
		local strategy = TransitRoutesConstructionStrategy(this.to_feed_transport_nodes);
		local t = 22 * GameTime.DAY;
		this.best_action = UniversalAction(a_name, strategy, 4 * t, t);

		NewRoadTradeRouteEvent.AddListener(this);
	}
}

Terron_ClassTable.RegisterClass(RoadTransitHandler);
