/**
 * Helpers for the AI road part.
 */
class RoadUtils
{
/* public */
	/**
	 * Get station type suitable to handle specified cargo.
	 * @param cargo_id Cargo ID.
	 * @return Either AIStation.STATION_BUS_STOP or AIStation.STATION_TRUCK_STOP
	 */
	function GetStationType(cargo_id)
	{
		return AICargo.HasCargoClass(cargo_id, AICargo.CC_PASSENGERS) ?
			AIStation.STATION_BUS_STOP : AIStation.STATION_TRUCK_STOP;
	}

	/**
	 * Get road pathfinder considered to be most accurate of all.
	 * @return Road pathfinder adapter.
	 */
	function GetAccuratePF()
	{
		return SimplePF_Adapter();
	}

	/**
	 * Get vehicles station arriving frequency, for specified distance between
	 *  stations.<p>
	 * @param e Vehicle engine ID.
	 * @param one_way_len Distance between two stations, let's say A and B.
	 * @return 1 / ([travel time from A to B] + [travel time from B to A])
	 */
	static function GetFrequency(e, one_way_len)
	{
		if (one_way_len < 24) one_way_len = 24 + one_way_len;

		//return 1.0 / (8 + 668 * 2 * one_way_len / (24 * 0.9 * AIEngine.GetMaxSpeed(e)));
		//same as upper comment
		return 1.0 / (8 + 62 * one_way_len / AIEngine.GetMaxSpeed(e));
	}

	/**
	 * Get list of suitable tiles for road station.
	 * @param node Node where we want new station.
	 * @param cargo_id Cargo to handle.
	 * @param is_load_station True when we must transport cargo from station,
	 *  false else.
	 */
	static function GetPossibleStationLocations(node, cargo_id, is_load_station)
	{
		local r = AIStation.GetCoverageRadius(RoadUtils.GetStationType(cargo_id)); 
		local list = is_load_station ?
			node.AITileList_NodeProducing(r, cargo_id) :
			node.AITileList_NodeAccepting(r, cargo_id);

		list.Valuate(AITile.IsBuildable);
		list.RemoveValue(0);

		list.Valuate(AITile.IsWaterTile);
		list.KeepValue(0);

		if (node.type_id != NodeTypeID.NT_TRANSIT_DROP) {
			list.Valuate(AITile.GetSlope);
			list.KeepValue(AITile.SLOPE_FLAT);
		}
		return list;
	}

	/**
	 * Wrapper for the AIRoad.SetCurrentRoadType API function.
	 * @return Previos roadtype.
	 */
	static function SetRoadType(new_road_type)
	{
		local old_type = AIRoad.GetCurrentRoadType()
		AIRoad.SetCurrentRoadType(new_road_type);
		return old_type;
	}
}
