/**
 * Class that handles [road station terminal]-[road depot] link.
 */
class RoadStationDepot extends Terron_Object
{
/* public */
	/** Tile ID of this depot */
	location = -1;

	/** Reference to core tiles of the host terminal */
	terminal_core = null;

	/**
	 * RoadStationDepot class constructor.
	 * @param station_id ID of road station associated with new depot object.
	 */
	constructor(station_id, terminal_core)
	{
		::Terron_Object.constructor(null);

		this.station_id = station_id;
		this.terminal_core = terminal_core;
	}

	function Assign(depot_tile)
	{
		this.location = depot_tile;
		Terron.UpdateMemento();
	}

	/**
	 * Build a new road depot near this station.
	 * @return 0 if depot was built, -1 if build failed.
	 */
	function Build()
	{
		local s = AIStation.GetName(this.station_id);
		CodeUtils.Log("Building a depot near " + s + "...", 1);

		local entrances = this.terminal_core.GetEntrances();
		local t = entrances.len() > 0 ? entrances[0] : -1;
		if (t == -1) {
			entrances = this.terminal_core.GetExits();
			t = entrances.len() > 0 ? entrances[0] : -1;
		}

		if (t == -1) {
			/* use any other terminals tiles as own */
			local old_type = RoadUtils.SetRoadType(AIRoad.ROADTYPE_ROAD);

			local list = AITileList_StationType(
				this.station_id, AIStation.STATION_TRUCK_STOP
			);
			list.AddList(AITileList_StationType(
				this.station_id, AIStation.STATION_BUS_STOP
			));

			list.Valuate(AIRoad.GetRoadStationFrontTile);
			foreach (t, f in list) {
				if (AIRoad.AreRoadTilesConnected(t, f)) entrances.append(f);
				if (!AIRoad.IsDriveThroughRoadStationTile(t)) continue;

				local b = AIRoad.GetDriveThroughBackTile(t);
				if (AIRoad.AreRoadTilesConnected(t, b)) entrances.append(b);
			}

			AIRoad.SetCurrentRoadType(old_type);
 
			t = entrances.len() > 0 ? entrances[0] : -1;
		}

		if (t == -1) {
			CodeUtils.Log("Invalid station entrace/exit data " + s, 2);
			return -1;
		}

		this.location = this.doBuildDepot(t, 25, 8);
		if (this.location == -1) {
			AIController.Sleep(10 * Ticks.DAY);
			this.location = this.doBuildDepot(t, 25, 9);
		}

		if (this.location <= -1) {
			CodeUtils.Log("Failed to build a depot for " + s, 2);
		} else {
			CodeUtils.Log("...built", 1);
			Terron.UpdateMemento();
		}

		return this.location != -1 ? 0 : -1;
	}

	/**
	 * Check if depot is in place and connected to station.
	 * @return True if check passed, false otherwise.
	 */
	function Check()
	{
		local old_road_type = RoadUtils.SetRoadType(AIRoad.ROADTYPE_ROAD)

		local s = AIStation.GetName(this.station_id);
		CodeUtils.Log("Checking depot near " + s + "...", 1);

		local t = this.location;
		if (t == -1 || !AIRoad.IsRoadDepotTile(t)) {
			CodeUtils.Log("     deopt not found", 1);
			this.Build();
			return AIRoad.IsRoadDepotTile(this.location);
		}

		local front = AIRoad.GetRoadDepotFrontTile(t);
		if (!AIRoad.AreRoadTilesConnected(t, front)) {
			AIRoad.SetCurrentRoadType(old_road_type);
			CodeUtils.Log("     deopt front tile is damaged...", 1);
			return false;
		}

		local e_arr = this.terminal_core.GetEntrances();
		if (e_arr.len() == 0) {
			AIRoad.SetCurrentRoadType(old_road_type);
			// it's more safe to return true here
			// AI can have absolutely valid 0 tile stations, and
			//  if problems arise with such objects => code responsible for
			//  their creation must handle it
			return true;
			//return false;
		}

		local pf = RoadUtils.GetAccuratePF();
		if (!pf.AreTilesConnected(front, e_arr[0], AIRoad.ROADTYPE_ROAD)) {
			CodeUtils.Log("     depot is not connected to station..." , 1);
			pf = RoadUtils.GetAccuratePF();
			/*foreach (dummy_id, t in e_arr) {
				AISign.BuildSign(t, "e_arr" + dummy_id);
			}
			AISign.BuildSign(front, "front");*/

			if (pf.ConnectTiles(front, e_arr[0], AIRoad.ROADTYPE_ROAD) != 0) {
				AIRoad.SetCurrentRoadType(old_road_type);
				CodeUtils.Log("     connection attempt failed!", 1);
				return false;
			}
			CodeUtils.Log("     connection attempt succeed!", 1);
		}

		CodeUtils.Log("Depot for " + s + " check complete, location is valid", 2);
		AIRoad.SetCurrentRoadType(old_road_type);
		return true;
	}

/* private */
	/** Associated station id */
	station_id = null;
}

/**
 * Depot building implementation.
 * @param start_tile Supposed to be a road tile.
 * @param steps Max distance(tunnels and bridges counted as one tile, though)
 *  allowed between the depot and the start_tile.
 * @param r Distance from start_tile within which AI must find and use already
 *  existing depot(if it exist).
 * @return Tile ID of built depot, -1 if failed to build one. 
 */
/* private */ function RoadStationDepot::doBuildDepot(start_tile, steps, r)
{
	local t_existing = this.doFindDepot(start_tile, r);
	if (t_existing != -1 && AIMap.IsValidTile(t_existing)) return t_existing;

	local ti = AIMap.GetTileIndex;
	local offsets = [ti(-1, 0), ti(0, 1), ti(1, 0), ti(0, -1)];
	local list = AITileList();
	local expand_list = AITileList();
	local old_type = RoadUtils.SetRoadType(AIRoad.ROADTYPE_ROAD);

	list.AddTile(start_tile);

	/* Add tiles connected to start into special list */
	for (local i = 0; i < steps; i++, expand_list.Clear()) {
		foreach (t1, dummy_value in list) {
			/* foreach unchecked item in list add its road neighbour into expand_list */
			foreach (dummy_id, offset in offsets) {
				local t2 = t1 - offset;

				if (!AIRoad.AreRoadTilesConnected(t1, t2)) continue;
				if (list.HasItem(t2)) continue;
				if (AIRoad.IsRoadStationTile(t2)) {
					if (!AIRoad.IsDriveThroughRoadStationTile(t2)) continue;
				}
				if (AIBridge.IsBridgeTile(t2)) {
					expand_list.AddTile(AIBridge.GetOtherBridgeEnd(t2));
					continue;
				}
				if (AITunnel.IsTunnelTile(t2)) {
					expand_list.AddTile(AITunnel.GetOtherTunnelEnd(t2));
					continue;
				}

				expand_list.AddTile(t2);
				if (AIRoad.IsRoadDepotTile(t2)) {
					if (!AIRoad.AreRoadTilesConnected(t1, t2)) continue;
					if (AICompany.IsMine(AITile.GetOwner(t2))) {
						RoadUtils.SetRoadType(old_type);
						return t2;
					}
				}
			}
		}

		if (i < 5) {
			local new_tile_found = false;
			foreach (t, dummy in expand_list) {
				if (list.HasItem(t)) continue;
				new_tile_found = true;
				list.AddTile(t);
			}
			if (!new_tile_found) return -3;
			continue;
		}

		foreach (t1, dummy in expand_list) {
			if (AIBridge.IsBridgeTile(t1)) continue;
			if (AITunnel.IsTunnelTile(t1)) continue;
			/* Try to build depot if we need one and there is suitable tile in expand_list */
			foreach (dummy_offset_id, offset in offsets) {
				local t2 = t1 - offset;
				/* Don't build too close to a station */
				if (AIMap.DistanceManhattan(start_tile, t2) < 6) continue;
				if (AITile.GetMaxHeight(t2) > AITile.GetMaxHeight(t1)) continue;
				if (!AITile.IsBuildable(t2)) continue;
				/* Don't build when location tile is 3/4 water */
				if (AITile.GetMinHeight(t2) == 0) {
					local s = AITile.GetSlope(t2);
					if (s == AITile.SLOPE_N || s == AITile.SLOPE_W) continue;
					if (s == AITile.SLOPE_S || s == AITile.SLOPE_E) continue;
				}

				local need_road = (!AIRoad.AreRoadTilesConnected(t2, t1));
				{
					local test_mode = AITestMode;
					if (need_road) {
					 	if (!AIRoad.BuildRoad(t2, t1)) continue;
					}
					if (!AIRoad.BuildRoadDepot(t2, t1)) continue;
				}

				if (need_road) AIRoad.BuildRoad(t2, t1);
				AIRoad.BuildRoadDepot(t2, t1);
				if (AIRoad.IsRoadDepotTile(t2)) {
					if (AICompany.IsMine(AITile.GetOwner(t2))) {
						RoadUtils.SetRoadType(old_type);
						return t2;
					}
				}
			}
		}

		foreach (t, dummy in expand_list) list.AddTile(t);
	}

	RoadUtils.SetRoadType(old_type);
	return -1;
}

/* private */ function RoadStationDepot::doFindDepot(start_tile, r)
{
	local ti = AIMap.GetTileIndex;
	local offsets = [ti(-1, 0), ti(0, 1), ti(1, 0), ti(0, -1)];
	local list = AITileList();
	local expand_list = AITileList();
	local old_type = RoadUtils.SetRoadType(AIRoad.ROADTYPE_ROAD);

	list.AddTile(start_tile);

	/* Add tiles connected to start into special list */
	for (local i = 0; i < r; i++, expand_list.Clear()) {
		foreach (t1, dummy_value in list) {
			/* foreach unchecked item in list add its road neighbour into expand_list */
			foreach (dummy_id, offset in offsets) {
				local t2 = t1 - offset;

				if (!AIRoad.AreRoadTilesConnected(t1, t2)) continue;
				if (list.HasItem(t2)) continue;
				if (AIRoad.IsRoadStationTile(t2)) {
					if (!AIRoad.IsDriveThroughRoadStationTile(t2)) continue;
				}
				if (AIBridge.IsBridgeTile(t2)) {
					expand_list.AddTile(AIBridge.GetOtherBridgeEnd(t2));
					continue;
				}
				if (AITunnel.IsTunnelTile(t2)) {
					expand_list.AddTile(AITunnel.GetOtherTunnelEnd(t2));
					continue;
				}

				expand_list.AddTile(t2);
				if (AIRoad.IsRoadDepotTile(t2)) {
					if (!AIRoad.AreRoadTilesConnected(t1, t2)) continue;
					if (AICompany.IsMine(AITile.GetOwner(t2))) {
						RoadUtils.SetRoadType(old_type);
						return t2;
					}
				}
			}
		}

		foreach (t, dummy in expand_list) list.AddTile(t);
	}

	RoadUtils.SetRoadType(old_type);
	return -1;
}
