/**
 * Class that handles common tasks for different freight transport approaches.
 */
class FreightTradeOverseer extends AbstractTransportSystem
{
/* public */
	static function GetClassName()
	{
		return "FreightTradeOverseer";
	}

	/**
	 * FreightTradeOverseer constructor.
	 * @param transport_nodes_class_name_to_oversee Class name to watch for.
	 * @param enable_stealing Defines whether routes staeling should be on.
	 */
	constructor(transport_nodes_class_name_to_oversee, enable_stealing)
	{
		::AbstractTransportSystem.constructor();

		local c_name = transport_nodes_class_name_to_oversee;
		this.target_class_name = c_name;
		GlobalTransportMap.Get().SetUpNewTransportNodeClass(c_name);

		NewRoadTradeRouteEvent.AddListener(this);
		Terron_Event.IndustryClosed.AddListener(this);

		if (enable_stealing) {
			local types = GetNodeTypesToHandle();
			local cfg = RoadSettings.Get();
			local min_len = cfg.min_route_length;
			min_len += (cfg.max_route_length - cfg.min_route_length) / 3;
			local fx = function(x) : (min_len) { return x.length > min_len;}
			this.AddSubsystem(RoadsStealingSection(c_name, types, fx));
		}

		AIStartedEvent.AddListener(this);
	}

	function GetName()
	{
		return this.target_class_name + " overseer";
	}

	/**
	 * Restore own trade lines after game load.
	 */
	function OnAIStarted(is_game_loaded)
	{
		if (is_game_loaded) {
			local a = AISearchRoadTradeRoutesAction(target_class_name);
			Corporation.Get().AddAction(a);
		}
	}

/* private */
	/** String, representing the class name of transport nodes to deal with */
	target_class_name = null;

	/**
	 * Close corresponding transport system when industry collapses.
	 * @param industry_id Just closed industry ID.
	 */
	function OnIndustryClose(industry_id)
	{
		local type_id = AIIndustry.GetIndustryType(industry_id);
		local map = GlobalTransportMap.Get();
		map.RemoveTransportNodeEx(this.target_class_name, type_id, industry_id);
	}

	/**
	 * Recolor freight links when new trade appears.
	 * @trade_route Newly founded trade route.
	 */
	function OnNewRoadTradeRoute(trade_route)
	{
		/* Get "plain" nodes here */
		local s = trade_route.road_route.GetStart();
		local e = trade_route.road_route.GetEnd();

		/* And receive corresponding transport nodes */
		local tmp = GlobalTransportMap.Get().structured_transport_nodes;
		local all_road_nodes = tmp[this.target_class_name];

		local tmp_type_specific = all_road_nodes[s.GetTypeID()];
		if (!(s.node_id in tmp_type_specific)) return;
		s = tmp_type_specific[s.node_id];
		//if (s.GetClassName() != this.target_class_name) return;

		local tmp_type_specific = all_road_nodes[e.GetTypeID()];
		if (!(e.node_id in tmp_type_specific)) return;
		e = tmp_type_specific[e.node_id];
		//if (e.GetClassName() != this.target_class_name) return;

		if (!(e.GetID() in s.links_out)) return;

		/* Update corresponding links colors */
		// the one with trade should be yellow
		s.links_out[e.GetID()].SetColor(LinkColor.LC_YELLOW);
		if (s.node.GetClassName() == HeapNode.GetClassName()) {
			foreach (dummy_id, link in s.links_out) {
				if (link.color != LinkColor.LC_GREEN) continue;
				foreach (dummy_route_id, route in link.routes) {
					if (route.GetCargoID() == trade_route.GetCargoID()) {
						link.SetColor(LinkColor.LC_BLUE);
					}
				}
			}
		}

		// and set to green links which is start where the trade route ends
		local cc = {};
		foreach (dummy_link_id, link in e.links_in) {
			if (link.color == LinkColor.LC_YELLOW) {
				foreach (dummy_route_id, route in link.routes) {
					cc[route.GetCargoID()] <- 1;
				}
			}
		}

		local c_out = e.node.GetType().GetOutput(cc);
		foreach (dummy_id, link in e.links_out) {
			if (link.color != LinkColor.LC_RED) continue;
			foreach (dummy_route_id, route in link.routes) {
				if (route.GetCargoID() in c_out.output_result) {
					link.SetColor(LinkColor.LC_GREEN);
					break;
				}
			}
		}
	}

	/**
	 * Define what we'll want to "steal".
	 * @return Table with node types that must be scanned for opponents present.
	 *  If the node type is not in this table => AI will be unable to find
	 *  and use/steal roads build by other players near nodes of ths type.
	 */
	function GetNodeTypesToHandle()
	{
		local types = TransportSchema.Get().node_types;

		local tmp = {};
		local result = {};
		foreach (type_id, type in types) {
			if (type_id == NodeTypeID.NT_RAW_TOWN || type.raw_production) {
				tmp[type_id] <- type;
				result[type_id] <- type;
			} 
		}

		foreach (type_id, type in tmp) {
			foreach (supplier_type_id, dummy in type.supply_tree.tree) {
				result[supplier_type_id] <- types[supplier_type_id];
			}
			foreach (helper_type_id, dummy in type.consumer_node_types) {
				result[helper_type_id] <- types[helper_type_id];
			}
			foreach (consumer_type_id, consumer in type.consumer_node_types) {
				result[consumer_type_id] <- consumer;
			}
		}
		return result;
	}
}
