/**
 * Class that controls build process for the DumbTruckControlSection.
 */
class DumbTruckRoutesConstructionStrategy extends DefaultRoadConstructionStrategy
{
/* public */
	function SelectObjectToBuild()
	{
		local best_route = null;
		local best_rank = APriority.BAD;

		local links = GlobalTransportMap.Get().link_map[TruckHub.GetClassName()];
		/* Just greedy select route with highest rank */
		local good_colors = [LinkColor.LC_GREEN, LinkColor.LC_YELLOW];
		foreach (m, color in good_colors) {
			foreach (link_id, link in links[color]) {
				local n = link.routes.len();
				// this is just minor improvment to not ignore cases when
				//  there is a 2 or more cargo types to be delivered
				// e.g. both LVST and GRAI goes to factory in default setup,
				//  but we'd completely ignore farms without this snippet due to
				// fact that LVST and GRAI are much cheaper than e.g COAL.
				//  Now it's sum of LVST and GRAI profit vs COAL, so it's fair
				//   comprasion.

				// ignore links with 0 routes and yellow links with 1 route
				if (n > m) {
					local r = this.RankRoadRoutes(link.routes);
					if (r <= best_rank) continue;

					local best_route_rank = APriority.BAD;
					foreach (dummy_route_id, route in link.routes) { 
						local r2 = (n > 1) ? this.RankRoadRoute(route) : r;
						if (r2 > best_route_rank) {
							best_rank = r;
							best_route_rank = r2;
							best_route = route;
						}
					}
				}
			}
		}

		return {priority = best_rank, context = best_route};
	}

	function Build(route)
	{
		if (RoadSettings.Get().accurate_construction) {
			if (route.connection.state == ConnectionState.CS_ACCURATE_COST_UNKNOWN) {
				this.CalculateConnectionCost(route);
				return {err_code = 1, s_station = null, e_station = null};
			}
		}

		if (route.connection.state == ConnectionState.CS_ROAD_BUILT) {
			this.CalculateConnectionCost(route);
			route.connection.state = ConnectionState.CS_ACCURATE_COST_CALCULATED;
			this.double_checked_routes[route.GetID()] <- 1;
			return {err_code = 1, s_station = null, e_station = null};
		}

		local pf = this.GetDefaultPathfinder();
		if (route.GetID() in this.double_checked_routes) {
			pf = SimplePF_RoadsReuseAdapter();
		}

		return RoadRouteBuilder.BuildRoute(route, pf);
	}

	function HandleResult(result, route)
	{
		local err_code = result.err_code;
		/* Building failed, stop future retrying then... */
		if (err_code == -3 || err_code == -1) {
			// ...at least for year
			route.MakeNefarious(GameTime.YEAR);
			result.clear();
			return -1;
		}

		if (err_code == -2) {
			this.double_checked_routes[route.GetID()] <- 1;
		}

		/* Route complete */
		if (err_code == 0) {

			local s = route.GetStart();
			local tmp = GlobalTransportMap.Get().structured_transport_nodes;
			s = tmp[TruckHub.GetClassName()][s.GetTypeID()][s.node_id];

			if (s.node.GetClassName() == HeapNode.GetClassName()) {
				local c    = route.GetCargoID();
				local s_id = result.s_station.GetStationID();
				local ci   = route.normal_cargo_income;

				RoadTransitHandler.Get().EnableHeapNodeGathering(s, s_id, c);
				TransitIncomeCalculatedEvent.Fire(
					{station_id = s_id, cargo_id = c, cargo_income = ci}
				);
			} else if (RoadSettings.Get().accurate_construction) {
				Corporation.Get().AddAction(AISleepAction(AIDate.GetCurrentDate() + 45 * GameTime.DAY));
			}

			/* Start trade control */
			this.FoundNewTradeRoute(route, result.s_station, result.e_station);

			/* Suspend some routes building, to lazy renew cached values */
			foreach (dummy_link_id, link in s.links_out) {
				if (link.color != LinkColor.LC_GREEN) continue;
				foreach (dummy_route_id, to_suspend_route in link.routes) {
					to_suspend_route.MakeNefarious(67 * GameTime.DAY);
				}
			}
		}

		result.clear();
		return err_code;
	}

/* private */
	/**
	 * Stores routes with "almost existing" road.<p>
	 * E.g. with known another road nearby or abandoned during construction
	 *  due to lack of money.
	 */
	double_checked_routes = {};
}
