/**
 * Class that controls road network construction.
 */
class RoadNetworkConstructionStrategy extends DefaultRoadConstructionStrategy
{
/* public */
	/**
	 * Creates RoadNetworkConstructionStrategy object.
	 * @param road_network Road network to construct/handle.
	 */
	constructor(road_network)
	{
		this.road_network = road_network;
	}

	function SelectObjectToBuild()
	{
		local best_link = null;
		local best_rank = APriority.BAD;

		local used = RoadNetwork.nodes_already_in_networks;
		local this_name = this.road_network.GetName();
		local links = this.road_network.GetNetworkLinks();
		foreach (link_id, link in links) {
			local r = this.RankRoadRoutes(link.routes);
			if (r <= APriority.BAD) {
				foreach (dummy_route_id, route in link.routes) {
					if (route.IsNefarious() || route.IsCompletelyUpgraded()) {
						continue;
					}
					/*if (link.color == LinkColor.LC_GREEN ||
					link.color == LinkColor.LC_YELLOW || link.color == LinkColor.LC_BLUE) {
						if (route.GetStart().GetFreeProduction(route.GetCargoID()) > 0) r = 200;
					}*/
					//if (route.GetStart().type.raw_production) r = 100;
				}
			} else if (link.from_id in used) {
				foreach (dummy_id, info in used[link.from_id]) {
					if (info.net_id == this_name) continue;
					foreach (dummy_route_id, route in link.routes) {
						if (route.GetCargoID() == info.cargo_id) r = APriority.BAD;
					}
				}
			}

			if (r > best_rank) {
				best_link = link;
				best_rank = r;
			}
		}

		if (best_rank == APriority.BAD) {
			//AILog.Warning("NO GOOD LINKS" + this.road_network.GetName());
			this.road_network.Disable(GameTime.YEAR);
			//this.road_network.MarkAsUnprofitable();
		}/* else {
			AILog.Warning(this.road_network.GetName());
			foreach (dummy_route_id, route in best_link.routes) {
				AILog.Warning("[" + best_rank + "]" + route.GetName());
			}
		}*/

		return {priority = best_rank, context = best_link};
	}

	function Build(link)
	{
		if (!this.double_checked) {
			this.road_network.Refresh();
			this.double_checked = true;
			return [
				{err_code = 1, s_station = null, e_station = null, route = null}
			];
		}

		local build_results = [];
		foreach (dummy_id, route in link.routes) {
			if (route.IsCompletelyUpgraded()) continue;
			local pf = route.connection.state == ConnectionState.CS_ROAD_BUILT ?
				SimplePF_RoadsReuseAdapter() : this.GetDefaultPathfinder();

			local result = RoadRouteBuilder.BuildRoute(route, pf);
			build_results.append({
				err_code = result.err_code,
				s_station = result.s_station,
				e_station = result.e_station,
				route = route
			});
			if (result.err_code != 0) break;
		}

		return build_results;
	}

	function HandleResult(result, link_to_build)
	{
		foreach (dummy, sub_result in result) {
			local err_code = sub_result.err_code;

			/* Building failed, let the parent decide what to do */
			if (err_code == -3 || err_code == -1) {
				this.road_network.HandleBadLink(link_to_build);
				return -1;
			}

			if (err_code == 1) return 0;
			if (err_code != 0) return err_code;

			local route = sub_result.route;
			if (route.IsCompletelyUpgraded()) continue;
	
			local used = RoadNetwork.nodes_already_in_networks;
			if (!(link_to_build.from_id in used)) used[link_to_build.from_id] <- [];
			used[link_to_build.from_id].append({
				net_id = this.road_network.GetName(),
				cargo_id = route.GetCargoID()
			});
	
			local s = route.GetStart();
			local tmp = GlobalTransportMap.Get().structured_transport_nodes;
			s = tmp[TruckHub.GetClassName()][s.GetTypeID()][s.node_id];
			if (s.node.GetClassName() == HeapNode.GetClassName()) {
				local c    = route.GetCargoID();
				local s_id = sub_result.s_station.GetStationID();
				local ci   = route.normal_cargo_income;
	
				RoadTransitHandler.Get().EnableHeapNodeGathering(s, s_id, c);
				TransitIncomeCalculatedEvent.Fire(
					{station_id = s_id, cargo_id = c, cargo_income = ci}
				);
			}
	
			this.FoundNewTradeRoute(
				route, sub_result.s_station, sub_result.e_station
			);

			sub_result.clear();
		}

		return 0;
	}

	function RankRoadRoutes(routes)
	{
		local build_cost = 1, income = 0, full_cost = 1;
		foreach (dummy_id, route in routes) {
			if (route.IsNefarious() || route.IsCompletelyUpgraded()) continue;
			if (route.connection.state != ConnectionState.CS_ROAD_BUILT) {
				if (build_cost > 1) continue;
				build_cost = this.GetConstructionCost(route);
			}
			local i = route.GetEstimatedIncome();
			if (i <= 0) {
				local summary_production_incoming = 0;
				local s = route.GetStart();
				local links = this.road_network.GetNetworkLinks();
				foreach (link_id, link in links) {
					foreach (dummy_id, sub_route in link.routes) {
						if (sub_route.GetEnd() != s) continue;
						if (!sub_route.IsCompletelyUpgraded()) continue;
						local c = sub_route.GetCargoID();
						local p = sub_route.GetStart().GetLastMonthProduction(c);
						summary_production_incoming += p;
					}
				}
				summary_production_incoming = 0.75 * summary_production_incoming;
				i = route.GetEngine() == -1 ? 0 :
					(summary_production_incoming / route.transport_rate).tointeger();
			}
			income += i;
			full_cost += (build_cost + route.GetVehiclesCost());
		}
		return ActionPriority.CalculatePriority(build_cost, full_cost, 0, income);
	}

/* protected */
	function GetActionCost(link)
	{
		local station_cost = RoadSettings.Get().GetRoadStationCost();

		local result = 0
		foreach (dummy_id, route in link.routes) {
			result += station_cost + route.connection.cost + route.best_engine_price;
		}

		return result + station_cost;
	}

/* private */
	double_checked = false;

	road_network = null;

	transport_politics = null;
}
