/**
 * Here we delegate BusHubControlSection behaviour.
 */
class BusHubControlSection_LocalBuildStrategy extends DefaultRoadConstructionStrategy
{
/* public */
	/**
	 * Creates BusHubControlSection.LocalBuildStrategy object.
	 * @param bus_hub_control_section Object hold bus hubs as subsystems.
	 */
	constructor(bus_hub_control_section)
	{
		::DefaultRoadConstructionStrategy.constructor();
		/*
		 * This two lines is only initialization required for 'routes' fields.
		 * Further text explains why.
		 * ReshakeTowns task updates available routes and the current
		 *  "good" set will be recorded into the second param table.
		 *  Since we set 'this.routes' as second param,
		 *  ReshakeTowns will always maintain set of "good" routes in it =>
		 *  => in current class we don't need to worry about this.routes at all,
		 *  we can and will just use.
		 */
		this.routes = {};
		BusHubControlSection_ReshakeTowns(bus_hub_control_section, this.routes);
	}

/* protected */
	function SelectObjectToBuild()
	{
		local best_route = null;
		local best_route_rank = APriority.BAD; 
		foreach (dummy_route_id, pass_mail_table in this.routes) {
			// no pass route for some reason => can not establish trade
			if (pass_mail_table.pass == null) continue;
			local r = this.RankRoadRoute(pass_mail_table.pass);
			if (r > best_route_rank) {
				best_route = pass_mail_table.pass;
				best_route_rank = r;
			}
		}
		return {priority = best_route_rank, context = best_route};
	}

	function Build(route)
	{
		if (!(route.GetID() in this.routes)) {
			return {err_code = -1, s_station = null, e_station = null};
		};

		local must_be_accurate = RoadSettings.Get().accurate_construction;
		if (must_be_accurate
		&& route.connection.state == ConnectionState.CS_ACCURATE_COST_UNKNOWN) {
			this.CalculateConnectionCost(route);
			return {err_code = 1, s_station = null, e_station = null};
		}

		local pf = must_be_accurate && (route.GetCurrentRouteLevel() == 0) ?
			this.GetDefaultPathfinder() : SimplePF_ExcellentRoadsAdapter();
		return RoadRouteBuilder.BuildRoute(route, pf);
	}

	function HandleResult(result, route)
	{
		if (result.err_code == 1 || result.err_code == -2) return result;

		/* Even if succeed, still make a delay to prevent double trade making */
		// Since game access is cached (and even without it information
		//  has a small delay) other AI parts will need some time to "recognize"
		//  that game situation has changed. So here we provide this time.
		// Without this delay next AI decision can be exactly the same
		//  as previous(since inner data doesn't change) => AI'll want to do
		//  the same stuff again - unnecessary duplication.
		route.MakeNefarious(
			result.err_code == 0 ? 18 * GameTime.DAY : GameTime.YEAR
		);

		if (result.err_code == 0) {
			local s = result.s_station;
			local e = result.e_station;
			this.FoundNewTradeRoute(route, s, e);
			/*
			 * If pass route built => create new mail route.
			 * Existing stations will be expanded for trucks and mail
			 *  delivery will start.
			 */
			// (Responsible code is inside RoadTradeRoute class).
			if (route.cargo_id == CorporationUtils.pass_cargo_id.value) {
				local mail_route = this.routes[route.GetID()].mail;
				if (mail_route != null) {
					this.FoundNewTradeRoute(mail_route, s, e);
				}
			}

			if (RoadSettings.Get().accurate_construction) {
				Corporation.Get().AddAction(
					AISleepAction(AIDate.GetCurrentDate() + GameTime.MONTH)
				);
			}
		}

		return result.err_code;
	}

/* private */
	/** AI section that hold bus hubs as subsystems. */
	bus_hub_control_section = null;

	/** Container with all currently available bus routes. */
	routes = null;
}
