/**
 * Class that handles docks building.
 */
class DockBuilder
{
/* public */
	/**
	 * Build new dock.
	 * @param node Node where to build.
	 * @param c Cargo to transport. 
	 * @param water_basin_id Id of the water basin where to build.
	 * @param is_load_station Flag that defines whenever dock should be
	 *  load station, or cargo drop station.
	 * @return New dock station ID, or -1 if fails, or -2 if not enough money.
	 */
	static function BuildDock(node, c, water_basin_id, is_load_station)
	{
		AICompany.SetLoanAmount(AICompany.GetMaxLoanAmount());

		local list = DockBuilder.GetTilesWhereCanBuild(node, c, water_basin_id, is_load_station);

		local result = DockBuilder.BuildDockStation(list, AIStation.STATION_NEW, false);
		if (result >= 0 || result == -2) return result;
		result = DockBuilder.BuildDockStation(list, AIStation.STATION_NEW, true);
		if (result >= 0 || result == -2) return result;

		while (true) {
			if (node.IsTown()) {
				if (node.GetMyRating() <= AITown.TOWN_RATING_POOR) return -1;
			}

			local terraform_result = DockBuilder.TerraformCoast(node, c, is_load_station);
			if (terraform_result == -2) return -2;
			if (terraform_result == -1) return result;
			if (terraform_result == 0) {
				list = DockBuilder.GetTilesWhereCanBuild(node, c, water_basin_id, is_load_station);
				if (list.IsEmpty()) return -1;

				result = DockBuilder.BuildDockStation(list, AIStation.STATION_NEW, false);
				if (result >= 0) return result;
				result = DockBuilder.BuildDockStation(list, AIStation.STATION_NEW, true);
				if (result >= 0) return result;
			}
		}

		return result;
	}

/* private */
	/**
	 * Dock building implementation.
	 * @param tiles List with tiles(candidates) for new dock.
	 * @param station_type Either AIStation.STATION_NEW or AIStation.JOIN_ADJACENT. 
	 * @return New dock station ID, or -1 if fails, or -2 if not enough money.
	 */
	static function BuildDockStation(tiles, station_type, is_terraform_allowed)
	{
		if (tiles.Count() == 0) return -1;

		local err_auth = AIError.ERR_LOCAL_AUTHORITY_REFUSES;
		local err_cash = AIError.ERR_NOT_ENOUGH_CASH;
		foreach (t, dummy in tiles) {
			/* Try to clear tile */
			{
				local test_mode = AITestMode();
				if (!AITile.DemolishTile(t)) {
					// must be sure to destroy *mode objects,
					// or very bad bugs can happen
					test_mode = null;
					continue;
				}
			}
			if (!AITile.DemolishTile(t)) continue;

			/* Terraform land to create safe harbour */
			local x = AIMap.GetTileX(t);
			local y = AIMap.GetTileY(t);
			local exit = DockBuilder.GetDockExitTile(x, y);
			local good_harbour_sides_counter = 0;
			for (local k = -2; k < 2; k++) {
				local tx = exit - AIMap.GetTileIndex(k % 2, (k + 1) % 2);
				if (tx == t) continue;
				if (AITile.GetMaxHeight(tx) != 0) {
					if (!is_terraform_allowed) continue;
					{
						local test_mode = AITestMode();
						if (!TileUtils.LevelTile(tx, 0)) {
							test_mode = null;
							continue;
						}
					}

					if (TileUtils.LevelTile(tx, 0)) {
						good_harbour_sides_counter++;
					}
				} else {
					good_harbour_sides_counter++;
				}
			}
			if (good_harbour_sides_counter < 3) continue;

			/* Try to build dock */
			{
				local test_mode = AITestMode();
				if (!AIMarine.BuildDock(t, station_type)) {
					// must be sure to destroy *mode objects,
					// or very bad bugs can happen
					test_mode = null;
					continue;
				}
			}
			if (AIMarine.BuildDock(t, station_type)) {
				return AIStation.GetStationID(t);
			}

			/* Handle errors */
			local err = AIError.GetLastError();
			if (err == err_auth || err == err_cash) {
				return err == err_cash ? -2 : -1;
			}
		}
		return -1;
	}

	/**
	 * Returns tiles which are good candidates to place dock.
	 * @param node Node where to search.
	 * @param c Cargo to transport. 
	 * @param water_basin_id Id of the water basin where to build.
	 * @param is_load_station Flag that defines whenever dock should be
	 *  load station, or cargo drop station.
	 * @return AITileList with tiles safe enough to place dock.
	 */
	static function GetTilesWhereCanBuild(node, c, water_basin_id, is_load_station)
	{
		local qt = ShipDepartment.Get().water_qt;
		local seas = ShipDepartment.Get().seas;
		if (!(water_basin_id in seas)) return AITileList();

		local r = AIStation.GetCoverageRadius(AIStation.STATION_DOCK);
		local list = is_load_station ? node.AITileList_NodeProducing(r, c) :
			node.AITileList_NodeAccepting(r, c);

		list.Valuate(AITile.IsCoastTile);
		list.RemoveValue(0);

		local transit_allowed = (AIController.GetSetting("BusModule") >= 2);

		local result = AITileList();
		foreach (t, dummy in list) {
			if (transit_allowed) {
				if (TileUtils.GetAmountOfFlatTilesAround(t) <= 0) continue;
			}
			if (DockBuilder.IsCoastTileGoodForDock(t, qt, water_basin_id)) {
				result.AddTile(t);
			}
		}

		result.Valuate(AITile.GetCargoProduction, c, 1, 1, r);
		list.RemoveBelowValue(WaterSettings.Get().min_acceptance_for_dock);

		result.Sort(AIList.SORT_BY_VALUE, false);

		return result;
	}

	/**
	 * Terraform coast line near the given node to create suitable sites for
	 *  dock construction.
	 * @param node Node where dock is needed.
	 * @param c Cargo ID which future dock must handle.
	 * @param is_load_station Flag showing whether future dock must be used as
	 *  cargo drop point or cargo loading point(by ships of course).
	 */
	static function TerraformCoast(node, c, is_load_station)
	{
		local r = AIStation.GetCoverageRadius(AIStation.STATION_DOCK);
		local list = is_load_station ? node.AITileList_NodeProducing(r, c) :
			node.AITileList_NodeAccepting(r, c);

		list.Valuate(AITile.IsCoastTile);
		list.RemoveValue(0);
		list.Valuate(AITile.IsBuildable);
		list.RemoveValue(0);

		foreach (t, dummy in list) {
			local s = AITile.GetSlope(t);
			if (TileUtils.IsSlopeWithOneCornerRaised(s)) {
				if (TileUtils.LevelTile(t, 0)) {
					AIController.Sleep(3 * Ticks.DAY);
					return 0;
				}
				if (AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH) return -2;
			}
		}

		return -1;
	}

	/**
	 * Checks whether given coast tile are good enough for dock.
	 * @param t The tile to check.
	 * @param big_qt Quadtree describing node's region. 
	 * @param water_basin_id Id of the water basin where to build.
	 * @return True if and only if the tile is good dock location.
	 */
	static function IsCoastTileGoodForDock(t, big_qt, water_basin_id)
	{
		local exit = DockBuilder.GetDockExitTile(AIMap.GetTileX(t), AIMap.GetTileY(t));
		if (exit == -1) return false;

		for (local k = -2; k < 2; k++) {
			local tx = exit - AIMap.GetTileIndex(k % 2, (k + 1) % 2);
			if (AITile.IsStationTile(tx)) return false;
			if (tx != t && AITile.GetMaxHeight(tx) != 0) {
				if (!AITile.IsBuildable(tx)) return false;
			}
		}

		local r = 2;
		local x = AIMap.GetTileX(exit);
		local y = AIMap.GetTileY(exit);
		local water_around = big_qt.GetIntersection(x - r, y - r, x + r, y + r, 0);

		foreach (dummy_id, water_patch in water_around) {
			foreach (dummy_id, qt in big_qt.GetAdjacent(water_patch)) {
				if (qt.value < SeaLevelRecognition.SL_SEA_SHORE) continue;
				if (qt.custom_data.rawget(MarineBasin.key) != water_basin_id) continue;

				x = big_qt.x_shift + qt.x_min + qt.edge_len / 2;
				y = big_qt.y_shift + qt.y_min + qt.edge_len / 2;
				local sea_tile = AIMap.GetTileIndex(x, y);
				if (!AITile.IsWaterTile(sea_tile)) continue;
				if (WHPASTAR_Adapter().AreTilesConnectedShort(sea_tile, exit)) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Returns dock's water end tile.
	 * @param x X coordinate of presumably dock's land end tile.
	 * @param y Y coordinate of presumably dock's land end tile.
	 * @return ID of a tile which should be the dock's water exit.
	 */
	static function GetDockExitTile(x, y)
	{
		if (x <= 1 || x >= AIMap.GetMapSizeX() - 2) return -1;
		if (y <= 1 || y >= AIMap.GetMapSizeY() - 2) return -1;
		switch (AITile.GetSlope(AIMap.GetTileIndex(x, y))) {
			case AITile.SLOPE_NE:
				return AIMap.GetTileIndex(x + 1, y);
			case AITile.SLOPE_SE:
				return AIMap.GetTileIndex(x, y - 1);
			case AITile.SLOPE_SW:
				return AIMap.GetTileIndex(x - 1, y);
			case AITile.SLOPE_NW:
				return AIMap.GetTileIndex(x, y + 1);
		}
		return -1;
	}
}
