/**
 * Class that builds docs, buoys, and water depots.
 */
class WaterRouteBuilder
{
/* public */
	/**
	 * Build all water related infrastructure for successful sea trade.
	 * @param water_route Water route.
	 * @return Table with err_code(error code), s_dock(first built dock) and
	 *  e_dock(second dock) fields.
	 */
	static function BuildRoute(s_node, s_basin_id, e_node, e_basin_id, is_one_way, c, accurate_length)
	{
		//CodeUtils.Log("Build command requested for " + water_route.GetName(), 1);
		CodeUtils.Log("Building water path from " + s_node.GetName() + " to " + e_node.GetName(), 1);

		local result = {err_code = -1, s_dock = null, e_dock = null, new_accurate_length = -1};

		/* Need to know stuff */
		/*local c = water_route.cargo_id;
		local s_basin_id = water_route.start_basin_id;
		local e_basin_id = water_route.end_basin_id;
		local s_node = water_route.GetStart();
		local e_node = water_route.GetEnd();*/
		//local s_node = s_node.node;
		//local e_node = e_node.node;

		/* Nodes became invalid, don't build anything */
		// actual because of cached access:
		//  while data is cleared from general places, it still may be in cache
		//  => we can access something that points to garbage/null
		if (s_node.is_closed || e_node.is_closed) return result;

		/* Find/build start dock */
		local s = WaterRouteBuilder.GetDockNearNode(s_node, c, false);
		if (s == null) {
			local id = WaterRouteBuilder.BuildNewDock(s_node, c, s_basin_id, true);
			if (id < 0) return result;
			s = WaterRouteBuilder.GetDockNearNode(s_node, c, false);
			if (s == null) return result;
		}

		//local is_one_way = water_route.IsOneWay();
		/* Find/build end dock */
		local e = WaterRouteBuilder.GetDockNearNode(e_node, c, is_one_way);
		if (e == null) {
			local id = WaterRouteBuilder.BuildNewDock(e_node, c, e_basin_id, !is_one_way);
			if (id < 0) return result;
			e = WaterRouteBuilder.GetDockNearNode(e_node, c, is_one_way);
			if (e == null) return result;
		}

		/* Build depots */
		if (!(s.depot.Check() && e.depot.Check())) return result;

		/* Build buoys */
		local accurate_route_length = WaterRouteBuilder.ConnectDocks(s, e);
		if (accurate_route_length < 0) {
			result.err_code = accurate_route_length;
		} else {
			result.s_dock = s;
			result.e_dock = e;
			result.err_code = 0;
			if (accurate_route_length > 27 * accurate_length / 20) {
				result.err_code = -3;
			}
			result.new_accurate_length = accurate_route_length;
			//water_route.SetAccurateLength(accurate_route_length);
		}
		return result;
	}

/* private */
	/**
	 * Build new dock.
	 * @param node Node where to build.
	 * @param c Cargo to handle.
	 * @param basin_id Water basid(ID) to operate within.
	 * @param must_load Flag, defining if new dock must accept or supply cargo.
	 * @return ID of the newly bult dock, or negative number error code.
	 */
	static function BuildNewDock(node, c, basin_id, must_load)
	{
		CodeUtils.Log("Building a dock near " + node.GetName() + "...", 1);
		if (ShipDepartment.Get().IsPassive()) {
			CodeUtils.Log("... failed, new docks unwanted by sea HQ", 1);
			return -1;
		}

		/* Actually, build a dock */
		local id = DockBuilder.BuildDock(node, c, basin_id, must_load);
		if (!AIStation.IsValidStation(id)) {
			local msg = "   ...failed: ";
			if (id == -2) msg += "not enough money";
			if (id == -3) msg += "nowhere to build, or locals are angry";
			CodeUtils.Log(msg, 1);
			return id;
		}

		local new_dock = Dock(id, node.node_id, node.GetTypeID(), c);
		CodeUtils.Log("\"" + AIStation.GetName(id) + "\" built", 1);
		node.GetAllStations().AddStation(new_dock);
		return id;
	}

	/**
	 * Connects two docks with buoys.
	 * @param start_dock First dock.
	 * @param end_dock Second dock.
	 * @return Connection length if succeed, -1 if failed,
	 *  -2 if failed due to lack of money.
	 */
	static function ConnectDocks(start_dock, end_dock)
	{
		local dep = ShipDepartment.Get();
		local s_arr = CodeUtils.ListToArray(start_dock.GetWaterExit());
		local e_arr = CodeUtils.ListToArray(end_dock.GetWaterExit());

		local d = WaterSettings.Get().max_buoys_interval;
		local buoys = WHPASTAR_Adapter.FindBuoyPath(s_arr, e_arr, dep.water_qt, d);
		if (buoys == null) {
			CodeUtils.Log("Failed to find water path", 2);
			return -1;
		}

		if (!WHPASTAR_Adapter.BuildBuoys(buoys)) {
			CodeUtils.Log("Failed to build buoys to connect docks", 2);
			return AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH ? -2 : -1;
		}

		local s_id = start_dock.GetStationID();
		local e_id = end_dock.GetStationID();
		dep.water_paths_register.RegisterBuoySequence(s_id, e_id, buoys);
		return WaterlineUtils.GetBuoyPathLength(s_arr[0], e_arr[0], buoys);
	}

	/**
	 * Find a dock near the given node.
	 * @param node Node to check.
	 * @param c ID of the cargo which target dock must handle.
	 * @param for_unload_only Flag, defining if new dock must accept or supply cargo.
	 * @return Dock object, or null if nothing found.
	 */
	static function GetDockNearNode(node, c, for_unload_only)
	{
		local docks = [];
		local c_name = Dock.GetClassName();
		foreach (dummy_id, s in node.stations.GetClassStations(c_name)) {
			if (for_unload_only) {
				if (s.CanAccept(c)) docks.append(s);
			} else {
				if (s.CanSupply(c)) docks.append(s);
			}
		}
		if (docks.len() == 0) return null;
		if (docks.len() == 1) return docks[0];

		/* Return a dock with lowest attached ships counter */
		local best_dock = null;
		local n = 1024;
		foreach (dummy_id, dock in docks) {
			local i = AIVehicleList_Station(dock.GetStationID()).Count();
			if (i < n) {
				n = i;
				best_dock = dock;
			}
		}
		return best_dock;
	}
}
