/**
 * Class that represents sea dock station.
 */
class Dock extends Station
{
/* public */
	static function GetClassName()
	{
		return "Dock";
	}

	static function Restore(memento)
	{
		local c = memento.main_terminal.GetCargoID();
		local dock = Dock(memento.station_id, memento.node_id, memento.node_type, c);

		dock.depot = memento.depot;
		dock.main_terminal = memento.main_terminal;
		dock.RegisterTerminal(dock.main_terminal);

		return dock;
	}

	/**
	 * Returns docks builded to service the given node.
	 * @param node Node (town, industy, e.t.c.).
	 * @return Array with docks assigned to the given node.
	 */
	static function GetDocks(node)
	{
		local result = [];
		foreach (dummy_id, item in node.GetClassStations("Dock")) {
			result.append(item);
		}
		return result;
	}

	/** Dock's service depot. */
	</ must_save = true />
	depot = null;

	/**
	 * Creates a dock station.
	 * @param station_id In-game dock's station ID.
	 * @param node_id ID of a node where the dock is built.
	 * @param node_type_id Type ID of the node where the dock is built.
	 * @param c ID of the cargo to handle.
	 */
	constructor(station_id, node_id, node_type_id, c)
	{
		::Station.constructor(station_id, node_id, node_type_id);

		if (SaveLoadUtils.IsLoading()) return;

		this.depot = WaterDepot(station_id);
		this.main_terminal = DockTerminal(station_id, c);
		this.RegisterTerminal(this.main_terminal); //(!) watch for override
	}

	/**
	 * Get the dock's terminal.
	 * @return Main dock's terminal.
	 */
	function GetTerminal()
	{
		return this.main_terminal;
	}

	/**
	 * Get water tiles around the dock.
	 * @return Water tiles adjacent to the dock.
	 */
	function GetWaterExit()
	{
		local ti = AIMap.GetTileIndex;
		local offsets = [ti(-1, 0), ti(0, 1), ti(1, 0), ti(0, -1)];
		local result = AITileList();
		foreach (t, dummy in AITileList_StationType(this.GetStationID(), AIStation.STATION_DOCK)) {
			foreach (dummy_id, offset in offsets) {
				if (AITile.IsWaterTile(t - offset)) result.AddTile(t - offset);
			}
		}
		return result;
	}

/* private */
	/** Main(and single) terminal for this station */
	</ must_save = true />
	main_terminal = null;
}

Terron_ClassTable.RegisterClass(Dock);
