/**
 * Class that represent significant water space, like sea or strait.
 */
class MarineBasin extends Terron_Object
{
/* public */
	/** ID to read/write quadtree custom data. */
	static key = "WBID";

	/**
	 * Create list of basins corresponding to the given game world area.
	 * @param region_water_qt Quadtree describing desired game world area.
	 * @return Array of MarineBasin objects.
	 */
	static function GetRegionSeas(region_water_qt)
	{
		local qt = region_water_qt;
		local result = TableContainer.new("Seas");

		local lx = qt.x_shift + qt.GetLength() - 3;
		local ly = qt.y_shift + qt.GetLength() - 3;
		local key = MarineBasin.key;
		local all_qt_nodes = qt.GetIntersection(qt.x_shift + 2, qt.y_shift + 2, lx, ly, -1);
		foreach (dummy_id, qt_node in all_qt_nodes) {
			qt_node.custom_data[key] <- -1;
		}

		foreach (dummy_id, qt_node in all_qt_nodes) {
			if (qt_node.value < SeaLevelRecognition.SL_OPEN_SEA) continue;
			if (qt_node.custom_data[key] == -1 && qt_node.edge_len >= 32) {
				result.AddItem(MarineBasin(qt_node, qt, result));
			}
		}

		foreach (dummy_id, qt_node in all_qt_nodes) {
			if (qt_node.value < SeaLevelRecognition.SL_OPEN_SEA) continue;
			if (qt_node.custom_data[key] == -1 && qt_node.edge_len >= 16) {
				result.AddItem(MarineBasin(qt_node, qt, result));
			}
		}

		/* Remove small lakes */
		local tmp_patches = qt.GetIntersection(qt.x_shift + 2, qt.y_shift + 2, lx, ly, SeaLevelRecognition.SL_SEA_SHORE - 1);
		foreach (basin_id, basin in result) {
			if (basin.adjacent_basins.len() == 0 && basin.basin_area <= 1536) {
				result.RemoveItem(basin.GetID());
				foreach (id, dummy in basin.adjacent_basins) {
					local tmp = result[id];
					delete tmp.adjacent_basins[basin_id];
				}
				foreach (dummy_id, patch in tmp_patches) {
					if (patch.custom_data[key] == basin_id) patch.custom_data[key] = -1;
				}
			}
		}

		/* Mark water formations near map edges as "seas" */
		foreach (dummy_id, qt in region_water_qt.GetIntersection(2, 2, lx - 3, 17, 0)) {
			if (qt.custom_data[key] == -1) continue;
			local basin = result[qt.custom_data[key]];
			if (basin.basin_type == null) basin.basin_type = BasinType.BT_SEA;
		}
		foreach (dummy_id, qt in region_water_qt.GetIntersection(2, ly - 3, lx - 3, ly - 20, 0)) {
			if (qt.custom_data[key] == -1) continue;
			local basin = result[qt.custom_data[key]];
			if (basin.basin_type == null) basin.basin_type = BasinType.BT_SEA;
		}
		foreach (dummy_id, qt in region_water_qt.GetIntersection(2, 2, 17, ly - 3, 0)) {
			if (qt.custom_data[key] == -1) continue;
			local basin = result[qt.custom_data[key]];
			if (basin.basin_type == null) basin.basin_type = BasinType.BT_SEA;
		}
		foreach (dummy_id, qt in region_water_qt.GetIntersection(2, ly - 3, lx - 3, ly - 20, 0)) {
			if (qt.custom_data[key] == -1) continue;
			local basin = result[qt.custom_data[key]];
			if (basin.basin_type == null) basin.basin_type = BasinType.BT_SEA;
		}

		/* Mark "large" basins as "seas" */
		foreach (dummy_id, basin in result) {
			if (basin.basin_area >= 8192) basin.basin_type = BasinType.BT_SEA;
		}

		/*
		 * Mark basins without neighbours as "seas" or "lakes", depending on
		 *  basin's size.
		 */
		foreach (dummy_id, basin in result) {
			if (basin.adjacent_basins.len() != 0) continue;
			basin.basin_type = basin.basin_area <= 4096 ? BasinType.BT_LAKE : BasinType.BT_SEA;
		}

		/*
		 * Mark basins with one neighbour as "bays" or "lakes", depending on
		 *  basin's size.
		 */
		foreach (dummy_id, basin in result) {
			if (basin.adjacent_basins.len() != 1 || basin.basin_type != null) continue;
			local s = basin.basin_area;

			local neighbour = null;
			foreach (dummy_id, item in basin.adjacent_basins) {
				neighbour = result[dummy_id];
			}

			if (basin.base_size > 16 && neighbour.base_size > 16) {
				basin.basin_type = (s > neighbour.basin_area) ? BasinType.BT_LAKE : BasinType.BT_BAY
			} else if (basin.base_size > 16 && neighbour.base_size <= 16) {	
				basin.basin_type = (s > neighbour.basin_area) ? BasinType.BT_LAKE : BasinType.BT_BAY
			} else if (basin.base_size <= 16 && neighbour.base_size <= 16) {
				basin.basin_type = s >= 4096 || s > neighbour.basin_area ?
					BasinType.BT_LAKE : BasinType.BT_BAY;
			} else {
				basin.basin_type = s >= 6144 || s > neighbour.basin_area ?
					BasinType.BT_LAKE : BasinType.BT_BAY;
			}
		}

		/*
		 * Mark "medium" basins as "seas" if basin consists of
		 *  small(16x16 and lesser) squares.
		 */
		foreach (dummy_id, basin in result) {
			if (basin.basin_type == null && basin.base_size < 32 && basin.basin_area >= 6144) {
				basin.basin_type = BasinType.BT_SEA;
			}
		}

		foreach (dummy_id, basin in result) {
			if (basin.adjacent_basins.len() < 2 || basin.basin_type != null || basin.base_size < 32) continue;
			local all_neighbour_are_bays = true;
			foreach (dummy_id, item in basin.adjacent_basins) {
				if (result[dummy_id].basin_type != BasinType.BT_BAY) {
					all_neighbour_are_bays = false;
					break;
				}
			}

			if (all_neighbour_are_bays) {
				basin.basin_type = basin.basin_area <= 2048 ? BasinType.BT_LAKE : BasinType.BT_SEA;
			} else {
				basin.basin_type = basin.basin_area <= 3072 ? BasinType.BT_STRAIT : BasinType.BT_SEA;
			}
		}

		foreach (dummy_id, basin in result) {
			if (basin.basin_type != null) continue;
			local all_neighbour_are_bays = true;
			foreach (dummy_id, item in basin.adjacent_basins) {
				if (result[dummy_id].basin_type != BasinType.BT_BAY) {
					all_neighbour_are_bays = false;
					break;
				}
			}

			if (all_neighbour_are_bays) {
				basin.basin_type = basin.basin_area <= 3072 ? BasinType.BT_LAKE : BasinType.BT_SEA;
			} else {
				basin.basin_type = basin.basin_area <= 4192 ? BasinType.BT_STRAIT : BasinType.BT_SEA;
			}
		}
		return result;
	}

	/** Table with IDs of basins adjacent to this one. */
	adjacent_basins = null;

/* private */
	/** This basin area. */
	basin_area = 0;

	/** Type (e.g sea, or lake, or bay) to this basin. */
	basin_type = null;

	/** Table with land tiles that form basin's shore. */
	shore_tiles = null;

	/** Length of the "base" water square edge. */
	base_size = null;

	/**
	 * Create new water basin describing object.
	 * @param start_qt Quadtree node, representing one of the terrain squares
	 *  belonging to basin.
	 * @param region_water_qt Quadtree that indexes water disposition in
	 *  (relatively) large area. 
	 * @param all_seas Table with all other water basins.
	 */
	constructor(start_qt, region_water_qt, all_seas)
	{
		::Terron_Object.constructor(null);

		this.adjacent_basins = {};
		this.base_size = start_qt.edge_len >= 32 ? 32 : 16;
		this.shore_tiles = {};
		local tmp = {}
		this.Expand(start_qt, region_water_qt, all_seas, tmp);
	}

	/**
	 * Expand this basin by adding given terrain square.
	 * @param qt Quadtree node, representing terrain square to add.
	 * @param region_water_qt Quadtree that indexes water disposition in
	 *  (relatively) large area.
	 * @param all_seas Table with all other water basins.
	 * @param tmp Table for temporary data.
	 * @return None.
	 */
	function Expand(qt, region_water_qt, all_seas, tmp)
	{
		local key = MarineBasin.key;
		if (qt.custom_data[key] != -1) return;
		this.basin_area += qt.edge_len * qt.edge_len;
		qt.custom_data[key] = this.GetID();

		local dx = region_water_qt.x_shift;
		local dy = region_water_qt.y_shift;
		foreach (dummy_id, neighbour in region_water_qt.GetAdjacent(qt)) {
			if (!(key in neighbour.custom_data)) continue;

			local v = neighbour.value;
			local x = dx + neighbour.x_min + (neighbour.edge_len - 1) / 2;
			local y = dy + neighbour.y_min + (neighbour.edge_len - 1) / 2;
			local t = AIMap.GetTileIndex(x, y);

			/*
			 * If adjacent terrain square is piece of land =>
			 *  add its center to own shore list.
			 */
			if (v < SeaLevelRecognition.SL_SEA_SHORE) {
				this.shore_tiles[t] <- neighbour.edge_len;
				continue;
			}

			/*
			 * If adjacent terrain square has already been assigned
			 *  to another basin => remember that basin as neighbour.
			 */
			local m = neighbour.custom_data[key];
			if (m != -1) {
				if (m != this.GetID() && !(m in this.adjacent_basins)) {
					local sea = all_seas.GetItem(m);
					this.adjacent_basins[sea.GetID()] <- 1;
					sea.adjacent_basins[this.GetID()] <- 1;
				}
				continue;
			}

			/* If adjacent terrain square is mixed sea and land. */
			if (v < SeaLevelRecognition.SL_OPEN_SEA) {
				neighbour.custom_data[key] = this.GetID();
				local around = region_water_qt.GetAdjacent(neighbour);
				foreach (dummy, item in around) {
					if (!(key in item.custom_data)) continue;
					if (item.value >= SeaLevelRecognition.SL_SEA) {
						local m = item.custom_data[key];
						if (m != -1) {
							if (m != this.GetID() && !(m in this.adjacent_basins)) {
								local sea = all_seas.GetItem(m);
								this.adjacent_basins[sea.GetID()] <- 1;
								sea.adjacent_basins[this.GetID()] <- 1;
							}
						}
						continue;
					}
					t = AIMap.GetTileIndex(dx + item.x_min + (item.edge_len - 1) / 2, dy + item.y_min + (item.edge_len - 1) / 2);
					this.shore_tiles[t] <- item.edge_len;
				}
				continue;
			}

			/* Special stop clause to prevent generation of too big seas. */
			if (neighbour.edge_len > qt.edge_len && neighbour.edge_len < this.base_size) {
				local t = AIMap.GetTileIndex(dx + neighbour.x_min, dy + neighbour.y_min);
				if (!(t in tmp)) {
					tmp[t] <- 1;
					continue;
				} else if (tmp[t] < 2) {
					tmp[t]++;
					continue;
				}
				delete tmp[t];
			}
			this.Expand(neighbour, region_water_qt, all_seas, tmp);
		}
	}
}
