/**
 * Class that describes AI corporation "ship department".
 */
class ShipDepartment extends AbstractTransportSystem
{
/* public */
	static function GetClassName()
	{
		return "ShipDepartment";
	}

	/**
	 * Get the ShipDepartment.
	 */
	static function Get()
	{
		if (ShipDepartment.instance[0] == null) {
			ShipDepartment();
		}
		return ShipDepartment.instance[0];
	}

	static function Restore(memento)
	{
		local result = ShipDepartment();
		result.water_paths_register = memento.water_paths_register;
		return result;
	}

	function GetName()
	{
		return "Trade Fleet HQ";
	}

	/**
	 * Check whether ship department is in passive now.
	 * @return True <=> ship department is in "passive" mode.
	 */
	function IsPassive()
	{
		return this.is_in_passive_mode.Get();
	}

	function GetBestAction()
	{
		return this.IsPassive() ? null :
			::AbstractTransportSystem.GetBestAction();
	}

	/** Container with all water basins suitable for trade. */
	seas = null;

	/** Quadtree with information about water regions inside. */
	water_qt = null;

/* private */
	/** The single instance of the class AirSettings */
	static instance = [null];

	/** Container with known buoy paths between */
	</ must_save = true />
	water_paths_register = null;

	/** True <=> the department is in passive mode. */
	is_in_passive_mode = null;

	constructor()
	{
		::AbstractTransportSystem.constructor();

		if (ShipDepartment.instance[0] != null) assert(null);
		ShipDepartment.instance[0] = this;

		this.is_in_passive_mode = Property(this.IsPassive_Check.bindenv(this), GameTime.HALF_YEAR);
		this.water_paths_register = BuiltWaterPathsRegister();

		local cfg = WaterSettings.Get();
		if (cfg.ships_limit != 0) {
			this.AddSubsystem(CruiseCentralControl());
			//this.AddSubsystem(QuadrofreightControl());

			DepartmentRequestsPriorityEvent.AddListener(this);

			if (SaveLoadUtils.IsLoading()) {
				AIStartedEvent.AddListener(this);
			} else {
				this.OnAIStarted(false);
			}
		}
	}

	/**
	 * Checks whether ship department is passive.
	 * @return True if and only if the department must not build new docks.
	 */
	function IsPassive_Check()
	{
		local vehicles = AIVehicleList();
		local total_vehicles = vehicles.Count();

		vehicles.Valuate(AIVehicle.GetVehicleType);
		vehicles.KeepValue(AIVehicle.VT_WATER);

		local cfg = WaterSettings.Get();
		if (1.0 * vehicles.Count() / (cfg.ships_limit + 1) > 1.0) return true;

		return total_vehicles < cfg.min_vehicles_to_enable_ships;
	}

	/**
	 * Init important data.
	 */
	function OnAIStarted(is_game_loaded)
	{
		local cfg = WaterSettings.Get();
		local regions = PlayableRegions.Get().map_regions_to_play;
		local r_id = cfg.region_to_operate != -1 ? cfg.region_to_operate :
			regions[regions.len() - 1];
		cfg.region_to_operate = r_id;

		this.water_qt = WaterformScaner().GetRegionWaterQuadTree(r_id);
		if (this.water_qt == null) return;

		CodeUtils.Log("World seas recognition...", 1);
		local tt = AIController.GetTick();
		this.seas = MarineBasin.GetRegionSeas(this.water_qt);
		CodeUtils.Log("... complete, ticks " + (AIController.GetTick() - tt), 1);

		Terron.UpdateMemento();
	}

	/** Disable self (=> other department can build) */
	function OnDepartmentPriorityRequest(department)
	{
		if (department.GetClassName() == AirDepartment.GetClassName()) {
			this.Disable(GameTime.YEAR);
		}
	}
}

Terron_ClassTable.RegisterClass(ShipDepartment);
